﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Xpad のエミュレーション設定に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/settings/settings_GenericPadAxis.h>
#include <nn/settings/settings_GenericPadButton.h>

namespace nn { namespace settings {

/**
 * @brief       基本的な構成を持つ Xpad に対する汎用ゲームパッドの割り当てを表す構造体です。
 */
struct BasicXpadGenericPadMap
{
    int32_t genericPadId;       //!< 基本的な構成を持つ Xpad に割り当てる汎用ゲームパッドの識別子です。
    int32_t buttonA;            //!< 基本的な構成を持つ Xpad の A ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonB;            //!< 基本的な構成を持つ Xpad の B ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonX;            //!< 基本的な構成を持つ Xpad の X ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonY;            //!< 基本的な構成を持つ Xpad の Y ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonStickL;       //!< 基本的な構成を持つ Xpad の L スティックボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonStickR;       //!< 基本的な構成を持つ Xpad の R スティックボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonL;            //!< 基本的な構成を持つ Xpad の L ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonR;            //!< 基本的な構成を持つ Xpad の R ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonZL;           //!< 基本的な構成を持つ Xpad の ZL ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonZR;           //!< 基本的な構成を持つ Xpad の ZR ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonStart;        //!< 基本的な構成を持つ Xpad の Start ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonSelect;       //!< 基本的な構成を持つ Xpad の Select ボタンに割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonLeft;         //!< 基本的な構成を持つ Xpad の十字ボタン左に割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonUp;           //!< 基本的な構成を持つ Xpad の十字ボタン上に割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonRight;        //!< 基本的な構成を持つ Xpad の十字ボタン右に割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t buttonDown;         //!< 基本的な構成を持つ Xpad の十字ボタン下に割り当てる汎用ゲームパッドのデジタルボタンです。
    int32_t axisAnalogStickRX;  //!< 基本的な構成を持つ Xpad の右アナログスティックの x 軸に割り当てる汎用ゲームパッドの座標軸です。
    int32_t axisAnalogStickRY;  //!< 基本的な構成を持つ Xpad の右アナログスティックの y 軸に割り当てる汎用ゲームパッドの座標軸です。
    int32_t axisAnalogStickLX;  //!< 基本的な構成を持つ Xpad の左アナログスティックの x 軸に割り当てる汎用ゲームパッドの座標軸です。
    int32_t axisAnalogStickLY;  //!< 基本的な構成を持つ Xpad の左アナログスティックの y 軸に割り当てる汎用ゲームパッドの座標軸です。
    GenericPadAxisAttributeSet attributes;  //!< 汎用ゲームパッドの座標軸の属性です。
};

/**
 * @brief       基本的な構成を持つ Xpad に対するキーボードの割り当てを表す構造体です。
 */
struct BasicXpadKeyboardMap
{
    int32_t buttonA;        //!< 基本的な構成を持つ Xpad の A ボタンに割り当てるキーボードのキーです。
    int32_t buttonB;        //!< 基本的な構成を持つ Xpad の B ボタンに割り当てるキーボードのキーです。
    int32_t buttonX;        //!< 基本的な構成を持つ Xpad の X ボタンに割り当てるキーボードのキーです。
    int32_t buttonY;        //!< 基本的な構成を持つ Xpad の Y ボタンに割り当てるキーボードのキーです。
    int32_t buttonStickL;   //!< 基本的な構成を持つ Xpad の L スティックボタンに割り当てるキーボードのキーです。
    int32_t buttonStickR;   //!< 基本的な構成を持つ Xpad の R スティックボタンに割り当てるキーボードのキーです。
    int32_t buttonL;        //!< 基本的な構成を持つ Xpad の L ボタンに割り当てるキーボードのキーです。
    int32_t buttonR;        //!< 基本的な構成を持つ Xpad の R ボタンに割り当てるキーボードのキーです。
    int32_t buttonZL;       //!< 基本的な構成を持つ Xpad の ZL ボタンに割り当てるキーボードのキーです。
    int32_t buttonZR;       //!< 基本的な構成を持つ Xpad の ZR ボタンに割り当てるキーボードのキーです。
    int32_t buttonStart;    //!< 基本的な構成を持つ Xpad の Start ボタンに割り当てるキーボードのキーです。
    int32_t buttonSelect;   //!< 基本的な構成を持つ Xpad の Select ボタンに割り当てるキーボードのキーです。
    int32_t buttonLeft;     //!< 基本的な構成を持つ Xpad の十字ボタン左に割り当てるキーボードのキーです。
    int32_t buttonUp;       //!< 基本的な構成を持つ Xpad の十字ボタン上に割り当てるキーボードのキーです。
    int32_t buttonRight;    //!< 基本的な構成を持つ Xpad の十字ボタン右に割り当てるキーボードのキーです。
    int32_t buttonDown;     //!< 基本的な構成を持つ Xpad の十字ボタン下に割り当てるキーボードのキーです。
};

/**
 * @brief       Xpad に対する汎用ゲームパッドの割り当てを取得します。
 *
 * @details     プレイヤー番号で指定した Xpad に対する汎用ゲームパッドの割り当てを取得します。
                Xpad に割り当てられている汎用ゲームパッドは識別子 BasicXpadGenericPadMap::genericPadId から識別することができます。
 *              識別子はプラットフォーム固有の値であり、プラットフォーム毎に規定されます。
 *              BasicXpadGenericPadMap::buttonA の様な各デジタルボタンへの割り当ての指定には GenericPadButton で定義された値が使用されます。
 *              例として、BasicXpadGenericPadMap::buttonA に GenericPadButton_1 が指定されていた場合、
 *              識別子が BasicXpadGenericPadMap::genericPadId である汎用ゲームパッドのボタン 1 の押下状態が Xpad のボタン A に対応付けられます。
 *              設定値として 0 が指定された場合は、そのデジタルボタンに対応する汎用ゲームパッドのボタンは無いものとして扱われます。
 *              BasicXpadGenericPadMap::axisAnalogStickRX の様な各座標軸への割り当ての指定には GenericPadAxis で定義された値が使用されます。
 *              例として、 BasicXpadGenericPadMap::axisAnalogStickRX に GenericPadAxis_X が指定されていた場合、
 *              識別子が BasicXpadGenericPadMap::genericPadId である汎用ゲームパッドの x 軸の座標が Xpad の右アナログスティックの x 軸に対応付けられます。
 *              設定値として 0 が指定された場合は、その座標軸に対応する汎用ゲームパッドの座標軸は無いものとして扱われます。
 *              座標値の増減方向が反転されているか否かは BasicXpadGenericPadMap::attributes から判定することができます。
 *              例として、BasicXpadGenericPadMap::attributes.Test<GenericPadAxisAttribute::IsXAxisInverted>() の値が真である場合、
 *              汎用ゲームパッドの x 軸は座標値の増減方向を反転して利用されています。
 *              Xpad に対する汎用ゲームパッドの割り当てが行われていない状態で当該関数を呼び出した場合は、
 *              内部的に解決した割り当てを返します。
 *
 * @param[out]  outValue                    Xpad に対する汎用ゲームパッドの割り当てを読み出すバッファ
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 *
 * @pre
 *              - outValue != nullptr
 */
void GetXpadGenericPadMap(BasicXpadGenericPadMap* outValue, int playerNumber
                          ) NN_NOEXCEPT;

/**
 * @brief       Xpad に対する汎用ゲームパッドの割り当てを設定します。
 *
 * @details     プレイヤー番号で指定した Xpad に対する汎用ゲームパッドの割り当てを設定します。
 *              Xpad ID の再解決後も割り当ての設定は維持されます。
 *              BasicXpadGenericPadMap の各メンバの扱いについては GetXpadGenericPadMap() を参照してください。
 *              割り当ての設定は当該関数の呼び出し完了直後から反映されます。
 *              不正な割り当てが指定された場合、設定自体は行われますが、割り当て関係は無いものとして扱われます。
 *              設定の解除には ResetXpadGenericPadMap() を使用してください。
 *
 * @param[in]   value                       Xpad に対する汎用ゲームパッドの割り当て
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 */
void SetXpadGenericPadMap(const BasicXpadGenericPadMap& value, int playerNumber
                          ) NN_NOEXCEPT;

/**
 * @brief       Xpad に対する汎用ゲームパッドの割り当てをリセットします。
 *
 * @details     プレイヤー番号で指定した Xpad に対して割り当てが設定されている場合はこれを削除し、
 *              内部的な割り当ての解決を有効に戻します。
 *              割り当てのリセットは当該関数の呼び出し完了直後から反映されます。
 *
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 */
void ResetXpadGenericPadMap(int playerNumber) NN_NOEXCEPT;

/**
 * @brief       Xpad に対するキーボードの割り当てを取得します。
 *
 * @details     プレイヤー番号で指定した Xpad に対するキーボードの割り当てを取得します。
 *              BasicXpadKeyboardMap::buttonA の様な各デジタルボタンへの割り当ての指定には ::nn::hid::KeyboardKey のビット位置が使用されます。
 *              例として、 BasicXpadGenericPadMap::buttonA に ::nn::hid::KeyboardKey::A の Index が指定されていた場合、
 *              キーボードのキー A の押下状態が Xpad のボタン A に対応付けられます。
 *              設定値として 0 が指定された場合は、そのデジタルボタンに対応するキーボードのキーは無いものとして扱われます。
 *              Xpad に対するキーボードの割り当てが行われていない状態で当該関数を呼び出した場合は、
 *              全てのデジタルボタンについて割り当てが無いことを示す値を返します。
 *
 * @param[out]  outValue                    Xpad に対するキーボードの割り当てを読み出すバッファ
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 *
 * @pre
 *              - outValue != nullptr
 */
void GetXpadKeyboardMap(BasicXpadKeyboardMap* outValue, int playerNumber
                        ) NN_NOEXCEPT;

/**
 * @brief       Xpad に対するキーボードの割り当てを設定します。
 *
 * @details     プレイヤー番号で指定した Xpad に対するキーボードの割り当てを設定します。
 *              Xpad ID の再解決後も割り当ての設定は維持されます。
 *              BasicXpadKeyboardMap の各メンバの扱いについては GetXpadKeyboardMap() を参照してください。
 *              割り当ての設定は当該関数の呼び出し完了直後から反映されます。
 *              不正な割り当てが指定された場合、設定自体は行われますが、割り当て関係は無いものとして扱われます。
 *              設定の解除には ResetXpadKeyboardMap() を使用してください。
 *
 * @param[in]   value                       Xpad に対するキーボードの割り当て
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 */
void SetXpadKeyboardMap(const BasicXpadKeyboardMap& value, int playerNumber
                        ) NN_NOEXCEPT;

/**
 * @brief       Xpad に対するキーボードの割り当てをリセットします。
 *
 * @details     プレイヤー番号で指定した Xpad に対して割り当てが設定されている場合はこれを削除します。
 *              割り当てのリセットは当該関数の呼び出し完了直後から反映されます。
 *
 * @param[in]   playerNumber                Xpad のプレイヤー番号
 */
void ResetXpadKeyboardMap(int playerNumber) NN_NOEXCEPT;

}} // namespace nn::settings
