﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ndd/ndd_Types.h>
#include <nn/ndd/detail/ndd_Storage.h>

namespace nn { namespace ndd {

//[todo]255->0のような、型依存の回り込み処理を無くす
//AddとClearのみ排他している。AddはWorkerとReceiveから呼ばれ、ClearはWorkerから呼ばれる
//カウンタが回り込んだ瞬間、0より前のデータは消失扱いとなるが、カウンタが4Byteのため実運用上発生しない
//m_Counterが指すのは次のAdd対象
//GetとDumpは、ReadCacheを書き換え得るため、非const
class RingBuffer {
public:
    static const int BufferCountMax = 50;
    void Initialize();
    void Add(const ReceiveDataDescription& dataDescription);
    bool Get(ReceiveDataDescription* pBuffer, uint32_t* pNextCounter, uint32_t counter);
    uint32_t GetCurrentCounter() const;
    uint32_t GetOldestCounter() const;
    uint32_t GetNextCounter() const;
    uint32_t GetRecentCounter(uint32_t counter, int count) const;
    uint16_t GetAvailableCount(uint64_t counter) const;
    void Clear();
    void Dump();
private:
    mutable nn::os::SdkRecursiveMutex m_Mutex;
    uint32_t m_Counter;//次のAdd対象
    Storage m_Storage;
    uint16_t GetIndex(uint32_t counter) const;
    bool IsValidCounter(uint32_t counter) const;//無効なカウンタで無いことを確認
    uint32_t NormalizeCounter(uint32_t counter) const;//最古より古いカウンタ値の場合、最古に変換
};

//リングバッファとfsのデータ数は同一にしている。理由は、
//リングバッファで指定可能なデータはfs上にも存在しなければならず、またfsをリングバッファより大きくすることに利点が無いため
NN_STATIC_ASSERT(File::DataCountMax *  DummyFs::FileCountMax == RingBuffer::BufferCountMax);

//[todo]排他の追加
class RejectDataIdContainer {
public:
    RejectDataIdContainer();
    void Add(const DataId& id);
    void Clear();
    bool IsReject(const DataId& id) const;
private:
    bool IsSameId(const DataId& idA, const DataId& idB) const;
    static const int IdCountMax = 1000;
    uint16_t m_Count;//登録数
    uint16_t m_Index;//更新対象。登録上限に達すると、m_CountはIdCountMaxを維持し、m_Indexは0に回り込む
    DataId m_Id[IdCountMax];
};

class ScanBuffer
{
public:
    ScanBuffer();
    void Add(const ReceiveDataDescription& dataDescription);
    void Clear();
    int GetCount() const;
    ReceiveDataDescription Get(int index) const;
private:
    mutable nn::os::SdkMutex m_Mutex;
    static const int BufferCountMax = 50;
    uint8_t m_Count;
    ReceiveDataDescription m_Rdd[BufferCountMax];
    bool IsSameData(const ReceiveDataDescription& dataDescription) const;
};

}}
