﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/ndd/ndd_Types.h>
#include <nn/ndd/detail/ndd_Cipher.h>
#include <nn/crypto/crypto_Sha256Generator.h>

namespace nn { namespace ndd {

//データとパケットを相互に変換する
//[todo]可変長ヘッダ、バージョン更新を考慮
//[todo]名称整理。案 MessageHeader -> DataHeader
class Packet {
public:
    static const size_t PacketSizeMax = 1300;

    Packet();

    //Packetの生成。再呼び出し時は既存の内容を破棄し、再生成
    void SetFromSendDataDescription(const SendDataDescription& sendDataDescription);
    bool SetFromPacket(const void* pData, size_t size);//不正Packetでfalse

    //Packetの情報を取得。 SetFromSendDataDescription 後に使用可能
    const uint8_t* GetPacketPtr() const;
    int GetPacketSize() const;

    //Packetの情報を取得。 SetFromPacket 後に使用可能
    const uint8_t* GetDataPtr() const;
    int GetDataSize() const;
    DataId GetDataId() const;

private:
    struct PacketHeader
    {
        uint8_t version;
        uint8_t _reserve[49];
    };

    struct MessageHeader
    {
        uint8_t version;
        uint8_t dataId[DataIdSize];
        uint8_t hash[nn::crypto::Sha256Generator::HashSize];//[todo]削減
        uint8_t _reserve[9];
    };

    struct InternalPacket
    {
        uint8_t raw[PacketSizeMax];
    };

    enum State
    {
        State_NoPacket,
        State_Decrypted,
        State_Encrypted,
    };

    static const uint8_t PacketHeaderVersion = 0xA0;
    static const uint8_t MessageHeaderVersion = 0xB0;
    static const size_t DataSizeMax = PacketSizeMax - sizeof(PacketHeader) - sizeof(MessageHeader);

    //設計上最大値でも溢れないことをチェック（内部実装における境界のため、両レイヤの最大値を比較）
    NN_STATIC_ASSERT(DataSizeMax == SendDataSizeMax);
    NN_STATIC_ASSERT(DataSizeMax == ReceiveDataSizeMax);

    InternalPacket m_InternalPacket;
    size_t m_DataSize;
    Cipher m_Cipher;
    State m_State;

    //ユーティリティ
    int GetPacketHeaderIndex() const;
    int GetMessageHeaderIndex() const;
    int GetDataIndex() const;
    void GetDataHash(uint8_t hash[], size_t size) const;

    //Packetの操作
    void Clear();
    void Encrypt();
    void Decrypt();

    //Packet生成
    void GeneratePacketHeader();
    void GenerateMessageHeader(const SendDataDescription& sendDataDescription);
    void GenerateData(const SendDataDescription& sendDataDescription);
    void GenerateHash();

    //Packet構造からのPacketの生成
    bool GeneratePacket(const void* pData, size_t size);//サイズ異常（小サイズ）時はfalse

    //Packetの確認
    bool IsValid() const;
};

}}
