﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <nn/nn_Common.h>
#include <nn/account/account_Types.h>
#include <nn/nd/nd_TypesForSystem.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_SdkConditionVariable.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_Span.h>

namespace nn { namespace nd { namespace service {

class UserIdManager
{
public:
    UserIdManager(void* threadStack, size_t threadStackSize) NN_NOEXCEPT
        : m_ThreadStack(threadStack)
        , m_ThreadStackSize(threadStackSize)
    {};

    void Initialize(const util::Span<const account::Uid>& uids) NN_NOEXCEPT;
    void Update(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT;

    LocalUserId GetLocalUserId(const account::Uid& uid) NN_NOEXCEPT;
    util::optional<NetworkUserId> GetNetworkUserId(const account::Uid& uid) NN_NOEXCEPT;

    void RefreshNetworkServiceAccountId() NN_NOEXCEPT;

    os::Event& GetNetworkUserIdUpdateEventRef() NN_NOEXCEPT
    {
        return m_NetworkUserIdUpdatedEvent;
    }
private:
    void UpdateUnsafe(const util::Span<const account::Uid>& newUids) NN_NOEXCEPT;
    static void UpdateNetworkUserIdThread(void* arg) NN_NOEXCEPT;
    void WaitUpdateNetworkUserIdRequest() const NN_NOEXCEPT;

    class UserId
    {
    public:
        account::Uid m_Uid = account::InvalidUid;
        util::optional<account::NetworkServiceAccountId> m_NsaId;
        LocalUserId m_LocalUserId;
        util::optional<NetworkUserId> m_NetworkUserId;

        bool m_IsNetworkUserIdUpdateRequested;

        void Initialize(const account::Uid& uid) NN_NOEXCEPT;
        void Reset() NN_NOEXCEPT;
    };
    //! 未実装
    static Result StoreUserId(const UserId& u) NN_NOEXCEPT;
    //! 未実装
    static Result LoadUserId(UserId* pOut, const account::Uid& uid) NN_NOEXCEPT;

    std::array<UserId, account::UserCountMax> m_UserIds;

    UserId* FindOrInitUserIdUnsafe(const account::Uid& uid) NN_NOEXCEPT;

    os::Event m_NetworkUserIdUpdatedEvent{ os::EventClearMode_ManualClear };
    mutable os::SdkConditionVariable m_ConditionalVariable;
    mutable os::SdkMutex m_Mutex;
    os::ThreadType m_ThreadType;
    void* m_ThreadStack;
    size_t m_ThreadStackSize;
};

}}} // ~nn::nd::service
