﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/migration/migration_AsyncContext.h>
#include <nn/migration/migration_UserMigrationTypes.h>

namespace nn { namespace migration { namespace user {
class IServer;
}}} // ~namespace nn::migration::user

namespace nn { namespace migration {

/**
    @brief ユーザー移行のサーバー
    @details
        *this がユーザー移行処理と関連づいている場合に限り、 *this の状態を「有効である」とします。
*/
class UserMigrationServer
{
    friend void swap(UserMigrationServer& o0, UserMigrationServer& o1) NN_NOEXCEPT;

private:
    user::IServer* m_Ptr;

public:
    /**
        @brief デフォルト構築子
        @post *this は無効。
    */
    UserMigrationServer() NN_NOEXCEPT;
    /**
        @brief ムーブ構築子
        @post *this が有効な場合、中断されます。
    */
    UserMigrationServer(UserMigrationServer&& rhs) NN_NOEXCEPT;
    /**
        @brief 解体子
        @post *this が有効な場合、中断されます。
    */
    ~UserMigrationServer() NN_NOEXCEPT;

    /**
        @brief ムーブ代入演算子
        @post *this が有効な場合、中断されます。
    */
    UserMigrationServer& operator=(UserMigrationServer&& rhs) NN_NOEXCEPT;

    /**
        @brief サーバー作成時に設定したプロファイル情報を取得します。
        @pre *this が有効。
    */
    void GetServerProfile(UserMigrationServerProfile* pOut) const NN_NOEXCEPT;

    /**
        @brief 移行対象のユーザーアカウント識別子を取得します。
        @pre *this が有効。
    */
    account::Uid GetUid() const NN_NOEXCEPT;

    /**
        @brief ユーザー移行の事前準備を行います。
        @pre *this が有効。
        @details
            取得した AsyncContext の GetResult() が失敗を返したとき、問題を解消して再度呼び出すことができます。

            この関数が返す AsyncContext は次の Result を返す場合があります。

            <ul>
                <li>
                    nn::migration::ResultSaveDataCountExceedingLimit
                    <ul><li>現在のユーザーに関して、移行をサポートしていない数のセーブデータが存在します。</li></ul>
                </li>
            </ul>
    */
    AsyncContext PrepareAsync() NN_NOEXCEPT;

    /**
        @brief 接続処理が必要かどうかを返します。
        @pre PrepareAsync() が成功した。
        @details
            この関数が true を返す場合、 ProcessConnectionAsync() と ProcessTransferAsync() を順に実行する必要があります。
            そうでない場合、これらの呼び出しを省略して CompleteAsync() を実行します。
    */
    bool IsConnectionRequired() const NN_NOEXCEPT;

    /**
        @brief クライアントとの接続待ちを行います。
        @pre PrepareAsync() が成功した。
        @details
            取得した AsyncContext の GetResult() が失敗を返したとき、現在のインスタンスではこれ以上処理を継続できません。

            この関数が返す AsyncContext は次の Result を返す場合があります。

            <ul>
                <li>
                    nn::migration::ResultConnectionWaitingTimeout
                    <ul><li>ユーザー移行のクライアントの接続待ちがタイムアウトしました。</li></ul>
                </li>
            </ul>
    */
    AsyncContext WaitConnectionAsync() NN_NOEXCEPT;

    /**
        @brief 接続したクライアントのプロファイル情報を取得します。
        @pre WaitConnectionAsync() が成功した。
    */
    void GetClientProfile(UserMigrationClientProfile* pOut) const NN_NOEXCEPT;

    /**
        @brief クライアントの接続を受け入れます。
        @pre WaitConnectionAsync() が成功した。
        @details
            取得した AsyncContext の GetResult() が失敗を返したとき、現在のインスタンスではこれ以上処理を継続できません。
    */
    AsyncContext AcceptConnectionAsync() NN_NOEXCEPT;

    /**
        @brief クライアントの接続を拒否します。
        @pre WaitConnectionAsync() が成功した。
        @post
            - 現在のインスタンスではこれ以上処理を継続できない
    */
    AsyncContext DeclineConnectionAsync() NN_NOEXCEPT;

    /**
        @brief 移行処理を行います。
        @pre AcceptConnectionAsync() が成功した。
        @details
            取得した AsyncContext の GetResult() が失敗を返したとき、現在のインスタンスではこれ以上処理を継続できません。
    */
    AsyncContext ProcessTransferAsync() NN_NOEXCEPT;

    /**
        @brief ユーザー移行の終了処理を行います。
        @pre 次のいずれかをみたす。
            - ProcessTransferAsync() が成功した。
            - IsConnectionRequired() が false を返す。
        @post
            - 現在のインスタンスではこれ以上処理を継続できない。
            - サーバーを再開不可。
    */
    AsyncContext CompleteAsync() NN_NOEXCEPT;

    // これらの関数は通常使用しません。
    explicit UserMigrationServer(user::IServer* ptr) NN_NOEXCEPT;
    Result Abort() NN_NOEXCEPT;
};

void swap(UserMigrationServer& o0, UserMigrationServer& o1) NN_NOEXCEPT;

}} // ~namespace nn::migration
