﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/migration/migration_AsyncContext.h>
#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/ncm/ncm_ContentMetaId.h>

namespace nn { namespace migration { namespace user {
class IClient;
}}} // ~namespace nn::migration::user

namespace nn { namespace migration {

/**
    @brief ユーザー移行のクライアント
    @details
        *this がユーザー移行処理と関連づいている場合に限り、 *this の状態を「有効である」とします。
*/
class UserMigrationClient
{
    friend void swap(UserMigrationClient& o0, UserMigrationClient& o1) NN_NOEXCEPT;

private:
    user::IClient* m_Ptr;

public:
    /**
        @brief デフォルト構築子
        @post *this は無効。
    */
    UserMigrationClient() NN_NOEXCEPT;

    /**
        @brief ムーブ構築子
        @post *this が有効な場合、中断されます。
    */
    UserMigrationClient(UserMigrationClient&& rhs) NN_NOEXCEPT;

    /**
        @brief 解体子
        @post *this が有効な場合、中断されます。
    */
    ~UserMigrationClient() NN_NOEXCEPT;

    /**
        @brief ムーブ代入演算子
        @post *this が有効な場合、中断されます。
    */
    UserMigrationClient& operator=(UserMigrationClient&& rhs) NN_NOEXCEPT;

    /**
        @brief ユーザーのログイン処理のためのセッションを作成します。
        @pre *this が有効。
        @details
            この関数で取得したセッションIDを nn::account::ShowUiToIntroduceExternalNetworkServiceAccountForRegistration() に指定するとログインのためのUIが表示されます。
            この関数を呼び出すと、過去作成したセッションは破棄されます。
    */
    account::SessionId CreateLoginSession() const NN_NOEXCEPT;

    /**
        @brief ユーザーがログインしたネットワークサービスアカウントを取得します。
        @pre ユーザーがログイン済み。
    */
    account::NetworkServiceAccountId GetNetworkServiceAccountId() const NN_NOEXCEPT;

    /**
        @brief ユーザーがログインしたネットワークサービスアカウントのニックネームを取得します。
        @pre ユーザーがログイン済み。
    */
    void GetUserNickname(account::Nickname* pOut) const NN_NOEXCEPT;

    /**
        @brief ユーザーがログインしたネットワークサービスアカウントのプロフィール画像を取得します。
        @pre ユーザーがログイン済み。
    */
    size_t GetUserProfileImage(void* buffer, size_t bufferSize) const NN_NOEXCEPT;

    /**
        @brief ユーザー移行を準備します。
        @pre ユーザーがログイン済み。
        @details
            取得した AsyncContext の GetResult() が失敗を返したとき、問題を解消して再度呼び出すことができます。

            この関数が返す AsyncContext は次の Result を返す場合があります。

            <ul>
                <li>
                    nn::migration::ResultUnexpectedNetworkServiceAccountLoggedIn
                    <ul><li>ユーザー移行を再開しましたが、前回移行中だったネットワークサービスアカウントと異なるアカウントでログインされました。</li></ul>
                </li>
            </ul>
    */
    AsyncContext PrepareAsync() NN_NOEXCEPT;

    /**
        @brief 接続処理が必要かどうかを返します。
        @pre PrepareAsync() が成功した。
        @details
            この関数が true を返す場合、 ConnectAsync() を実行する必要があります。
            そうでない場合 CompleteAsync() を実行します。
    */
    bool IsConnectionRequired() const NN_NOEXCEPT;

    /**
        @brief 接続先となるサーバーを検索します。
        @pre PrepareAsync() が成功した。
        @details
            接続先となるのは、クライアントでログイン済みのネットワークサービスアカウント(NSA)と同一のNSAを持つユーザーアカウントを指定して作成されたサーバーに限ります。
            この関数を呼び出すと、 ListServers() で列挙したサーバーの情報は無効になります。

            取得した AsyncContext の GetResult() が失敗を返したとき、現在のインスタンスではこれ以上処理を継続できません。
    */
    AsyncContext ScanServersAsync() NN_NOEXCEPT;

    /**
        @brief ScanServersAsync() で検索したサーバーを列挙します。
        @pre ScanServerAsync() が成功した。
        @details
            取得される UserMigrationServerInfo の profile メンバーを扱う際には、通信路で改ざんされている可能性を考慮してください。
    */
    size_t ListServers(UserMigrationServerInfo* pOutArray, size_t arraySize) const NN_NOEXCEPT;

    /**
        @brief ListServers で列挙したサーバーを指定して接続します。
        @pre ScanServerAsync() が成功した。
        @details
            取得した AsyncContext の GetResult() が失敗を返したとき、現在のインスタンスではこれ以上処理を継続できません。

            この関数が返す AsyncContext は次の Result を返す場合があります。

            <ul>
                <li>
                    nn::migration::ResultConnectingTimeout
                    <ul><li>ユーザー移行のサーバーへの接続がタイムアウトしました。</li></ul>
                </li>
            </ul>
    */
    AsyncContext ConnectAsync(const util::Uuid& serverId) NN_NOEXCEPT;

    /**
        @brief 移行されるユーザーの nn::account::Uid を取得します。
        @pre 次のいずれかをみたす。
            - ConnectAsync() が成功した。
            - IsConnectionRequired() が false を返す。
    */
    account::Uid GetImmigrantUid() const NN_NOEXCEPT;

    /**
        @brief ユーザー移行に不足しているユーザー領域のストレージサイズを取得します。
        @pre 次のいずれかをみたす。
            - ConnectAsync() が成功した。
            - IsConnectionRequired() が false を返す。
    */
    int64_t GetStorageShortfall() const NN_NOEXCEPT;

    /**
        @brief 転送し得るデータの総量を取得します。
        @pre 次のいずれかをみたす。
            - ConnectAsync() が成功した。
            - IsConnectionRequired() が false を返す。
    */
    TransferInfo GetTotalTransferInfo() const NN_NOEXCEPT;

    /**
        @brief 転送済みデータの総量を取得します。
        @pre 次のいずれかをみたす。
            - ConnectAsync() が成功した。
            - IsConnectionRequired() が false を返す。
        @details
            ユーザー移行を再開した場合、前回までの有効な転送の内容を含みます。
    */
    TransferInfo GetCurrentTransferInfo() const NN_NOEXCEPT;

    /**
        @brief 転送済みデータと関連するアプリケーションのIDを取得します。
        @pre 次のいずれかをみたす。
            - ConnectAsync() が成功した。
            - IsConnectionRequired() が false を返す。
        @details
            ユーザー移行を再開した場合、前回までの有効な転送の内容を含みます。
    */
    size_t GetCurrentRelatedApplications(ncm::ApplicationId* pOutArray, size_t arraySize) const NN_NOEXCEPT;

    /**
        @brief 本体システムによって決定された転送対象リストの次の要素を転送します。
        @pre ConnectAsync() が成功した。
        @details
            取得した AsyncContext の GetResult() が失敗を返したとき、現在のインスタンスではこれ以上処理を継続できません。

            この関数が返す AsyncContext は次の Result を返す場合があります。

            <ul>
                <li>
                    nn::migration::ResultInsufficientUserDataCapacity
                    <ul><li>転送しようとしたデータを格納するには、ユーザー領域が不足しています。</li></ul>
                </li>
            </ul>
    */
    AsyncContext TransferNextAsync() NN_NOEXCEPT;

    /**
        @brief 本体システムによって決定された転送対象リストの残りの要素をすべて転送します。
        @pre ConnectAsync() が成功した
        @details
            この関数は転送可能な全てのデータに関して TransferNextAsync() を実施するユーティリティです。
            この関数は場合により呼び出し元を長時間ブロックします。

            この関数呼び出しが失敗した場合、現在のインスタンスではこれ以上処理を継続できません。

            この関数が返す AsyncContext は次の Result を返す場合があります。

            <ul>
                <li>
                    nn::migration::ResultInsufficientUserDataCapacity
                    <ul><li>ユーザー領域が不足しています。</li></ul>
                </li>
            </ul>
    */
    Result TransferAll() NN_NOEXCEPT;

    /**
        @brief ユーザー移行処理を中断します。
        @pre *this が有効。
        @post
            - 現在のインスタンスではこれ以上処理を継続できない。
            - *this が再開されたクライアントか ConnectAsync() が成功した場合、クライアントを再開可能。
    */
    AsyncContext SuspendAsync() NN_NOEXCEPT;

    /**
        @brief ユーザー移行処理を完了します。
        @pre 次のいずれかをみたす。
            - ConnectAsync() が成功した。
            - IsConnectionRequired() が false を返す。
        @post
            - 現在のインスタンスではこれ以上処理を継続できない。
            - クライアントを再開不可。
    */
    AsyncContext CompleteAsync() NN_NOEXCEPT;

    // これら関数は通常使用しません。
    explicit UserMigrationClient(user::IClient* ptr) NN_NOEXCEPT;
    Result Abort() NN_NOEXCEPT;
    AsyncContext DebugSynchronizeStateInFinalization() NN_NOEXCEPT;
};

void swap(UserMigrationClient& o0, UserMigrationClient& o1) NN_NOEXCEPT;

}} // ~namespace nn::migration
