﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

namespace nn { namespace migration { namespace idc {

/**
 * @brief   保持する対象のポインタと、その実体が使用しているリソースのロック状態を管理するクラス。
 *          本クラスのデストラクタで保持しているポインタのデストラクタを呼び、ロックを開放します。
 */
template <typename UserCommandProcessorType>
class UserCommandProcessorHolder
{
    NN_DISALLOW_COPY(UserCommandProcessorHolder);
public:
    UserCommandProcessorHolder() NN_NOEXCEPT;
    UserCommandProcessorHolder(UserCommandProcessorType* pProcessor, os::Mutex* pMutex) NN_NOEXCEPT;
    UserCommandProcessorHolder(UserCommandProcessorHolder<UserCommandProcessorType>&& rhs) NN_NOEXCEPT;
    ~UserCommandProcessorHolder() NN_NOEXCEPT;

    UserCommandProcessorHolder& operator =(UserCommandProcessorHolder<UserCommandProcessorType>&& rhs) NN_NOEXCEPT;

    UserCommandProcessorType& GetProcessorRef() NN_NOEXCEPT;
private:
    UserCommandProcessorType*   m_pProcessor;
    os::Mutex*                  m_pMutex;
};

}}} // ~nn::migration::idc

// 以下実装。

namespace nn { namespace migration { namespace idc {

template <typename UserCommandProcessorType>
UserCommandProcessorHolder<UserCommandProcessorType>::UserCommandProcessorHolder() NN_NOEXCEPT
    : m_pProcessor(nullptr)
    , m_pMutex(nullptr)
{
}

template <typename UserCommandProcessorType>
UserCommandProcessorHolder<UserCommandProcessorType>::UserCommandProcessorHolder(UserCommandProcessorType* pProcessor, os::Mutex* pMutex) NN_NOEXCEPT
    : m_pProcessor(pProcessor)
    , m_pMutex(pMutex)
{
    NN_SDK_ASSERT(m_pMutex->IsLockedByCurrentThread());
}

template <typename UserCommandProcessorType>
UserCommandProcessorHolder<UserCommandProcessorType>::UserCommandProcessorHolder(UserCommandProcessorHolder<UserCommandProcessorType>&& rhs) NN_NOEXCEPT
    : m_pProcessor(rhs.m_pProcessor)
    , m_pMutex(rhs.m_pMutex)
{
    rhs.m_pProcessor = nullptr;
    rhs.m_pMutex = nullptr;
}

template <typename UserCommandProcessorType>
UserCommandProcessorHolder<UserCommandProcessorType>::~UserCommandProcessorHolder() NN_NOEXCEPT
{
    if( m_pProcessor )
    {
        m_pProcessor->~UserCommandProcessorType();
        m_pMutex->Unlock();
    }
}

template <typename UserCommandProcessorType>
UserCommandProcessorHolder<UserCommandProcessorType>& UserCommandProcessorHolder<UserCommandProcessorType>::operator =(UserCommandProcessorHolder<UserCommandProcessorType>&& rhs) NN_NOEXCEPT
{
    m_pProcessor = rhs.m_pProcessor;
    m_pMutex = rhs.m_pMutex;
    rhs.m_pProcessor = nullptr;
    rhs.m_pMutex = nullptr;
    return *this;
}

template <typename UserCommandProcessorType>
UserCommandProcessorType& UserCommandProcessorHolder<UserCommandProcessorType>::GetProcessorRef() NN_NOEXCEPT
{
    return *m_pProcessor;
}

}}} // ~nn::migration::idc
