﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/migration/detail/migration_AsyncExecutionResource.h>
#include <nn/migration/idc/migration_CommandTypes.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/idc/migration_UserCommandEncryptor.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace idc {

template <typename EncryptionPolicy>
struct UserCommandMediatorWorkBuffer
{
    Bit8 plainBlockBuffer[EncryptionPolicy::UserCommandBlockSize];
    Bit8 encryptedBlockBuffer[EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize];
};

/**
 * @brief    ユーザーコンテキストの入力（平文）と通信路上の入力（暗号文）を仲介するクラス。
 */
template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
class UserCommandConsumeMediator
{
public:
    UserCommandConsumeMediator(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT;

    Result Consume(void* stream, size_t size) NN_NOEXCEPT;

private:
    Result ConsumeBody(void* stream, size_t size) NN_NOEXCEPT;

    bool m_IsConsumingBody;
    UserCommandDecryptor<typename EncryptionPolicy::MessageEncryptor> m_UserCommandDecryptor;
    UserContextType* m_pUserContext;

    UserCommandMediatorWorkBuffer<EncryptionPolicy>* m_pWorkBuffer;
    size_t m_AvailableEncryptedUserCommandSize;
};

/**
 * @brief   ユーザーコンテストの出力（平文）と通信路上の出力（暗号文）を仲介するクラス。
 */
template <typename EncryptionPolicy, typename UserContextType>
class UserCommandProduceMediator
{
public:
    UserCommandProduceMediator(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT;

    Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT;

private:
    Result ProduceBody(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT;

    bool m_IsProducingBody;
    UserCommandEncryptor<typename EncryptionPolicy::MessageEncryptor> m_UserCommandEncryptor;
    UserContextType* m_pUserContext;

    UserCommandMediatorWorkBuffer<EncryptionPolicy>* m_pWorkBuffer;
    size_t m_AvailablePlainUserCommandSize;
};

}}}

// 以下実装。

#include <nn/migration/detail/migration_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace idc {

// UserCommandConsumeMediator

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
UserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::UserCommandConsumeMediator(
    typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
    UserContextType* pUserContext,
    UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    : m_IsConsumingBody(false)
    , m_UserCommandDecryptor(messageEncryptor)
    , m_pUserContext(pUserContext)
    , m_pWorkBuffer(pWorkBuffer)
    , m_AvailableEncryptedUserCommandSize(0)
{
    NN_SDK_ASSERT_NOT_NULL(pUserContext);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
Result UserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::Consume(void* stream, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stream);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, 0u);
    if( !m_IsConsumingBody )
    {
        // ヘッダ分受信するまでバッファリング。
        auto copySize = std::min(sizeof(UserCommandHeader) - m_AvailableEncryptedUserCommandSize, size);
        std::memcpy(m_pWorkBuffer->encryptedBlockBuffer + m_AvailableEncryptedUserCommandSize, reinterpret_cast<Bit8*>(stream), copySize);
        m_AvailableEncryptedUserCommandSize += copySize;
        if( !(m_AvailableEncryptedUserCommandSize >= sizeof(UserCommandHeader)) )
        {
            NN_RESULT_THROW(ResultConsumeCommandContinue());
        }
        NN_SDK_ASSERT_EQUAL(m_AvailableEncryptedUserCommandSize, sizeof(UserCommandHeader));
        auto pHeader = reinterpret_cast<const UserCommandHeader*>(m_pWorkBuffer->encryptedBlockBuffer);
        NN_RESULT_THROW_UNLESS(m_UserCommandDecryptor.Initialize(*pHeader), typename ErrorConfig::ResultInvalidHeader());
        NN_RESULT_THROW_UNLESS(m_UserCommandDecryptor.GetBlockSize() == EncryptionPolicy::UserCommandBlockSize, typename ErrorConfig::ResultInvalidBlockSize());
        m_AvailableEncryptedUserCommandSize = 0;
        m_IsConsumingBody = true;

        m_pUserContext->SetByteSizeToConsume(m_UserCommandDecryptor.GetCommandSize());

        size_t remainSize = size - copySize;
        if( remainSize > 0u )
        {
            auto remainStream = reinterpret_cast<Bit8*>(stream) + copySize;
            return ConsumeBody(remainStream, remainSize);
        }
        NN_RESULT_THROW(ResultConsumeCommandContinue());
    }
    return ConsumeBody(stream, size);
}

template <typename EncryptionPolicy, typename UserContextType, typename ErrorConfig>
Result UserCommandConsumeMediator<EncryptionPolicy, UserContextType, ErrorConfig>::ConsumeBody(void* stream, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsConsumingBody);
    NN_SDK_REQUIRES_NOT_NULL(stream);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    size_t totalUsedSize = 0;

    while( totalUsedSize < size )
    {
        void* decryptBuffer = nullptr;
        size_t decryptableSize = 0u;

        size_t plainSize = 0u;
        if( m_AvailableEncryptedUserCommandSize > 0u )
        {
            // バッファにデータが残っている場合、未復号分のデータと合わせて1ブロック分 or 残り分になるまでバッファリング用領域にデータをコピーして復号対象にする。。
            auto copySize = std::min((EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize) - m_AvailableEncryptedUserCommandSize, size - totalUsedSize);
            std::memcpy(m_pWorkBuffer->encryptedBlockBuffer + m_AvailableEncryptedUserCommandSize, reinterpret_cast<Bit8*>(stream) + totalUsedSize, copySize);
            m_AvailableEncryptedUserCommandSize += copySize;
            totalUsedSize = copySize;

            NN_SDK_ASSERT_LESS_EQUAL(m_AvailableEncryptedUserCommandSize, EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize);

            decryptableSize = m_UserCommandDecryptor.GetAcceptableUpdateSize(m_AvailableEncryptedUserCommandSize);
            if( !(decryptableSize > 0u) )
            {
                NN_SDK_ASSERT_EQUAL(totalUsedSize, size);
                NN_RESULT_THROW(ResultConsumeCommandContinue());
            }
            decryptBuffer = m_pWorkBuffer->encryptedBlockBuffer;
            // バッファリング用領域からの復号の場合は decryptableSize 分利用可能バッファサイズを減らす（totalUsedSize はコピー時に加算済み）。
            NN_SDK_ASSERT_EQUAL(decryptableSize, m_AvailableEncryptedUserCommandSize);
            m_AvailableEncryptedUserCommandSize = 0u;
        }
        else
        {
            // バッファにデータがない場合は stream を復号対象にする。
            decryptableSize = std::min(EncryptionPolicy::UserCommandBlockSize + UserCommandAuthenticationSize, m_UserCommandDecryptor.GetAcceptableUpdateSize(size - totalUsedSize));
            if( !(decryptableSize > 0u) )
            {
                NN_SDK_ASSERT_LESS(size - totalUsedSize, sizeof(m_pWorkBuffer->encryptedBlockBuffer));
                std::memcpy(m_pWorkBuffer->encryptedBlockBuffer, reinterpret_cast<Bit8*>(stream) + totalUsedSize, size - totalUsedSize);
                m_AvailableEncryptedUserCommandSize = (size - totalUsedSize);
                NN_RESULT_THROW(ResultConsumeCommandContinue());
            }
            decryptBuffer = reinterpret_cast<Bit8*>(stream) + totalUsedSize;
            // stream からの復号の場合は decryptableSize 分 totalUsedSize に加算する。
            totalUsedSize += decryptableSize;
        }

        NN_SDK_ASSERT_GREATER(decryptableSize, 0u);
        NN_SDK_ASSERT_NOT_NULL(decryptBuffer);
        NN_RESULT_THROW_UNLESS(
            m_UserCommandDecryptor.Update(
                &plainSize,
                m_pWorkBuffer->plainBlockBuffer, sizeof(m_pWorkBuffer->plainBlockBuffer),
                decryptBuffer, decryptableSize),
            typename ErrorConfig::ResultInvalidData());

RetryUserConsume:

        NN_SDK_ASSERT_GREATER(plainSize, 0u);

        auto result = m_pUserContext->Consume(m_pWorkBuffer->plainBlockBuffer, plainSize);
        NN_RESULT_THROW_UNLESS(result.IsSuccess() || ResultConsumeCommandContinue::Includes(result) || ResultConsumeCommandPause::Includes(result), result);
        if( ResultConsumeCommandPause::Includes(result) )
        {
            os::SleepThread(nn::TimeSpan::FromSeconds(1));
            goto RetryUserConsume;
        }
        else if( ResultConsumeCommandContinue::Includes(result) )
        {
            continue;
        }
        NN_SDK_ASSERT(result.IsSuccess());
        NN_RESULT_THROW_UNLESS(totalUsedSize == size, typename ErrorConfig::ResultInvalidSize());
        m_IsConsumingBody = false;
        NN_RESULT_SUCCESS;
    }
    NN_SDK_ASSERT_EQUAL(totalUsedSize, size);
    NN_RESULT_THROW(ResultConsumeCommandContinue());
}

// UserCommandProduceMediator

template <typename EncryptionPolicy, typename UserContextType>
UserCommandProduceMediator<EncryptionPolicy, UserContextType>::UserCommandProduceMediator(
    typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
    UserContextType* pUserContext,
    UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    : m_IsProducingBody(false)
    , m_UserCommandEncryptor(messageEncryptor)
    , m_pUserContext(pUserContext)
    , m_pWorkBuffer(pWorkBuffer)
    , m_AvailablePlainUserCommandSize(0)
{
    NN_SDK_ASSERT_NOT_NULL(pUserContext);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
}

template <typename EncryptionPolicy, typename UserContextType>
Result UserCommandProduceMediator<EncryptionPolicy, UserContextType>::Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutProducedSize);
    NN_SDK_REQUIRES_NOT_NULL(outStream);
    if( !m_IsProducingBody )
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(UserCommandHeader));
        auto pHeader = reinterpret_cast<UserCommandHeader*>(outStream);
        m_UserCommandEncryptor.Initialize(pHeader, this->m_pUserContext->GetProducableByteSize(), EncryptionPolicy::UserCommandBlockSize);
        *pOutProducedSize = sizeof(UserCommandHeader);
        m_AvailablePlainUserCommandSize = 0;
        m_IsProducingBody = true;
        NN_RESULT_THROW(ResultProduceCommandContinue());
    }
    return ProduceBody(pOutProducedSize, outStream, outStreamSize);
}

template <typename EncryptionPolicy, typename UserContextType>
Result UserCommandProduceMediator<EncryptionPolicy, UserContextType>::ProduceBody(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, GetEncryptedUserCommandBlockSize(EncryptionPolicy::UserCommandBlockSize));
    NN_SDK_REQUIRES(m_IsProducingBody);

    *pOutProducedSize = 0;
    size_t plainUserCommandSize = 0u;
    auto result = m_pUserContext->Produce(&plainUserCommandSize, m_pWorkBuffer->plainBlockBuffer + m_AvailablePlainUserCommandSize, sizeof(m_pWorkBuffer->plainBlockBuffer) - m_AvailablePlainUserCommandSize);
    m_AvailablePlainUserCommandSize += plainUserCommandSize;
    NN_RESULT_THROW_UNLESS(result.IsSuccess() || ResultProduceCommandContinue::Includes(result) || ResultProduceCommandPause::Includes(result), result);
    if( ResultProduceCommandPause::Includes(result) )
    {
        NN_SDK_ASSERT_EQUAL(plainUserCommandSize, 0u);
        NN_RESULT_THROW(result);
    }
    else if( ResultProduceCommandContinue::Includes(result) )
    {
        NN_RESULT_THROW_UNLESS(m_AvailablePlainUserCommandSize == EncryptionPolicy::UserCommandBlockSize, ResultProduceCommandPause());

        m_UserCommandEncryptor.Update(pOutProducedSize, outStream, outStreamSize, m_pWorkBuffer->plainBlockBuffer, m_AvailablePlainUserCommandSize);
        m_AvailablePlainUserCommandSize = 0;
        NN_RESULT_THROW(result);
    }
    NN_SDK_ASSERT(result.IsSuccess());
    m_UserCommandEncryptor.Update(pOutProducedSize, outStream, outStreamSize, m_pWorkBuffer->plainBlockBuffer, m_AvailablePlainUserCommandSize);
    m_AvailablePlainUserCommandSize = 0;
    m_IsProducingBody = false;
    NN_RESULT_SUCCESS;
}

}}}
