﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/migration/idc/migration_CommandTypes.h>
#include <nn/migration/idc/migration_ThreadedUserCommandMediator.h>
#include <nn/migration/idc/migration_UserCommandMediator.h>
#include <nn/migration/idc/migration_UserCommandProcessorHolder.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace migration { namespace idc {

struct ClientUserCommandErrorConfig
{
    typedef ResultInvalidResponseSizeForUser        ResultInvalidSize;
    typedef ResultInvalidResponseDataForUser        ResultInvalidData;
    typedef ResultInvalidResponseHeaderForUser      ResultInvalidHeader;
    typedef ResultInvalidResponseBlockSizeForUser   ResultInvalidBlockSize;
};


template <typename EncryptionPolicy, typename UserContextType>
class ClientUserCommandProducer
{
public:
    ClientUserCommandProducer(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
        : m_Mediator(messageEncryptor, pUserContext, pWorkBuffer)
    {
    }
    Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
    {
        return m_Mediator.Produce(pOutProducedSize, outStream, outStreamSize);
    }
private:
    typedef UserCommandProduceMediator<EncryptionPolicy, UserContextType> Mediator;
    Mediator m_Mediator;
};

template <typename EncryptionPolicy, typename UserContextType>
class ClientUserCommandConsumer;

template <typename EncryptionPolicy, typename UserContextType>
class ThreadedClientUserCommandProducer
{
public:
    ThreadedClientUserCommandProducer(
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        migration::detail::ThreadResourceType* pThreadResource,
        ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
        : m_Mediator(messageEncryptor, pUserContext, pThreadResource, pWorkBuffer)
    {
    }
    Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
    {
        return m_Mediator.Produce(pOutProducedSize, outStream, outStreamSize);
    }
private:
    typedef ThreadedUserCommandProduceMediator<EncryptionPolicy, UserContextType> Mediator;
    Mediator m_Mediator;
};

template <typename EncryptionPolicy, typename UserContextType>
class ThreadedClientUserCommandConsumer;

template <typename EncryptionPolicy, typename UserContextType>
using ClientUserCommandProducerHolder = UserCommandProcessorHolder<ClientUserCommandProducer<EncryptionPolicy, UserContextType>>;

template <typename EncryptionPolicy, typename UserContextType>
using ClientUserCommandConsumerHolder = UserCommandProcessorHolder<ClientUserCommandConsumer<EncryptionPolicy, UserContextType>>;

template <typename EncryptionPolicy, typename UserContextType>
using ThreadedClientUserCommandProducerHolder = UserCommandProcessorHolder<ThreadedClientUserCommandProducer<EncryptionPolicy, UserContextType>>;

template <typename EncryptionPolicy, typename UserContextType>
using ThreadedClientUserCommandConsumerHolder = UserCommandProcessorHolder<ThreadedClientUserCommandConsumer<EncryptionPolicy, UserContextType>>;

/**
* @brief    クライアント（移行先）の状態遷移を管理するコンテキスト。
*/
template <typename EncryptionPolicyType>
class ClientContext
{
public:
    typedef EncryptionPolicyType EncryptionPolicy;

    class Consumer
    {
    public:
        explicit Consumer(ClientContext* pContext) NN_NOEXCEPT
            : m_pContext(pContext)
            , m_AvailableConsumableDataSize(0u)
            , m_ExpectedConsumableDataSize(0u)
        {
        }
        Result Consume(void* stream, size_t size) NN_NOEXCEPT;
    protected:
        ClientContext* m_pContext;
    private:
        void ClearBufferedConsumableData() NN_NOEXCEPT;

        util::optional<CommandKind> m_ConsumeCommandKind;
        Bit8   m_ConsumableData[ResponseCommandSizeMax];
        size_t m_AvailableConsumableDataSize;
        size_t m_ExpectedConsumableDataSize;
    };

    class Producer
    {
    public:
        explicit Producer(ClientContext* pContext) NN_NOEXCEPT
            : m_pContext(pContext)
        {
        }
        Result Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT;
    protected:
        ClientContext* m_pContext;
    };

    ClientContext() NN_NOEXCEPT;
    ~ClientContext() NN_NOEXCEPT;

    void SetCommandKind(CommandKind kind) NN_NOEXCEPT;
    Result CheckUserCommandAcceptability() const NN_NOEXCEPT;

    Consumer GetConsumer() NN_NOEXCEPT
    {
        return Consumer(this);
    }

    Producer GetProducer() NN_NOEXCEPT
    {
        return Producer(this);
    }

    template <typename UserContextType>
    ClientUserCommandProducerHolder<EncryptionPolicy, UserContextType> AcquireProducerHolder(
        UserContextType* pContext, UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    {
        typedef ClientUserCommandProducer<EncryptionPolicy, UserContextType> ProducerType;
        NN_STATIC_ASSERT(sizeof(ProducerType) <= sizeof(m_UserCommandProcessorStorage));
        NN_STATIC_ASSERT(std::alignment_of<UserCommandProcessorStorage>::value % std::alignment_of<ProducerType>::value == 0u);

        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_ABORT_UNLESS(m_UserCommandProcessorStorageMutex.TryLock());

        auto pProducer = new (&m_UserCommandProcessorStorage) ProducerType(m_MessageEncryptor, pContext, pWorkBuffer);
        return ClientUserCommandProducerHolder<EncryptionPolicy, UserContextType>(pProducer, &m_UserCommandProcessorStorageMutex);
    }

    template <typename UserContextType>
    ClientUserCommandConsumerHolder<EncryptionPolicy, UserContextType> AcquireConsumerHolder(
        UserContextType* pContext, UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    {
        typedef ClientUserCommandConsumer<EncryptionPolicy, UserContextType> ConsumerType;
        NN_STATIC_ASSERT(sizeof(ConsumerType) <= sizeof(m_UserCommandProcessorStorage));
        NN_STATIC_ASSERT(std::alignment_of<UserCommandProcessorStorage>::value % std::alignment_of<ConsumerType>::value == 0u);

        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_ABORT_UNLESS(m_UserCommandProcessorStorageMutex.TryLock());

        auto pConsumer = new (&m_UserCommandProcessorStorage) ConsumerType(this, m_MessageEncryptor, pContext, pWorkBuffer);
        return ClientUserCommandConsumerHolder<EncryptionPolicy, UserContextType>(pConsumer, &m_UserCommandProcessorStorageMutex);
    }

    template <typename UserContextType>
    ThreadedClientUserCommandProducerHolder<EncryptionPolicy, UserContextType> AcquireThreadedProducerHolder(
        UserContextType* pContext, migration::detail::ThreadResourceType* pThreadResource, ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    {
        typedef ThreadedClientUserCommandProducer<EncryptionPolicy, UserContextType> ProducerType;
        NN_STATIC_ASSERT(sizeof(ProducerType) <= sizeof(m_UserCommandProcessorStorage));
        NN_STATIC_ASSERT(std::alignment_of<UserCommandProcessorStorage>::value % std::alignment_of<ProducerType>::value == 0u);

        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_SDK_REQUIRES_NOT_NULL(pThreadResource);
        NN_ABORT_UNLESS(m_UserCommandProcessorStorageMutex.TryLock());

        auto pProducer = new (&m_UserCommandProcessorStorage) ProducerType(m_MessageEncryptor, pContext, pThreadResource, pWorkBuffer);
        return ThreadedClientUserCommandProducerHolder<EncryptionPolicy, UserContextType>(pProducer, &m_UserCommandProcessorStorageMutex);
    }

    template <typename UserContextType>
    ThreadedClientUserCommandConsumerHolder<EncryptionPolicy, UserContextType> AcquireThreadedConsumerHolder(
        UserContextType* pContext, migration::detail::ThreadResourceType* pThreadResource, ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
    {
        typedef ThreadedClientUserCommandConsumer<EncryptionPolicy, UserContextType> ConsumerType;
        NN_STATIC_ASSERT(sizeof(ConsumerType) <= sizeof(m_UserCommandProcessorStorage));
        NN_STATIC_ASSERT(std::alignment_of<UserCommandProcessorStorage>::value % std::alignment_of<ConsumerType>::value == 0u);

        NN_SDK_REQUIRES_NOT_NULL(pContext);
        NN_SDK_REQUIRES_NOT_NULL(pThreadResource);
        NN_ABORT_UNLESS(m_UserCommandProcessorStorageMutex.TryLock());

        auto pConsumer = new (&m_UserCommandProcessorStorage) ConsumerType(this, m_MessageEncryptor, pContext, pThreadResource, pWorkBuffer);
        return ThreadedClientUserCommandConsumerHolder<EncryptionPolicy, UserContextType>(pConsumer, &m_UserCommandProcessorStorageMutex);
    }

private:
    Result CheckCommandAcceptability(CommandKind kind) const NN_NOEXCEPT;
    Result ConsumeInitiate0Response(const Initiate0Response& response) NN_NOEXCEPT;
    Result ConsumeInitiate1Response(const Initiate1Response& response) NN_NOEXCEPT;
    Result ConsumeResume0Response(const Resume0Response& response) NN_NOEXCEPT;
    Result ConsumeResume1Response(const Resume1Response& resopnse) NN_NOEXCEPT;
    Result ConsumeTerminateResponse(const TerminateResponse& response) NN_NOEXCEPT;
    Result ConsumeErrorResponse(const ErrorResponse& response) NN_NOEXCEPT;

    void ProduceInitiate0Request(Initiate0Request* pOut) NN_NOEXCEPT;
    void ProduceInitiate1Request(Initiate1Request* pOut) NN_NOEXCEPT;
    void ProduceResume0Request(Resume0Request* pOut) NN_NOEXCEPT;
    void ProduceResume1Request(Resume1Request* pOut) NN_NOEXCEPT;
    void ProduceTerminateRequest(TerminateRequest* pOut) NN_NOEXCEPT;

    void ClearClientChallenge() NN_NOEXCEPT;
    void ClearServerChallenge() NN_NOEXCEPT;

    CommandKind     m_CommandKind;
    bool            m_IsKeyExchanged;

    typename EncryptionPolicy::KeyEncryptor     m_KeyEncryptor;
    typename EncryptionPolicy::MessageEncryptor m_MessageEncryptor;

    // Optionable にするための wrapper 構造体。
    struct ChallengeStruct
    {
        KeyExchangeCommandConfig::Challenge data;
    };

    util::optional<ChallengeStruct> m_ClientChallenge;
    util::optional<ChallengeStruct> m_ServerChallenge;

    typedef ClientUserCommandConsumer<EncryptionPolicy, void> DummyUserCommandConsumerType;
    typedef ThreadedClientUserCommandConsumer<EncryptionPolicy, void> DummyThreadedUserCommandConsumerType;
    typedef ThreadedClientUserCommandProducer<EncryptionPolicy, void> DummyThreadedUserCommandProducerType;
    NN_STATIC_ASSERT(sizeof(DummyUserCommandConsumerType) <= sizeof(DummyThreadedUserCommandConsumerType));
    NN_STATIC_ASSERT(sizeof(DummyThreadedUserCommandProducerType) <= sizeof(DummyThreadedUserCommandConsumerType));
    typedef typename std::aligned_storage<sizeof(DummyThreadedUserCommandConsumerType), std::alignment_of<DummyThreadedUserCommandConsumerType>::value>::type UserCommandProcessorStorage;
    UserCommandProcessorStorage m_UserCommandProcessorStorage;
    os::Mutex m_UserCommandProcessorStorageMutex;
};

}}} // ~nn::migration::idc

// 以下実装

#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/idc/migration_CommandApi.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/idc/detail/migration_HandleErrorInfo.h>
#include <nn/migration/idc/detail/migration_Result.h>
#include <nn/migration/idc/detail/migration_Util.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace idc {

// ClientContext

template <typename EncryptionPolicy>
ClientContext<EncryptionPolicy>::ClientContext() NN_NOEXCEPT
    : m_CommandKind(CommandKind::Error)
    , m_IsKeyExchanged(false)
    , m_ClientChallenge(nullptr)
    , m_ServerChallenge(nullptr)
    , m_UserCommandProcessorStorageMutex(false)
{
}

template <typename EncryptionPolicy>
ClientContext<EncryptionPolicy>::~ClientContext() NN_NOEXCEPT
{
    if( m_ClientChallenge )
    {
        ClearClientChallenge();
    }
    if( m_ServerChallenge )
    {
        ClearServerChallenge();
    }
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::SetCommandKind(CommandKind kind) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(kind, CommandKind::Error);
    m_CommandKind = kind;
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::CheckUserCommandAcceptability() const NN_NOEXCEPT
{
    return CheckCommandAcceptability(CommandKind::User);
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::CheckCommandAcceptability(CommandKind kind) const NN_NOEXCEPT
{
    if( kind == m_CommandKind )
    {
        NN_RESULT_SUCCESS;
    }
    else if( kind == CommandKind::Error &&
        (m_CommandKind != CommandKind::Initiate0 &&
         m_CommandKind != CommandKind::Initiate1 &&
         m_CommandKind != CommandKind::Resume0   &&
         m_CommandKind != CommandKind::Resume1))
    {
        NN_RESULT_SUCCESS;
    }
    switch( m_CommandKind )
    {
    case CommandKind::Initiate0:
        NN_RESULT_THROW(ResultInvalidResponseKindForInitiate0Request());
    case CommandKind::Initiate1:
        NN_RESULT_THROW(ResultInvalidResponseKindForInitiate1Request());
    case CommandKind::Resume0:
        NN_RESULT_THROW(ResultInvalidResponseKindForResume0Request());
    case CommandKind::Resume1:
        NN_RESULT_THROW(ResultInvalidResponseKindForResume1Request());
    case CommandKind::User:
        NN_RESULT_THROW(ResultInvalidResponseKindForUserRequest());
    case CommandKind::Terminate:
        NN_RESULT_THROW(ResultInvalidResponseKindForTerminateRequest());
    case CommandKind::Error:
        // Error を Client から送信しないのでここには到達しない。
        NN_RESULT_THROW(ResultInvalidResponseKindForErrorRequest());
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::ConsumeInitiate0Response(const Initiate0Response& response) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckCommandAcceptability(CommandKind::Initiate0));

    NN_SDK_REQUIRES(!m_ServerChallenge);
    NN_SDK_REQUIRES(m_ClientChallenge);

    KeyExchangeCommandConfig::Passphrase passphrase = {};
    m_ServerChallenge.emplace();
    NN_RESULT_THROW_UNLESS(ParseInitiate0Response(passphrase, m_ServerChallenge->data, response, m_ClientChallenge->data, m_KeyEncryptor), ResultInvalidResponseDataForInitiate0());
    m_MessageEncryptor.Initialize(passphrase, sizeof(passphrase), EncryptionPolicy::KeyStretchingSalt, sizeof(EncryptionPolicy::KeyStretchingSalt), EncryptionPolicy::KeyStretchingIteration, 0);
    ClearClientChallenge();
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::ConsumeInitiate1Response(const Initiate1Response& response) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckCommandAcceptability(CommandKind::Initiate1));

    NN_RESULT_THROW_UNLESS(ParseInitiate1Response(response, m_MessageEncryptor), ResultInvalidResponseDataForInitiate1());
    m_IsKeyExchanged = true;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::ConsumeResume0Response(const Resume0Response& resopnse) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckCommandAcceptability(CommandKind::Resume0));

    NN_SDK_REQUIRES(m_ClientChallenge);
    NN_SDK_REQUIRES(!m_ServerChallenge);

    m_ServerChallenge.emplace();
    NN_RESULT_THROW_UNLESS(ParseResume0Response(m_ServerChallenge->data, resopnse, m_ClientChallenge->data, m_MessageEncryptor), ResultInvalidResponseDataForResume0());
    ClearClientChallenge();
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::ConsumeResume1Response(const Resume1Response& response) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckCommandAcceptability(CommandKind::Resume1));

    NN_RESULT_THROW_UNLESS(ParseResume1Response(response, m_MessageEncryptor), ResultInvalidResponseDataForResume1());
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::ConsumeTerminateResponse(const TerminateResponse& response) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckCommandAcceptability(CommandKind::Terminate));

    NN_RESULT_THROW_UNLESS(ParseTerminateResponse(response, m_MessageEncryptor), ResultInvalidResponseDataForTerminate());
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::ConsumeErrorResponse(const ErrorResponse& response) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckCommandAcceptability(CommandKind::Error));

    ErrorInfo errorInfo;
    NN_RESULT_THROW_UNLESS(ParseErrorResponse(&errorInfo, response, m_MessageEncryptor), ResultInvalidResponseDataForError());
    NN_RESULT_THROW(detail::HandleErrorInfo(errorInfo));
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::ProduceInitiate0Request(Initiate0Request* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(!m_IsKeyExchanged);
    NN_SDK_REQUIRES(!m_ClientChallenge);

    m_ClientChallenge.emplace();
    detail::GenerateRandomBytes(m_ClientChallenge->data, sizeof(m_ClientChallenge->data));
    CreateInitiate0Request(pOut, m_ClientChallenge->data, m_KeyEncryptor);
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::ProduceInitiate1Request(Initiate1Request* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(m_ServerChallenge);

    CreateInitiate1Request(pOut, m_ServerChallenge->data, m_MessageEncryptor);
    ClearServerChallenge();
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::ProduceResume0Request(Resume0Request* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(m_IsKeyExchanged);
    NN_SDK_REQUIRES(!m_ClientChallenge);

    m_ClientChallenge.emplace();
    detail::GenerateRandomBytes(m_ClientChallenge->data, sizeof(m_ClientChallenge->data));
    CreateResume0Request(pOut, m_ClientChallenge->data, m_MessageEncryptor);
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::ProduceResume1Request(Resume1Request* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(m_IsKeyExchanged);
    NN_SDK_REQUIRES(m_ServerChallenge);

    CreateResume1Request(pOut, m_ServerChallenge->data, m_MessageEncryptor);
    ClearServerChallenge();
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::ProduceTerminateRequest(TerminateRequest* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES(m_IsKeyExchanged);

    CreateTerminateRequest(pOut, m_MessageEncryptor);
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::Consumer::Consume(void* stream, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(stream);
    NN_SDK_REQUIRES_GREATER(size, 0u);

    if( !m_ConsumeCommandKind )
    {
        auto command = static_cast<CommandKind>(reinterpret_cast<Bit8*>(stream)[0]);
        NN_DETAIL_MIGRATION_TRACE("ClientContext::Consume : %s (0x%02X)\n", detail::GetCommandKindString(command), static_cast<Bit8>(command));
        m_ConsumeCommandKind = command;
        m_AvailableConsumableDataSize = 0u;
        m_ExpectedConsumableDataSize = GetResponseCommandSize(command);
    }

    std::memcpy(m_ConsumableData + m_AvailableConsumableDataSize, stream, std::min(size, sizeof(m_ConsumableData) - m_AvailableConsumableDataSize));
    m_AvailableConsumableDataSize += size;
    NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize >= m_ExpectedConsumableDataSize, ResultConsumeCommandContinue());

    switch( *m_ConsumeCommandKind )
    {
    case CommandKind::Initiate0:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(Initiate0Response), ResultInvalidResponseSizeForInitiate0());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pResponse = reinterpret_cast<Initiate0Response*>(m_ConsumableData);
            return m_pContext->ConsumeInitiate0Response(*pResponse);
        }
    case CommandKind::Initiate1:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(Initiate1Response), ResultInvalidResponseSizeForInitiate1());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pResponse = reinterpret_cast<Initiate1Response*>(m_ConsumableData);
            return m_pContext->ConsumeInitiate1Response(*pResponse);
        }
    case CommandKind::Resume0:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(Resume0Response), ResultInvalidResponseSizeForResume0());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pResponse = reinterpret_cast<Resume0Response*>(m_ConsumableData);
            return m_pContext->ConsumeResume0Response(*pResponse);
        }
    case CommandKind::Resume1:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(Resume1Response), ResultInvalidResponseSizeForResume1());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pResponse = reinterpret_cast<Resume1Response*>(m_ConsumableData);
            return m_pContext->ConsumeResume1Response(*pResponse);
        }
    case CommandKind::Terminate:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(TerminateResponse), ResultInvalidResponseSizeForTerminate());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pResponse = reinterpret_cast<TerminateResponse*>(m_ConsumableData);
            return m_pContext->ConsumeTerminateResponse(*pResponse);
        }
    case CommandKind::Error:
        {
            NN_RESULT_THROW_UNLESS(m_AvailableConsumableDataSize == sizeof(ErrorResponse), ResultInvalidResponseSizeForError());
            NN_UTIL_SCOPE_EXIT{ ClearBufferedConsumableData(); };
            auto pResponse = reinterpret_cast<ErrorResponse*>(m_ConsumableData);
            return m_pContext->ConsumeErrorResponse(*pResponse);
        }
    case CommandKind::User:
        {
            NN_RESULT_DO(m_pContext->CheckCommandAcceptability(CommandKind::User));
            // ユーザーコマンドは ClientUserCommandConsumer/ThreadedClientUserCommandConsumer で処理する。
            NN_RESULT_THROW(ResultInvalidResponseConsumerForUser());
        }

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

template <typename EncryptionPolicy>
Result ClientContext<EncryptionPolicy>::Producer::Produce(size_t* pOutProducedSize, void* outStream, size_t outStreamSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutProducedSize);
    NN_SDK_REQUIRES_NOT_NULL(outStream);
    NN_SDK_REQUIRES_GREATER(outStreamSize, 0u);
    NN_UNUSED(outStreamSize);

    NN_DETAIL_MIGRATION_TRACE("ClientContext::Produce : %s (0x%02X)\n",
        detail::GetCommandKindString(m_pContext->m_CommandKind), static_cast<Bit8>(m_pContext->m_CommandKind));

    switch( m_pContext->m_CommandKind )
    {
    case CommandKind::Initiate0:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(Initiate0Request));
            auto pRequest = reinterpret_cast<Initiate0Request*>(outStream);
            m_pContext->ProduceInitiate0Request(pRequest);
            *pOutProducedSize = sizeof(Initiate0Request);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::Initiate1:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(Initiate1Request));
            auto pRequest = reinterpret_cast<Initiate1Request*>(outStream);
            m_pContext->ProduceInitiate1Request(pRequest);
            *pOutProducedSize = sizeof(Initiate1Request);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::Resume0:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(Resume0Request));
            auto pRequest = reinterpret_cast<Resume0Request*>(outStream);
            m_pContext->ProduceResume0Request(pRequest);
            *pOutProducedSize = sizeof(Resume0Request);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::Resume1:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(Resume1Request));
            auto pRequest = reinterpret_cast<Resume1Request*>(outStream);
            m_pContext->ProduceResume1Request(pRequest);
            *pOutProducedSize = sizeof(Resume1Request);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::Terminate:
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(outStreamSize, sizeof(TerminateRequest));
            auto pRequest = reinterpret_cast<TerminateRequest*>(outStream);
            m_pContext->ProduceTerminateRequest(pRequest);
            *pOutProducedSize = sizeof(TerminateRequest);
            NN_RESULT_SUCCESS;
        }
    case CommandKind::User:  // ユーザーコマンドは ClientUserCommandProducer で処理する。
    case CommandKind::Error: // クライアントからエラーコマンドは送らない。
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::Consumer::ClearBufferedConsumableData() NN_NOEXCEPT
{
    m_ConsumeCommandKind = nullptr;
    m_ExpectedConsumableDataSize = 0;
    m_AvailableConsumableDataSize = 0;
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::ClearClientChallenge() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ClientChallenge);
    detail::SecureMemoryZero(m_ClientChallenge->data, sizeof(m_ClientChallenge->data));
    m_ClientChallenge = nullptr;
}

template <typename EncryptionPolicy>
void ClientContext<EncryptionPolicy>::ClearServerChallenge() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ServerChallenge);
    detail::SecureMemoryZero(m_ServerChallenge->data, sizeof(m_ServerChallenge->data));
    m_ServerChallenge = nullptr;
}

template <typename EncryptionPolicy, typename UserContextType>
class ClientUserCommandConsumer : public ClientContext<EncryptionPolicy>::Consumer
{
public:
    ClientUserCommandConsumer(
        ClientContext<EncryptionPolicy>* pContext,
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        UserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
        : ClientContext<EncryptionPolicy>::Consumer(pContext)
        , m_Mediator(messageEncryptor, pUserContext, pWorkBuffer)
    {
    }
    Result Consume(void* stream, size_t size) NN_NOEXCEPT
    {
        if( !m_CommandKind )
        {
            m_CommandKind = static_cast<CommandKind>(reinterpret_cast<Bit8*>(stream)[0]);
        }
        if( m_CommandKind == CommandKind::User )
        {
            NN_RESULT_DO(this->m_pContext->CheckUserCommandAcceptability());
            return m_Mediator.Consume(stream, size);
        }
        else
        {
            return ClientContext<EncryptionPolicy>::Consumer::Consume(stream, size);
        }
    }
private:
    typedef UserCommandConsumeMediator<EncryptionPolicy, UserContextType, ClientUserCommandErrorConfig> Mediator;
    Mediator m_Mediator;
    util::optional<CommandKind> m_CommandKind;
};

template <typename EncryptionPolicy, typename UserContextType>
class ThreadedClientUserCommandConsumer : public ClientContext<EncryptionPolicy>::Consumer
{
public:
    ThreadedClientUserCommandConsumer(
        ClientContext<EncryptionPolicy>* pContext,
        typename EncryptionPolicy::MessageEncryptor& messageEncryptor,
        UserContextType* pUserContext,
        migration::detail::ThreadResourceType* pThreadResource,
        ThreadedUserCommandMediatorWorkBuffer<EncryptionPolicy>* pWorkBuffer) NN_NOEXCEPT
        : ClientContext<EncryptionPolicy>::Consumer(pContext)
        , m_Mediator(messageEncryptor, pUserContext, pThreadResource, pWorkBuffer)
    {
    }
    Result Consume(void* stream, size_t size) NN_NOEXCEPT
    {
        if( !m_CommandKind )
        {
            m_CommandKind = static_cast<CommandKind>(reinterpret_cast<Bit8*>(stream)[0]);
        }
        if( m_CommandKind == CommandKind::User )
        {
            NN_RESULT_DO(this->m_pContext->CheckUserCommandAcceptability());
            return m_Mediator.Consume(stream, size);
        }
        else
        {
            return ClientContext<EncryptionPolicy>::Consumer::Consume(stream, size);
        }
    }
private:
    typedef ThreadedUserCommandConsumeMediator<EncryptionPolicy, UserContextType, ClientUserCommandErrorConfig> Mediator;
    Mediator m_Mediator;
    util::optional<CommandKind> m_CommandKind;
};

}}} // ~nn::migration::idc
