﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/account/account_Types.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/migration/migration_UserMigrationTypes.h>
#include <nn/migration/idc/migration_SocketConnection.h>
#include <nn/migration/detail/migration_Authenticator.h>
#include <nn/migration/detail/migration_IdcClient.h>
#include <nn/migration/detail/migration_IdcServer.h>
#include <nn/migration/detail/migration_LoginSession.h>
#include <nn/migration/detail/migration_Mutex.h>
#include <nn/nifm/nifm_NetworkConnection.h>

namespace nn { namespace migration { namespace detail {

class LdnServerConnectionCreator
{
private:
    os::Mutex m_Lock;
    std::atomic<bool> m_IsLdnInitialized;
    ldn::NetworkInfo m_NetworkInfoBuffer;

    Result EnableLdn() NN_NOEXCEPT;

    Result CreateNetwork(
        int* pOutListenSocket,
        const account::NetworkServiceAccountId& nsaId, const UserMigrationServerProfile& profile,
        const detail::Cancellable* pCancellable) NN_NOEXCEPT;

public:
    LdnServerConnectionCreator() NN_NOEXCEPT;
    ~LdnServerConnectionCreator() NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;

    template <typename EncryptionPolicy>
    Result CreateConnection(
        idc::SocketConnection* pOut,
        const account::NetworkServiceAccountId& nsaId, const UserMigrationServerProfile& profile,
        IdcServer<idc::SocketConnection, EncryptionPolicy>& server,
        const detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

class LdnClientConnectionCreator
{
private:
    static const size_t ScanCountMax = 32;
    static const size_t ListCountMax = 16;

    os::Mutex m_Lock;
    std::atomic<bool> m_IsLdnInitialized;
    ldn::NetworkInfo m_NetworkInfoBuffer;

    ldn::NetworkInfo m_ScannedInfos[ScanCountMax];
    size_t m_ServerCount;
    struct Server
    {
        util::Uuid id;
        ldn::NetworkInfo info;
    } m_Servers[ListCountMax];

    Result EnableLdnIfNot() NN_NOEXCEPT;
    Result ConnectImpl(
        idc::SocketConnection* pOut,
        const ldn::NetworkInfo& server, const account::NetworkServiceAccountId& nsaId,
        const Cancellable* pCancellable) NN_NOEXCEPT;

public:
    LdnClientConnectionCreator() NN_NOEXCEPT;
    ~LdnClientConnectionCreator() NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;

    Result ScanServers(
        size_t* pOutActual, UserMigrationServerInfo* pOut, size_t count,
        const account::NetworkServiceAccountId& nsaId,
        const detail::Cancellable* pCancellable) NN_NOEXCEPT;

    template <typename EncryptionPolicy>
    Result Connect(
        idc::SocketConnection* pOut,
        const util::Uuid& serverId, const account::NetworkServiceAccountId& nsaId,
        IdcClient<idc::SocketConnection, EncryptionPolicy>& client,
        const detail::Cancellable* pCancellable) NN_NOEXCEPT;
};

}}} // ~namespace nn::migration::detail


#include <nn/ldn/ldn_Api.h>
#include <nn/migration/migration_Result.h>
#include <nn/migration/detail/migration_ConnectionUtil.h>
#include <nn/migration/idc/detail/migration_HandleSocketError.h>
#include <nn/socket/socket_Api.h>

namespace nn { namespace migration { namespace detail {

template <typename EncryptionPolicy>
Result LdnServerConnectionCreator::CreateConnection(
    idc::SocketConnection* pOut,
    const account::NetworkServiceAccountId& nsaId, const UserMigrationServerProfile& profile,
    IdcServer<idc::SocketConnection, EncryptionPolicy>& server,
    const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    // LDN 初期化
    NN_MIGRATION_DETAIL_TRACE("[LdnServerConnectionCreator] Creating LDN network\n");
    NN_RESULT_DO(EnableLdn());
    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            Reset();
        }
    };

    // LDN 接続
    int acceptSocket;
    NN_RESULT_DO(CreateNetwork(&acceptSocket, nsaId, profile, pCancellable));
    idc::SocketConnection listenPort(acceptSocket);

    // クライアント接続待ち
    NN_MIGRATION_DETAIL_TRACE("[LdnServerConnectionCreator] Waiting for socket connection\n");
    {
        idc::SocketConnection con;
        NN_RESULT_DO(WaitConnection(&con, acceptSocket, GetTransferLowSpeedTimeoutSeconds(), pCancellable));

        // OTT の送信
        NN_RESULT_DO(server.HandleOnetimeTokenRequest(con, pCancellable));

        // クライアントはインターネットに接続するため、一度切断する
        while (NN_STATIC_CONDITION(true))
        {
            size_t _dummyReceived;
            Bit8 _dummy[8] = {};
            auto r = con.Receive(&_dummyReceived, _dummy, sizeof(_dummy), 0, pCancellable);
            if (!r.IsSuccess())
            {
                NN_RESULT_THROW_UNLESS(idc::ResultPeerClosed::Includes(r), r);
                break;
            }
            NN_MIGRATION_DETAIL_TRACE("[LdnServerConnectionCreator] Chunk %zu bytes received\n", _dummyReceived);
        }
        con.Close();
    }

    // クライアント再接続待ち
    NN_MIGRATION_DETAIL_TRACE("[LdnServerConnectionCreator] Waiting for socket re-connection\n");
    idc::SocketConnection con;
    NN_RESULT_DO(WaitConnection(&con, acceptSocket, GetTransferLowSpeedTimeoutSeconds(), pCancellable));

    // LDN 受付終了にする
    NN_MIGRATION_DETAIL_TRACE("[LdnServerConnectionCreator] done\n");
    NN_RESULT_DO(ldn::SetAdvertiseData(nullptr, 0));
    NN_RESULT_DO(ldn::SetStationAcceptPolicy(ldn::AcceptPolicy_AlwaysReject));

    // fsKeySeed 検証
    NN_RESULT_DO(server.HandleFsKeySeedVerificationRequest(con, pCancellable));

    *pOut = std::move(con);
    success = true;
    NN_RESULT_SUCCESS;
}

template <typename EncryptionPolicy>
Result LdnClientConnectionCreator::Connect(
    idc::SocketConnection* pOut,
    const util::Uuid& serverId, const account::NetworkServiceAccountId& nsaId,
    IdcClient<idc::SocketConnection, EncryptionPolicy>& client,
    const detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    const ldn::NetworkInfo* pServer = nullptr;
    for (size_t i = 0; i < m_ServerCount; ++ i)
    {
        const auto& s = m_Servers[i];
        if (s.id == serverId)
        {
            pServer = &s.info;
            break;
        }
    }
    NN_RESULT_THROW_UNLESS(pServer, ResultInvalidProtocol());

    NN_MIGRATION_DETAIL_TRACE("[LdnClientConnectionCreator] Connecting to access point (1 of 2)\n");
    {
        NN_RESULT_DO(EnableLdnIfNot());
        NN_UTIL_SCOPE_EXIT
        {
            // NOTE: 成功, 失敗に関わらず LDN を終了
            Reset();
        };

        idc::SocketConnection con;
        NN_RESULT_DO(ConnectImpl(&con, *pServer, nsaId, pCancellable));

        // OTT 取得
        NN_RESULT_DO(client.GetOnetimeToken(con, pCancellable));

        // 一度切断する
        con.Close();
        NN_RESULT_DO(ldn::Disconnect());
    }

    // FsKeySeed 取得
    NN_RESULT_DO(client.AuthenticateWithOnetimeToken(pCancellable));

    NN_MIGRATION_DETAIL_TRACE("[LdnClientConnectionCreator] Connecting to access point (2 of 2)\n");
    NN_RESULT_DO(EnableLdnIfNot());
    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            Reset();
        }
    };
    idc::SocketConnection con;
    NN_RESULT_DO(ConnectImpl(&con, *pServer, nsaId, pCancellable));

    // FsKeySeed 検証
    NN_RESULT_DO(client.VerifyFsKeySeed(con, pCancellable));

    NN_MIGRATION_DETAIL_TRACE("[LdnClientConnectionCreator] done\n");
    success = true;
    *pOut = std::move(con);
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::detail
