﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ldn/ldn_Types.h>

namespace nn { namespace ldn
{
    //! @name 通信状況を監視する API
    //! @{

    /**
     * @brief         ローカル通信ライブラリを監視モードで初期化します。
     *
     * @pre
     * - LDN ライブラリが監視モードで初期化されていないこと
     *
     * @details
     * ネットワークの構築や接続が目的では無く、
     * ローカル通信の使用状況を監視することが目的の場合には @ref Initialize 関数ではなく、
     * この関数を使用して LDN ライブラリを初期化してください。
     *
     * 使用できる機能は以下の関数に限定されてしまいますが、
     * 他のアプリケーションやアプレットがローカル通信を使用している場合であっても、
     * 並行して LDN ライブラリを使用することができます。
     *
     * - @ref GetStateForMonitor 関数
     * - @ref GetNetworkInfoForMonitor 関数
     * - @ref GetIpv4AddressForMonitor 関数
     *
     * 監視モードの使用を終了する際には、
     * @ref Finalize 関数ではなく @ref FinalizeMonitor 関数を呼びだしてください。
     */
    void InitializeMonitor() NN_NOEXCEPT;

    /**
     * @brief         ローカル通信ライブラリを解放します。
     *
     * @pre
     * - @ref InitializeMonitor 関数を使用して LDN ライブラリを監視モードで初期化していること
     *
     * @details
     * LDN ライブラリを @ref InitializeMonitor 関数で初期化した場合、
     * @ref Finalize 関数の代わりにこの関数を呼びだしてください。
     */
    void FinalizeMonitor() NN_NOEXCEPT;

    /**
     * @brief         ローカル通信の実行状態を取得します。
     *
     * @pre
     * - @ref InitializeMonitor 関数を使用して LDN ライブラリを監視モードで初期化していること
     *
     * @return        ローカル通信の実行状態です。
     *
     * @details
     * ローカル通信の実行状態を取得します。
     *
     * @ref GetState 関数は自分自身がローカル通信を開始した場合にしか状態を取得できません。
     * この関数では他のアプリケーションやアプレットがローカル通信を開始した場合でも、
     * ローカル通信の状態を取得できる点が異なります。
     */
    State GetStateForMonitor() NN_NOEXCEPT;

    /**
     * @brief         現在参加しているネットワークの情報を取得します。
     *
     * @param[out]    pOutNetwork   ネットワーク情報の出力先です。
     *
     * @pre
     * - @ref InitializeMonitor 関数を使用して LDN ライブラリを監視モードで初期化していること
     * - @a pOutNetwork != nullptr
     *
     * @retresult
     *  @handleresult{ResultSuccess}
     *  @handleresult{ResultInvalidState}
     * @endretresult
     */
    Result GetNetworkInfoForMonitor(NetworkInfo* pOutNetwork) NN_NOEXCEPT;

    /**
     * @brief         自身の IP アドレスを取得します。
     *
     * @param[out]    pOutAddress   IP アドレスの出力先です。
     * @param[out]    pOutMask      サブネットマスクの出力先です。
     *
     * @pre
     * - @ref InitializeMonitor 関数を使用して LDN ライブラリを監視モードで初期化していること
     * - @a pOutAddress != nullptr
     * - @a pOutMask != nullptr
     *
     * @retresult
     *  @handleresult{ResultSuccess}
     *  @handleresult{ResultInvalidState}
     * @endretresult
     */
    Result GetIpv4AddressForMonitor(Ipv4Address* pOutAddress, SubnetMask* pOutMask) NN_NOEXCEPT;

    //! @}

}} // end of namespace nn::ldn
