﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ldn/ldn_PrivateTypes.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceProfile.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkProfile.h>
#include <nn/ldn/detail/TcpIp/ldn_LdnAutoIp.h>
#include <nn/ldn/detail/TcpIp/ldn_TcpIpStackConfiguration.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace ldn { namespace detail { namespace impl
{
    /**
     * @brief           ノード情報です。
     */
    struct SessionNodeInfo
    {
        //! アクセスポイントから割り当てられた IPv4 アドレスです。
        Ipv4Address         ipv4Address;

        //! MAC アドレスです。
        MacAddress          macAddress;

        //! ノードの接続状態です。
        bool                isJoined;

        //! ノードの接続状態の変化です。
        Bit8                stateChange;

        //! ローカル通信バージョンです。
        int16_t             localCommunicationVersion;

        //! ノードが生成した疑似乱数です。
        Bit8                random[RandomSize];

        //! ユーザ名です。
        char                userName[UserNameBytesMax];

        //  予約領域です。
        char                _reserved2[16];
    };
    NN_STATIC_ASSERT(sizeof(SessionNodeInfo) == 80);

    /**
     * @brief           現在のセッションの情報です。
     */
    struct SessionContext
    {
        //! ネットワークの識別子です。
        NetworkId networkId;

        //! 自身のユーザ情報です。
        UserConfig user;

        //! 自身のセキュリティ設定です。
        SecurityConfig security;

        //! ネットワークに接続可能な最大端末数です。アクセスポイントも含みます。
        int8_t nodeCountMax;
        NN_PADDING1;

        //! ステーションの接続受入ポリシーです。
        Bit8 stationAcceptPolicy;
        NN_PADDING7;

        //! フィルタのエントリ数です。
        int16_t nextfilterEntryIndex;

        //! ステーションの接続を制限するフィルタです。
        MacAddress filterEntries[AcceptFilterEntryCountMax];

        //! ノード情報です。
        SessionNodeInfo nodes[NodeCountMax];

        //! 自身の Association ID です。
        int8_t aid;

        //! 自身の役割です。
        Bit8 role;

        //! ローカル通信識別子です。
        int16_t localCommunicationVersion;
        NN_PADDING4;
        NN_PADDING6;

        //! アドバータイズデータのサイズです。
        uint16_t advertiseDataSize;
        char advertiseData[AdvertiseDataSizeMax];

        //! ネットワーク・インタフェースです。
        NetworkInterfaceProfile networkInterfaceProfile;
    };
    NN_STATIC_ASSERT(sizeof(SessionContext) == 3120U);

    /**
     * @brief           SessionManager の初期化に使用するバッファです。
     */
    struct SessionManagerBuffer
    {
        char addressing[4096];
        SessionContext context;
        char _reserved[976];
    };
    NN_STATIC_ASSERT(sizeof(SessionManagerBuffer) == 8192U);

}}}} // namespace nn::ldn::detail::impl

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief           セッション上の役割です。
     */
    enum SessionRole
    {
        //! セッション管理を開始していません。
        SessionRole_None,

        //! セッションの管理者です。
        SessionRole_Administrator,

        //! セッションの参加者です。
        SessionRole_User
    };

    /**
     * @brief           ローカル通信のセッションを管理します。
     */
    class SessionManager
    {
    public:

        /**
         * @brief       初期化に必要なバッファサイズです。
         */
        static const size_t RequiredBufferSize = sizeof(impl::SessionManagerBuffer);

        /**
         * @brief       コンストラクタです。
         * @param[in]   buffer                  初期化に使用するバッファです。
         * @param[in]   bufferSize              初期化に使用するバッファのサイズです。
         * @pre
         * - buffer != nullptr
         * - RequiredBufferSize <= bufferSize
         */
        SessionManager(void* buffer, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです。
         */
        ~SessionManager() NN_NOEXCEPT;

        /**
         * @brief       SessionManager を初期化します。
         * @param[in]   profile                 使用するネットワーク・インタフェースです。
         * @param[in]   pStateChangeEvent       状態遷移時にシグナルするイベントです。
         * @pre
         * - SessionManager が初期化されていない、あるいは解放後であること
         */
        void Initialize(
            const NetworkInterfaceProfile& profile,
            nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT;

        /**
         * @brief       SessionManager を解放します。
         * @pre
         * - SessionManager が初期化されていること
         */
        void Finalize() NN_NOEXCEPT;

        /**
         * @brief       ネットワーク情報を取得します。
         * @param[out]  pOutNetworkId           ネットワーク識別子の出力先です。
         * @param[out]  pOutNetwork             ネットワーク情報の出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        Result GetNetworkInfo(
            NetworkId* pOutNetworkId, LdnNetworkInfo* pOutNetwork) const NN_NOEXCEPT;

        /**
         * @brief       ネットワーク情報を取得します。
         * @param[out]  pOutUpdates             ネットワークに参加しているノードの変化の出力先です。
         * @param[in]   bufferCount             pOutUpdates に格納できる NodeLatestUpdate の数です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        Result GetNodeLatestUpdate(NodeLatestUpdate* outUpdates, int bufferCount) NN_NOEXCEPT;

        /**
         * @brief       自身の IPv4 アドレスを取得します。
         * @param[out]  pOutNetworkId           ネットワーク識別子の出力先です。
         * @param[out]  pOutNetwork             ネットワーク情報の出力先です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        Result GetIpv4Address(Ipv4Address* pOutAddress, SubnetMask* pOutMask) const NN_NOEXCEPT;

        /**
         * @brief       ユーザ情報を設定します。
         * @param[in]   user                    ユーザ情報です。
         * @pre
         * - SessionManager が初期化されていること
         * - セッションに参加していないこと
         */
        void SetUser(const UserConfig& user) NN_NOEXCEPT;

        /**
         * @brief       セキュリティの設定を行います。
         * @param[in]   securityMode            セキュリティ・モードです。
         * @param[in]   serverRandom            サーバーが生成した乱数です。
         * @pre
         * - SessionManager が初期化されていること
         * - セッションに参加していないこと
         */
        void SetSecurity(
            const SecurityConfig& security, const Bit8 (&serverRandom)[RandomSize]) NN_NOEXCEPT;

        /**
         * @brief       ローカル通信バージョンを設定します。
         * @param[in]   version                 ローカル通信バージョンです。
         * @pre
         * - SessionManager が初期化されていること
         * - セッションに参加していないこと
         */
        void SetLocalCommunicationVersion(int version) NN_NOEXCEPT;

        /**
         * @brief       データ通信に使用するセッション鍵を生成します。
         * @param[out]  buffer                  鍵の出力先です。
         * @param[out]  pOutSize                鍵サイズの出力先です。
         * @param[in]   bufferSize              buffer のバイトサイズです。
         * @pre
         * - SessionManager が初期化されていること
         */
        void CreateDataKey(
            void* buffer, size_t* pOutSize, size_t bufferSize) const NN_NOEXCEPT;

        /**
         * @brief       ノードを検索します。
         * @param[in]   ipv4Address             ノードの IPv4 アドレスです。
         * @pre
         * - SessionManager が初期化されていること
         * @retval      -1                      見つかりませんでした。
         * @retval      上記以外                ステーションの AID です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultNodeNotFound}
         * @endretresult
         */
        int Find(Ipv4Address ipv4Address) const NN_NOEXCEPT;

        /**
         * @brief       セッションを構築します。
         * @param[in]   networkId               ネットワーク識別子です。
         * @param[in]   nodeCountMax            ノードの最大数です。
         * @param[in]   addressEntryCount       静的に割り当てる IPv4 アドレスの数です。
         * @param[in]   addressEntries          静的に割り当てる IPv4 アドレスです。
         * @pre
         * - SessionManager が初期化されていること
         * - nodeCountMax が 1 以上 NodeCountMax 以下であること
         */
        void CreateSession(
            const NetworkId& networkId, int nodeCountMax,
            const AddressEntry* addressEntries, int addressEntryCount) NN_NOEXCEPT;

        /**
         * @brief       セッションを破棄します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultBadRequest}
         * @endretresult
         * @pre
         * - CreateSession でセッションを構築していること
         */
        void DestroySession() NN_NOEXCEPT;

        /**
         * @brief       アドバータイズ・データを設定します。
         * @param[in]   data                    アドバータイズ・データです。
         * @param[in]   dataSize                アドバータイズ・データのバイトサイズです。
         * @pre
         * - SessionManager が初期化されていること
         */
        void SetAdvertiseData(const void* data, size_t dataSize) NN_NOEXCEPT;

        /**
         * @brief       ステーションの受け入れポリシーを設定します。
         * @param[in]   policy                  ステーションの受け入れポリシーです。
         * @pre
         * - SessionManager が初期化されていること
         */
        void SetStationAcceptPolicy(AcceptPolicy policy) NN_NOEXCEPT;

        /**
         * @brief       セッションへの追加可否を判定します。
         * @param[in]   macAddress              ユーザの MAC アドレスです。
         * @pre
         * - SessionManager が初期化されていること
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultConnectionRejected}
         * @endretresult
         */
        Result IsAcceptable(MacAddress macAddress) const NN_NOEXCEPT;

        /**
         * @brief       ステーションをセッションに追加します。
         * @param[in]   aid                     ステーションの AID です。
         * @param[in]   macAddress              ステーションの MAC アドレスです。
         * @param[in]   version                 ローカル通信バージョンです。
         * @param[in]   clientRandom            ステーションが生成した乱数です。
         * @param[in]   user                    ユーザ情報です。
         * @pre
         * - SessionManager が初期化されていること
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultConnectionRejected}
         * @endretresult
         */
        Result AddUser(
            int aid,
            MacAddress macAddress,
            int version,
            const Bit8 (&clientRandom)[RandomSize],
            const UserConfig& user) NN_NOEXCEPT;

        /**
         * @brief       ステーションをセッションから除外します。
         * @param[in]   aid                     ステーションの AID です。
         * @pre
         * - SessionManager が初期化されていること
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         *  @handleresult{ResultNodeNotFound}
         * @endretresult
         */
        Result RemoveUser(int aid) NN_NOEXCEPT;

        /**
         * @brief       ステーションの接続を制限するフィルタにエントリを追加します。
         * @param[in]   macAddress             対象のステーションの MAC アドレスです。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        Result AddAcceptFilterEntry(MacAddress macAddress) NN_NOEXCEPT;

        /**
         * @brief       フィルタに登録されているエントリを全て削除します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultInvalidState}
         * @endretresult
         */
        Result ClearAcceptFilter() NN_NOEXCEPT;

        /**
         * @brief       セッションに参加します。
         * @param[in]   networkId               ネットワーク識別子です。
         * @param[in]   ldn                     対象のネットワークの情報です。
         * @param[in]   aid                     自身の AID です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultLowerVersion}
         *  @handleresult{ResultHigherersion}
         * @endretresult
         * @pre
         * - SessionManager が初期化されていること
         */
        Result JoinSession(
            const NetworkId& networkId, const LdnNetworkInfo& ldn, int aid) NN_NOEXCEPT;

        /**
         * @brief       セッションから離脱します。
         */
        void LeaveSession() NN_NOEXCEPT;

        /**
         * @brief       セッションの状態を更新します。
         * @param[in]   ldn                     対象のネットワークの情報です。
         * @pre
         * - SessionManager が初期化されていること
         */
        Result UpdateNetworkInfo(const LdnNetworkInfo& ldn) NN_NOEXCEPT;

    private:

        void StartUp(IAddressGetter* pAddressGetter) NN_NOEXCEPT;
        void Down() NN_NOEXCEPT;
        bool UpdateAddress(
            int aid, const impl::SessionNodeInfo& prev, const NodeInfo& current) NN_NOEXCEPT;
        Result UpdateNetworkInfoImpl(bool* pOutIsUpdated, const LdnNetworkInfo& ldn) NN_NOEXCEPT;
        int GetNodeCount() const NN_NOEXCEPT;
        void SetJoin(int aid, bool isJoined) NN_NOEXCEPT;

        nn::os::SystemEventType* m_pStateChangeEvent;
        mutable nn::os::Mutex m_Mutex;
        impl::SessionManagerBuffer* m_Buffer;
        TcpIpStackConfiguration m_TcpIpStackConfiguration;
        LdnAutoIpServer m_AddressingServer;
        LdnAutoIpClient m_AddressingClient;
        IAddressGetter* m_pAddressGetter;
        bool m_IsInitialized;
    };

}}} // namespace nn::ldn::detail
