﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/TcpIp/ldn_Addressing.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         LDN 独自のプロトコルで AP から STA に IP アドレスを払い出します。
     *
     * @details       この方式ではアクセスポイントが IP アドレスを集中管理します。
     *
     *                割り当てられる IP アドレスは 169.254.x.y/24 の形式で、
     *                1 <= x <= 127 かつ 1 <= y <= 127 が常に成り立ちます。
     *                128 <= x <= 254 と 128 <= y <= 254 は将来の拡張に備えて予約されています。
     *
     *                一度切断したステーションの IP アドレスはアクセスポイントに返却されて、
     *                他のステーションが利用できる状態になりますが、原則として、
     *                MAC アドレスを参照して再接続時にも同じ IP アドレスを割り当てるように制御されています。
     *                ただし、割り当てられる IP アドレスの数には限りがありますので、
     *                多数の端末が接続してきた場合には IP アドレスが変動してしまう可能性もあります。
     */
    class LdnAutoIpServer : public IAddressingServer
    {
    public:

        /**
         * @brief           初期化に必要なバッファサイズを取得します。
         * @return          初期化に必要なバッファサイズです。
         */
        static size_t GetRequiredBufferSize() NN_NOEXCEPT;

        /**
         * @brief           コンストラクタです。
         * @param[in]       buffer          内部で使用するバッファです。
         * @param[in]       bufferSize      buffer のバイトサイズです。
         */
        LdnAutoIpServer(void* buffer, size_t bufferSize) NN_NOEXCEPT;

        /**
         * @brief           デストラクタです。
         */
        virtual ~LdnAutoIpServer() NN_NOEXCEPT;

        virtual void AddEntry(
            Ipv4Address ipv4Address, MacAddress macAddress) NN_NOEXCEPT NN_OVERRIDE;

        virtual void ClearEntries() NN_NOEXCEPT NN_OVERRIDE;

        virtual void StartServer(MacAddress macAddress) NN_NOEXCEPT NN_OVERRIDE;

        virtual void StopServer() NN_NOEXCEPT NN_OVERRIDE;

        virtual Ipv4Address Assign(MacAddress macAddress) NN_NOEXCEPT NN_OVERRIDE;

        virtual void Free(Ipv4Address ipv4Address) NN_NOEXCEPT NN_OVERRIDE;

        virtual Ipv4Address GetIpv4Address() const NN_NOEXCEPT NN_OVERRIDE;

        virtual Ipv4Address GetGatewayAddress() const NN_NOEXCEPT NN_OVERRIDE;

        virtual SubnetMask GetSubnetMask() const NN_NOEXCEPT NN_OVERRIDE;

    private:

        struct AddressInfo
        {
            uint32_t    counter;
            MacAddress  macAddress;
            bool        isAssigned;
            Bit8        _reserved[1];
        };

        AddressInfo* m_AddressSpace;
        uint32_t m_Counter;
        Ipv4Address m_NetworkAddress;
        Ipv4Address m_MyAddress;
    };

    /**
     * @brief         LDN 独自のプロトコルで割り当てられた IP アドレスを管理します。
     */
    class LdnAutoIpClient : public IAddressingClient
    {
    public:

        LdnAutoIpClient() NN_NOEXCEPT;

        virtual ~LdnAutoIpClient() NN_NOEXCEPT;

        virtual void StartClient(Ipv4Address server, Ipv4Address client) NN_NOEXCEPT NN_OVERRIDE;

        virtual void StopClient() NN_NOEXCEPT NN_OVERRIDE;

        virtual Ipv4Address GetIpv4Address() const NN_NOEXCEPT NN_OVERRIDE;

        virtual Ipv4Address GetGatewayAddress() const NN_NOEXCEPT NN_OVERRIDE;

        virtual SubnetMask GetSubnetMask() const NN_NOEXCEPT NN_OVERRIDE;

    private:

        Ipv4Address     m_MyAddress;
        Ipv4Address     m_NetworkAddress;
        Ipv4Address     m_Gateway;
    };

}}} // namespace nn::ldn::detail
