﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceProfile.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         IP アドレス割り当てのインタフェースです。
     */
    class IAddressGetter
    {
    public:

        /**
         * @brief           デストラクタです。
         */
        virtual ~IAddressGetter()
        {
        }

        /**
         * @brief           自身の IP アドレスを取得します。
         * @return          自身の IP アドレスです。
         */
        virtual Ipv4Address GetIpv4Address() const NN_NOEXCEPT = 0;

        /**
         * @brief           ゲートウェイアドレスを取得します。
         * @return          ゲートウェイアドレスです。
         */
        virtual Ipv4Address GetGatewayAddress() const NN_NOEXCEPT = 0;

        /**
         * @brief           サブネットマスクを取得します。
         * @return          サブネットマスクです。
         */
        virtual SubnetMask GetSubnetMask() const NN_NOEXCEPT = 0;
    };

    /**
     * @brief           IP アドレス割り当てのサーバ側インタフェースです。
     */
    class IAddressingServer : public IAddressGetter
    {
    public:

        /**
         * @brief           デストラクタです。
         */
        virtual ~IAddressingServer() NN_NOEXCEPT
        {
        }

        /**
         * @brief           静的エントリを追加します。
         * @param[in]       ipv4Address     IP アドレスです。
         * @param[in]       macAddress      MAC アドレスです。
         */
        virtual void AddEntry(Ipv4Address ipv4Address, MacAddress macAddress) NN_NOEXCEPT = 0;

        /**
         * @brief           静的エントリをクリアします。
         */
        virtual void ClearEntries() NN_NOEXCEPT = 0;

        /**
         * @brief           サーバを稼働します。
         * @param[in]       macAddress      MAC アドレスです。
         */
        virtual void StartServer(MacAddress macAddress) NN_NOEXCEPT = 0;

        /**
         * @brief           サーバの稼働を停止します。
         */
        virtual void StopServer() NN_NOEXCEPT = 0;

        /**
         * @brief           IP アドレスを割り当てます。
         * @param[in]       macAddress      対象のステーションの MAC アドレスです。
         * @return          割り当てられた IP アドレスです。
         */
        virtual Ipv4Address Assign(MacAddress macAddress) NN_NOEXCEPT = 0;

        /**
         * @brief           アドレスの割り当てを解除します。
         * @param[in]       ipv4Address     返却する IP アドレスです。
         */
        virtual void Free(Ipv4Address ipv4Address) NN_NOEXCEPT = 0;
    };

    /**
     * @brief         IP アドレス割り当てのクライアント側インタフェースです。
     */
    class IAddressingClient : public IAddressGetter
    {
    public:

        /**
         * @brief           デストラクタです。
         */
        virtual ~IAddressingClient() NN_NOEXCEPT
        {
        }

        /**
         * @brief           クライアントとしての動作を開始します。
         * @param[in]       server          サーバの IPv4 アドレスです。
         * @param[in]       client          クライアントの IPv4 アドレスです。
         */
        virtual void StartClient(
            Ipv4Address server, Ipv4Address client) NN_NOEXCEPT = 0;

        /**
         * @brief           クライアントとしての動作を停止します。
         */
        virtual void StopClient() NN_NOEXCEPT = 0;
    };

}}} // end of namespace nn::ldn::detail
