﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief         プロトコル識別子のモジュールです。
     */
    enum ProtocolModule
    {
        ProtocolModule_Reserved,
        ProtocolModule_Ldn
    };

    /**
     * @brief         LDN が定義するプロトコルです。
     */
    enum ProtocolCode
    {
        ProtocolCode_Reserved,
        ProtocolCode_Advertise,
        ProtocolCode_Authentication,
        ProtocolCode_Reject
    };

    /**
     * @brief         プロトコル識別子です。
     */
    struct ProtocolId
    {
        Bit8 module;
        Bit8 code;
    };

    /**
     * @brief         LDN 用の ProtocolId を生成します。
     *
     * @param[in]     code        LDN で定義したプロトコルです。
     *
     * @return        生成された ProtocolId です。
     */
    inline const ProtocolId MakeProtocolId(ProtocolCode code) NN_NOEXCEPT
    {
        ProtocolId protocolId = { static_cast<Bit8>(ProtocolModule_Ldn), static_cast<Bit8>(code) };
        return protocolId;
    }

    /**
     * @brief         ProtocolId を比較します。
     *
     * @param[in]     lhs         比較対象の ProtocolId
     * @param[in]     rhs         比較対象の ProtocolId
     *
     * @return        ProtocolId が等しい場合に true です。
     */
    inline bool operator == (ProtocolId lhs, ProtocolId rhs) NN_NOEXCEPT
    {
        return lhs.module == rhs.module && lhs.code == rhs.code;
    }

    /**
     * @brief         ProtocolId を比較します。
     *
     * @param[in]     lhs         比較対象の ProtocolId
     * @param[in]     rhs         比較対象の ProtocolId
     *
     * @return        ProtocolId が等しくない場合に true です。
     */
    inline bool operator != (ProtocolId lhs, ProtocolId rhs) NN_NOEXCEPT
    {
        return !(lhs == rhs);
    }

    /**
     * @brief         アドバータイズ用の ProtocolId です。
     */
     const ProtocolId AdvertiseProtocolId = MakeProtocolId(ProtocolCode_Advertise);

    /**
     * @brief         認証用の ProtocolId です。
     */
     const ProtocolId AuthenticationProtocolId = MakeProtocolId(ProtocolCode_Authentication);

    /**
     * @brief         切断理由通知の ProtocolId です。
     */
     const ProtocolId RejectProtocolId = MakeProtocolId(ProtocolCode_Reject);

}}} // namespace nn::ldn::detail
