﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterface.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceRequestParameter.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
    #include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceRequestImpl-os.win.h>
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include <nn/ldn/detail/NetworkInterface/ldn_NetworkInterfaceRequestImpl-os.horizon.h>
#endif

namespace nn { namespace ldn { namespace detail
{
    /**
     * @brief       適切なネットワークインタフェースを選択して取得します。
     */
    class NetworkInterfaceRequest
    {
        NN_DISALLOW_COPY(NetworkInterfaceRequest);

    public:

        /**
         * @brief       コンストラクタです。
         */
        NetworkInterfaceRequest() NN_NOEXCEPT
        {
        }

        /**
         * @brief       ムーブコンストラクタです。
         * @param[in]   other           移動元のオブジェクトです。
         */
        NetworkInterfaceRequest(NetworkInterfaceRequest&& other) NN_NOEXCEPT
        {
            *this = std::move(other);
        }

        /**
         * @brief       デストラクタです。
         */
        ~NetworkInterfaceRequest() NN_NOEXCEPT
        {
        }

        /**
         * @brief       移動代入演算子です。
         * @param[in]   other           移動元のオブジェクトです。
         */
        NetworkInterfaceRequest& operator = (NetworkInterfaceRequest&& other) NN_NOEXCEPT
        {
            if (this != &other)
            {
                m_Impl = std::move(other.m_Impl);
            }
            return *this;
        }

        /**
         * @brief       ネットワークインタフェースの占有を要求します。
         * @param[in]   param           ネットワークインタフェースに対する要求です。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultDeviceDisabled}
         * @endretresult
         */
        Result Submit(const NetworkInterfaceRequestParameter& param) NN_NOEXCEPT
        {
            return m_Impl.Submit(param);
        }

        /**
         * @brief       自身が占有しているネットワークインタフェースを解放します。
         */
        void Release() NN_NOEXCEPT
        {
            return m_Impl.Release();
        }

        /**
         * @brief       占有中のネットワークインタフェースを取得します。
         * @return      占有中のネットワークインタフェースです。
         */
        INetworkInterface* GetNetworkInterface() NN_NOEXCEPT
        {
            return m_Impl.GetNetworkInterface();
        }

        /**
         * @brief       ネットワークインタフェースの状態を取得します。
         * @retresult
         *  @handleresult{ResultSuccess}
         *  @handleresult{ResultDeviceOccupied}
         *  @handleresult{ResultDeviceDisabled}
         * @endretresult
         */
        Result GetResult() const NN_NOEXCEPT
        {
            return m_Impl.GetResult();
        }

        /**
         * @brief       ネットワークインタフェースの占有状態変更を通知するイベントを取得します。
         * @return      ネットワークインタフェースの占有状態変更を通知するイベントです。
         */
        nn::os::SystemEvent& GetRequestStateChangeEvent() NN_NOEXCEPT
        {
            return m_Impl.GetRequestStateChangeEvent();
        }

    private:

        impl::NetworkInterfaceRequestImpl m_Impl;
    };

}}} // namespace nn::ldn::detail
