﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

 /**
 * @file
 * @brief       ローカルコンテンツ配信ライブラリの公開 API の宣言
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/lcs/lcs_Config.h>
#include <nn/lcs/lcs_Types.h>
#include <nn/os/os_SystemEventTypes.h>

namespace nn { namespace lcs
{
    //! @name 共通の API
    //! @{

    /**
    * @brief       LCS ライブラリを初期化します。
    *
    * @param[in]   buffer                  LCS ライブラリに与えるバッファです。
    * @param[in]   bufferSize              LCS ライブラリに与えるバッファのサイズです。
    * @param[in]   config                  LCS ライブラリの設定です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultLdnNotInitialized}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    * @endretresult
    *
    * @pre
    * - GetState() == @ref State_None
    * - SOCKET ライブラリが初期化されていること
    * - @a buffer != nullptr
    * - @a buffer が RequiredBufferAlignment バイトにアライメントされていること
    * - RequiredBufferSize <= @a bufferSize
    *
    * @post
    * - GetState() == @ref State_Initialized
    *
    * @details
    * LCS ライブラリを初期化します。
    *
    * LCS ライブラリを初期化する前に nn::ldn::InitializeSystem() で LDN ライブラリを、
    * nn::socket::Initialize() で SOCKET ライブラリを、初期化する必要があります。
    * Finalize() を実行して LCS ライブラリの使用を完全に終了するまで、
    * nn::ldn::FinalizeSystem() や nn::socket::Finalize() を実行してはいけません。
    *
    * LCS ライブラリでは本体更新を行う場合があるので、初期化時点で
    * nn::ns::SystemUpdateControl を生成し、Occupy() を実行します。
    * nn::ns::SystemUpdateControl を使用しているときは LCS ライブラリの初期化前に必ず
    * Relieve() しておいてください。
    * LCS ライブラリが終了されるまで、Occupy() した状態なので注意してください。
    *
    * @a buffer には @ref RequiredBufferSize バイト以上のバッファを、
    * @ref RequiredBufferAlignment バイトでアライメントして与えてください。
    * @a buffer で LCS ライブラリに供給したバッファは、
    * Finalize() で LCS ライブラリの使用を終了するまで読み書きしないでください。
    *
    * @a config は、セッションに参加したときに使用されるユーザ名を指定します。
    * nn::settings::system::GetDeviceNickName() で取得できるデバイス名や、
    * アカウント名などを利用してください。
    * デバイス名やアカウント名でない、適当な名前を設定しても動作に問題はありません。
    * ユーザ名の詳細については、@ Config を参照してください。
    */
    Result Initialize(void* buffer, size_t bufferSize, const Config& config) NN_NOEXCEPT;

    /**
    * @brief       LCS ライブラリを終了します。
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @post
    * - GetState() == @ref State_None
    *
    * @details
    * LCS ライブラリの使用を終了します。
    *
    * @ref State_Initialized 状態ではこの関数は即座に終了します。
    * @ref State_Opened, @ref State_Joined, @ref State_Transferring,
    * @ref State_ContentsShareFailed, @ref State_Suspended, @ref State_Completed 状態では
    * 自動的にセッションから離脱し、その後で LCS ライブラリの使用を終了するため、
    * この関数は最大数秒程度処理を返しません。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief       状態の変更を通知するイベントを取得します。
    *
    * @param[in]   pEvent                  イベントの出力先です。
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a pEvent != nullptr
    *
    * @details
    * 状態の変更を通知するイベントを取得します。
    *
    * 以下のいずれかの場合にイベントがシグナルされます。
    * - 自分自身の状態 (@ref nn::lcs::State "State") が変化したとき
    * - セッションクライアントの参加や離脱が発生したとき
    *
    * この関数で取得したイベントは自動的にはクリアされないため、
    * 必要に応じて使用者自身がシグナルをクリアしてください。
    *
    * 取得したイベントは Finalize() を呼ぶまで削除しないでください。
    */
    void AttachStateChangeEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    /**
    * @brief       現在のライブラリの状態を取得します。
    *
    * @return      ライブラリの状態です。
    *
    * @details
    * 現在のライブラリの状態を取得します。
    * 取得できる状態の詳細については State を参照してください。
    */
    State GetState() NN_NOEXCEPT;

    /**
    * @brief       現在参加しているセッションの情報を取得します。
    *
    * @param[out]  pOutInfo                セッション情報の出力先です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a pOutInfo != nullptr
    *
    * @details
    * 現在参加しているセッションの情報を取得します。
    *
    * ホストは OpenSession() でセッションを作成した後、
    * クライアントは JoinSession() でセッションに参加した後にこの関数を使用できます。
    * @ref State_Opened, @ref State_Joined, @ref State_Transferring,
    * @ref State_Suspended, @ref State_Completed
    * 以外の状態でこの関数を実行した場合は @ref ResultInvalidState を返します。
    *
    * SessionInfo はクライアントの参加や離脱によって変化するため、
    * AttachStateChangeEvent() で取得したイベントがシグナルされたとき、
    * 必要に応じてこの関数で最新の SessionInfo を取得してください。
    */
    Result GetSessionInfo(SessionInfo* pOutInfo) NN_NOEXCEPT;

    /**
    * @brief       自身のインデックスを取得します。
    *
    * @return      自身のインデックスです。
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @details
    * 現在参加しているセッションで与えられている自身のインデックスをを取得します。
    *
    * ホストは OpenSession() でセッションを作成した後、
    * クライアントは JoinSession() でセッションに参加した後、
    * この関数で自身のインデックスを取得できます。
    * @ref State_Initialized でこの関数を実行した場合は 0 を返します。
    */
    uint32_t GetMyIndex() NN_NOEXCEPT;

    /**
    * @brief       セッションに参加している端末のリストを取得します。
    *
    * @param[out]  outNodes                端末情報の出力先です。
    * @param[out]  pOutCount               @a outNodes に出力された端末の数です。
    * @param[in]   bufferCount             @a outNodes に格納できる端末の数です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a outNodes != nullptr
    * - @a pOutCount != nullptr
    * - 0 < @a bufferCount
    *
    * @details
    * セッションに参加している端末のリストを取得します。
    *
    * ホストは OpenSession() でセッションを作成した後、
    * クライアントは JoinSession() でセッションに参加した後にこの関数を使用できます。
    * @ref State_Opened, @ref State_Joined, @ref State_Transferring,
    * @ref State_Suspended, @ref State_Completed
    * 以外の状態でこの関数を実行した場合は @ref ResultInvalidState を返します。
    *
    * クライアントの参加や離脱があると AttachStateChangeEvent()
    * で取得したイベントがシグナルされます。
    * セッション内の全ての端末の情報が含まれますので、出力数は最大で NodeCountMax です。
    */
    Result GetNodes(NodeInfo* outNodes, int* pOutCount, int bufferCount) NN_NOEXCEPT;

    /**
    * @brief       コンテンツのインストールに必要な空き容量を取得します。
    *
    * @param[out]  pOutSize                コンテンツのインストールに必要な空き容量の出力先です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a pOutSize != nullptr
    *
    * @details
    * LCS ライブラリで受信するコンテンツのインストールに必要なストレージの空き容量を取得します。
    *
    * ホストは OpenSession() でセッションを作成した後、
    * クライアントは JoinSession() でセッションに参加した後にこの関数を使用できます。
    * @ref State_Opened, @ref State_Joined, @ref State_Transferring,
    * @ref State_Suspended, @ref State_Completed
    * 以外の状態でこの関数を実行した場合は @ref ResultInvalidState を返します。
    *
    * 受信するコンテンツはクライアントの参加・離脱によって変化するため、
    * @ref State_Opened, @ref State_Joined 状態で取得できるサイズはその時点での見積りで、
    * 最終的に必要なサイズとは一致しない可能性があります。
    * 確定したインストールサイズを取得するには
    * @ref State_Suspended 状態に遷移してからこの関数を実行してください。
    *
    * 受信するコンテンツが 1 つも無い場合には pOutSize に 0 が出力されます。
    */
    Result GetRequiredStorageSize(size_t* pOutSize) NN_NOEXCEPT;

    /**
    * @brief       セッションの状態を取得します。
    *
    * @param[out]  pOutState                セッションの状態の出力先です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a isDiscard != nullptr
    *
    * @details
    * コンテンツ配信時のセッションの状態を取得します。
    *
    * この関数は @ref State_Transferring 状態でのみ使用できます。
    * それ以外の状態で使用すると ResultInvalidState を返します。
    *
    * コンテンツの配信が始まったあとのセッションの状態を取得できます。
    * 取得できるセッションの状態の詳細については SessionState のリファレンスを参照してください。
    */
    Result GetSessionState(SessionState* pOutState) NN_NOEXCEPT;

    /**
    * @brief       コンテンツ配信の進捗を取得します。
    *
    * @param[out]  pOutProgress            コンテンツ配信の進捗の出力先です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a pOutProgress != nullptr
    *
    * @details
    * コンテンツ配信の進捗を取得します。
    *
    * StartContentsShare() でコンテンツの配信を開始した後、 @ref State_Transferring,
    * @ref State_Suspended, @ref State_Completed 状態のいずれかで使用できます。
    * それ以外の状態でこの関数を実行すると ResultInvalidState を返します。
    *
    * LCS ライブラリで取得できる進捗は、ホスト・クライアントに依らず、
    * 自分自身の送信あるいは受信の状況だけであり、
    * 他の端末の進捗状況の詳細を取得することはできません。
    * 取得できる進捗状況の詳細については Progress のリファレンスを参照してください。
    */
    Result GetProgress(Progress* pOutProgress) NN_NOEXCEPT;

    /**
    * @brief       端末の進捗を取得します。
    *
    * @param[out]  pOutNodeProgress        端末の進捗の出力先です。
    * @param[in]   index                   進捗を取得する端末のインデックスです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultNodeNotFound}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a pOutProgress != nullptr
    * - @a client に GetNodes() で取得した NodeInfo の index を指定していること
    *
    * @details
    * 端末の進捗を取得します。
    *
    * StartContentsShare() でコンテンツの配信を開始した後、 @ref State_Transferring,
    * @ref State_Suspended, @ref State_Completed 状態のいずれかで使用できます。
    * それ以外の状態でこの関数を実行すると ResultInvalidState を返します。
    *
    * インデックスで指定したセッションに参加している端末の進捗状況を取得することができます。
    * 取得できる進捗状況の詳細については NodeProgress のリファレンスを参照してください。
    */
    Result GetNodeProgress(NodeProgress* pOutNodeProgress, uint32_t index) NN_NOEXCEPT;

    /**
    * @brief       ダウンロードが完了し利用可能になったコンテンツのリストを取得します。
    *
    * @param[out]  outContents             コンテンツリストの出力先です。
    * @param[out]  pOutCount               outContents に出力された ContentsInfo の数です。
    * @param[in]   bufferCount             outContents のバッファサイズを要素数で指定します。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a outContents != nullptr
    * - @a pOutCount != nullptr
    * - 0 < @a bufferCount
    *
    * @details
    * ダウンロードが完了し利用可能になったコンテンツのリストを取得します。
    *
    * StartContentsShare() でコンテンツの配信を開始した後、 @ref State_Transferring,
    * @ref State_ContentsShareFailed, @ref State_Suspended, @ref State_Completed の
    * いずれかの状態で使用できます。
    * それ以外の状態でこの関数を実行すると ResultInvalidState を返します。
    *
    * @a pOutCount で取得できる出力数は最大で @ref DownloadableContentsCountMax です。
    */
    Result GetDownloadedContents(
        ContentsInfo* outContents, int* pOutCount, int bufferCount) NN_NOEXCEPT;

    /**
    * @brief       EULA のサイズを取得します。
    *
    * @param[out]  pOutSize                EULA の出力サイズです。
    * @param[out]  eulaDataPath            サイズを取得する EULA のパスです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultNotExsistEula}
    *  @handleresult{ResultBadDataPath}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a pOutSize != nullptr
    * - @a eulaDataPath != nullptr
    *
    * @details
    * 受信した本体更新データに含まれる EULA のサイズを取得します。
    *
    * この関数は @ref State_Suspended 状態でのみ使用できます。
    * それ以外の状態で使用すると ResultInvalidState を返します。
    */
    Result GetDownloadedEulaDataSize(size_t* pOutSize, const char* eulaDataPath) NN_NOEXCEPT;

    /**
    * @brief       EULA を取得します。
    *
    * @param[out]  pOutSize                EULA の出力サイズです。
    * @param[out]  outBuffer               EULA の出力先です。
    * @param[in]   bufferSize              @a outBuffer のバッファサイズです。
    * @param[in]   eulaDataPath            取得する EULA のパスです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultBufferNotEnough}
    *  @handleresult{ResultNotExsistEula}
    *  @handleresult{ResultBadDataPath}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a pOutSize != nullptr
    * - @a buffer != nullptr
    * - @a eulaDataPath != nullptr
    *
    * @details
    * 受信した本体更新データに含まれる EULA を取得します。
    *
    * この関数は @ref State_Suspended 状態でのみ使用できます。
    * それ以外の状態で使用すると ResultInvalidState を返します。
    */
    Result GetDownloadedEulaData(
        size_t* pOutSize, void* outBuffer, size_t bufferSize, const char* eulaDataPath) NN_NOEXCEPT;

    /**
    * @brief       セッションに再合流するために必要なコンテキストを取得します。
    *
    * @param[out]  pOutSessionContext      再合流に必要なコンテキストの出力先です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a pOutSessionContext != nullptr
    *
    * @details
    * セッションに再合流するために必要なコンテキストを取得します。
    *
    * この関数は @ref State_Suspended 状態でのみ使用できます。
    * それ以外の状態で実行すると ResultInvalidState を返します。
    * 再合流の詳細については ResumeSession() を参照してください。
    */
    Result GetSessionContext(SessionContext* pOutSessionContext) NN_NOEXCEPT;

    /**
    * @brief       中断されていたコンテンツの配信を再開します。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @post
    * - GetState() == @ref State_Transferring
    *
    * @details
    * 中断されていたコンテンツの配信を再開します。
    *
    * なんらかの理由で一時的に配信が中断されると @ref State_Suspended 状態に遷移します。
    * @ref State_Suspended 状態に遷移した場合は GetSuspendedReason() を実行して理由を確認し、
    * 問題を解決してからこの関数を実行することでコンテンツ配信を再開できます。
    *
    * @ref State_Suspended 状態以外でこの関数を実行すると ResultInvalidState を返します。
    */
    Result ResumeContentsShare() NN_NOEXCEPT;

    /**
    * @brief       セッションに再合流します。
    *
    * @param[in]   sessionContext          合流先のセッションのコンテキストです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultDeviceOccupied}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultNoSharableContentsSession}
    *  @handleresult{ResultCommunicationError}
    *  @handleresult{ResultSessionNotFound}
    *  @handleresult{ResultConnectionFailed}
    *  @handleresult{ResultNodeCountLimitation}
    *  @handleresult{ResultInvalidContext}
    *  @handleresult{ResultApplyDeltaTaskExists}
    *  @handleresult{ResultFailureNotExistDownloadContents}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a sessionContext が GetSessionContext() で取得したセッションコンテキストであること
    *
    * @post
    * - GetState() == @ref State_Transferring || GetState() == @ref State_Suspended
    *
    * @details
    * セッションに再合流します。
    *
    * 通常、 @ref State_Suspended 状態に遷移した場合は GetSuspendedReason() で理由を確認し、
    * 問題を解決して ResumeContentsShare() を実行すると @ref State_Transferring 状態に戻ります。
    * ただし、 SuspendedReason によっては一旦 LCS ライブラリの使用を終了する必要があります。
    * その場合、 GetSessionContext() で取得した SessionContext を記憶しておいて、
    * 次回 LCS ライブラリを初期化した後で ResumeSession() を実行してください。
    *
    * この関数が成功すると、セッションで配信されるアプリケーションに対して、
    * 適用待ちとなっているタスクが存在する場合、適用待ちタスクは破棄されます。
    * Open 時、ホストが SessionSettings に設定していないアプリケーションの
    * 適用待ちタスクは破棄されません。
    * 破棄されたタスクは LeaveSession() を使用した際に再登録され、
    * インターネットに接続されると再度ダウンロードが行われます。
    * 但し、破棄されたタスクで得られるパッチと LCS ライブラリで取得できたパッチが
    * 同一バージョンの場合、タスクの再登録はされません。
    *
    * セッションから離脱する前にホストだった場合には再合流後にも必ずホストになり、
    * セッションから離脱する前にクライアントだった場合には再合流後にも必ずクライアントになります。
    *
    * この関数はセッションへの合流が完了するまで最大数秒間程度処理を返しません。
    */
    Result ResumeSession(const SessionContext& sessionContext) NN_NOEXCEPT;

    /**
    * @brief       セッションから離脱します。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @post
    * - GetState() == @ref State_Initialized
    *
    * @details
    * セッションから離脱します。
    *
    * この関数はホスト、クライアントの双方で使用することができます。
    * @ref State_Opened, @ref State_Joined, @ref State_Transferring,
    * @ref State_ContentsShareFailed, @ref State_Suspended, @ref State_Completed
    * 以外の状態で使用すると ResultInvalidState を返します。
    *
    * ホストがこの機能を使用した場合、セッションが破棄されるため、
    * ホストに接続されているクライアントも全て切断されます。
    *
    * 一方で、クライアントがこの機能を使用した場合、自身はセッションから離脱しますが、
    * ホストや他のクライアントはセッションを継続できる場合があります。
    * 具体的には、配信が開始される以前、つまり、 @ref State_Joined 状態であれば
    * セッションは確実に維持されます。また、配信が開始された後、
    * @ref State_Transferring, @ref State_Suspended 状態であっても、LCS ライブラリによって
    * 配信の継続が可能と判断された場合にはセッションが破棄されません。
    * ある端末が LeaveSession() を実行したことによってセッションが破棄された場合、他の端末は
    * GetContentsShareFailureReason() でセッションが終了した理由を取得することができます。
    *
    * この関数はセッションからの離脱が完了するまで最大数秒間程度処理を返しません。
    */
    Result LeaveSession() NN_NOEXCEPT;

    /**
    * @brief       セッションから一時離脱します。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @post
    * - GetState() == @ref State_Initialized
    *
    * @details
    * セッションから一時離脱します。
    *
    * この関数は @ref State_Suspended 状態でのみ使用できます。
    *
    * この機能を使用すると、一時的にセッションから離脱が行えます。
    * SuspendedReason_RebootRequired や、SuspendedReason_StorageSpaceNotEnough の中断が発生したときに使用できます。
    * それ以外の理由での中断が発生しているときに使用すると、LeaveSession() と同じ動作となります。
    *
    * この関数はセッションからの離脱が完了するまで最大数秒間程度処理を返しません。
    */
    Result SuspendSession() NN_NOEXCEPT;

    /**
    * @brief       コンテンツの配信に失敗した理由を取得します。
    *
    * @return      コンテンツの配信に失敗した理由です。
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @details
    * コンテンツの配信に失敗した理由を取得します。
    *
    * この関数は @ref State_ContentsShareFailed 状態に遷移したときに使用してください。
    * それ以外の状態では ContentsShareFailureReason_None を返します。
    *
    * このとき、コンテンツの配信に失敗した理由の詳細を取得するためにこの関数を使用します。
    * 理由の詳細については ContentsShareFailureReason のリファレンスを参照してください。
    * LeaveSession() でセッションから離脱すると前回の理由を取得できなくなり、
    * 次に配信に失敗するまで ContentsShareFailureReason_None を返すようになります。
    */
    ContentsShareFailureReason GetContentsShareFailureReason() NN_NOEXCEPT;

    /**
    * @brief       コンテンツの配信が中断された理由を取得します。
    *
    * @return      コンテンツの配信が中断された理由です。
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @details
    * コンテンツの配信が中断された理由を取得します。
    *
    * この関数は @ref State_Suspended 状態に遷移したときに使用してください。
    * それ以外の状態では SuspendedReason_None を返します。
    *
    * コンテンツの配信が中断された問題を解決した後は
    * ResumeContentsShare() や ResumeSession() で配信を再開できます。
    * 理由と解決方法の詳細については SuspendedReason のリファレンスを参照してください。
    */
    SuspendedReason GetSuspendedReason() NN_NOEXCEPT;

    /**
    * @brief       コンテンツの配信が失敗した理由を変換します。
    *
    * @return      コンテンツの配信が失敗したときのエラーコードです。
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @details
    * コンテンツの配信が失敗した理由をエラーコードに変換します。
    *
    * GetContentsShareFailureReason() で取得した失敗理由をエラーコードに変換できます。
    * @a ContentsShareFailureReason_None を与えた場合は、 ResultSuccess を返します。
    * 取得したエラーコードは ShowError() に使用して画面にエラーメッセージを表示できます。
    */
    Result ConvertFailureReasonToResult(ContentsShareFailureReason reason) NN_NOEXCEPT;

    /**
    * @brief       画面にエラーメッセージを表示します。
    *
    * @pre
    * - result.GetModule() == 211
    *
    * @details
    * 画面に LCS ライブラリのエラーメッセージを表示します。
    *
    * エラーコードに応じたエラーメッセージを画面に表示することができます。
    * LCS ライブラリで取得できるエラーのみを与えるようにしてください。
    */
    void ShowError(Result result) NN_NOEXCEPT;

    /**
    * @brief       適用待ちタスクが削除されたかを取得します。
    *
    * @param[out]  pIsDiscarded                破棄されたかの結果の出力先です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a isDiscard != nullptr
    *
    * @details
    * 適用待ちタスクが削除されたかを取得します。
    *
    * この関数はホスト、クライアントの双方で使用することができます。
    * @ref State_Opened, @ref State_Joined, @ref State_Transferring,
    * @ref State_ContentsShareFailed, @ref State_Suspended, @ref State_Completed
    * 以外の状態で使用すると ResultInvalidState を返します。
    *
    * 適用待ちタスクが破棄された場合は、@a isDiscard に true が出力されます。
    */
    Result IsWaitingPatchInstallTaskDiscard(bool* pIsDiscarded) NN_NOEXCEPT;


    //! @}
    //! @name ホスト用の API
    //! @{

    /**
    * @brief       ホストとしてセッションを開始します。
    *
    * @param[in]   settings                セッションの設定です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultApplicationNotFound}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    *  @handleresult{ResultNodeCountLimitation}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - 0 < @a settings.applicationCount &&
    *   @a settings.applicationCount <= @ref SharableContentsCountMax
    * - 0 < @a settings.nodeCountMax && @a settings.nodeCountMax <= @ref NodeCountMax
    * - @a settings.contentsShareVersionPolicy == @ref ContentsShareVersionPolicy_Latest ||
    *   @a settings.contentsShareVersionPolicy == @ref ContentsShareVersionPolicy_Host
    *
    * @post
    * - GetState() == @ref State_Opened
    *
    * @details
    * ホストとしてセッションを開始します。
    *
    * この関数は @ref State_Initialized 状態で使用してください。
    * それ以外の状態で使用すると ResultInvalidState を返します。
    *
    * セッションで配信するアプリケーションは @a settings で指定します。
    * 詳細については SessionSettings のリファレンスを参照してください。
    * 指定されたアプリケーションを配信できない場合には
    * ResultApplicationNotFound を返します。
    * 無線コントローラの接続数が多すぎてセッションを開始できない場合には
    * ResultNodeCountLimitation を返します。
    *
    * この関数が成功すると、@a settings で設定されているアプリケーションに対して、
    * 適用待ちとなっているタスクが存在する場合、適用待ちタスクは破棄されます。
    * Open 時、ホストが SessionSettings に設定していないアプリケーションの
    * 適用待ちタスクは破棄されません。
    * 破棄されたタスクは LeaveSession() を使用した際に再登録され、
    * インターネットに接続されると再度ダウンロードが行われます。
    * 但し、破棄されたタスクで得られるパッチと LCS ライブラリで取得できたパッチが
    * 同一バージョンの場合、タスクの再登録はされません。
    *
    * この関数でセッションを構築すると、クライアントから Scan() で発見し、
    * JoinSession() で接続できる状態になります。クライアントの新規参加や離脱があると
    * AttachStateChangeEvent() で取得したイベントがシグナルされますので、
    * 接続中のクライアントの情報を GetNodes() で取得してください。
    *
    * 最大クライアント数に達しない限りはクライアントの参加を受け入れる状態になっています。
    * クライアントの募集を締め切りたい場合には SetAcceptClientPolicy() に
    * @ref AcceptPolicy_AlwaysReject を指定して実行してください。
    *
    * SetAcceptClientPolicy() を使用しなくとも StartContentsShare()
    * でコンテンツの配信を開始するとクライアントの募集は自動的に締め切られます。
    * ただし、配信開始直前にクライアントが接続してきた場合、
    * 想定外のクライアントがコンテンツ配信に参加することになってしまいます。
    * これを防ぐには SetAcceptClientPolicy() でクライアントの募集を締め切ってから
    * GetNodes() でクライアントの参加状況を確認し、
    * 問題がなければ StartContentsShare() で配信を開始するようにしてください。
    *
    * この関数はセッションの構築が完了するまで最大 1 秒間程度処理を返しません。
    */
    Result OpenSession(const SessionSettings& settings) NN_NOEXCEPT;

    /**
    * @brief       クライアントの参加を許可・拒否する設定を行います。
    *
    * @param[in]   policy                  設定するポリシーです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a policy == @ref AcceptPolicy_AlwaysAccept ||
    *   @a policy == @ref AcceptPolicy_AlwaysReject
    *
    * @details
    * クライアントの参加を許可・拒否する設定を行います。
    *
    * この関数はホストが OpenSession() でセッションを構築した後、
    * StartContentsShare() で配信を開始する前に使用してください。 @ref State_Opened
    * 以外の状態でこの関数を使用した場合は ResultInvalidState を返して失敗します。
    *
    * OpenSession() でセッションを構築した直後は @ref AcceptPolicy_AlwaysAccept が設定されるため、
    * 最大クライアント数に達しない限りはクライアントの参加を受け入れます。
    * クライアントの募集を締め切りたい場合にはこの関数に
    * @ref AcceptPolicy_AlwaysReject を指定して実行してください。
    *
    * この関数を使用しなくとも StartContentsShare()
    * でコンテンツの配信を開始するとクライアントの募集は自動的に締め切られます。
    */
    Result SetClientAcceptPolicy(AcceptPolicy policy) NN_NOEXCEPT;

    /**
    * @brief       コンテンツ配信を開始します。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    *
    * @post
    * - GetState() == @ref State_Transferring ||
    *   GetState() == @ref State_Suspended
    *
    * @details
    * コンテンツ配信を開始します。
    *
    * この関数はホストが OpenSession() でセッションを構築した後、
    * コンテンツ配信に参加するクライアントが揃った後で使用してください。
    * コンテンツの配信を開始すると新しくクライアントがセッションに参加することはできなくなります。
    * @ref State_Opened 以外の状態でこの関数を使用した場合は
    * ResultInvalidState を返して失敗します。
    *
    * この関数に成功した後、すぐにコンテンツの配信を開始できる場合には
    * @ref State_Transferring 状態に、解決しなければならない問題がある場合には
    * @ref State_Suspended 状態に遷移します。
    * @ref State_Suspended 状態は受信に必要なストレージの空きが足りない場合や、
    * 再起動が必要などが発生した場合に遷移します。
    * @ref State_Suspended 状態に遷移したときは GetSuspendedReason()
    * で配信を開始できない理由を確認し、問題を解決してから ResumeContentsShare() や
    * ResumeSession() で配信を再開してください。
    * @ref State_Transferring 状態に遷移した場合、ホスト、クライアントに関係なく、
    * 送信者・受信者の役割が与えられ、配信は自動的に進行します。
    * コンテンツ配信の進捗や現在の役割は GetProgress() で取得できます。
    * 途中で再度 @ref State_Suspended 状態に遷移することもありますので注意してください。
    * 配信が完了すると @ref State_Completed 状態に遷移します。
    * 配信中や、配信完了後は GetDownloadedContents() で受信したコンテンツの情報を取得できます。
    * @ref State_Completed 状態からは LeaveSession() や Finalize() で終了してください。
    *
    * この関数は配信を開始するだけなので即座に処理を返します。
    */
    Result StartContentsShare() NN_NOEXCEPT;

    //! @}
    //! @name クライアント用の API
    //! @{

    /**
    * @brief       周囲のセッションを探索します。
    *
    * @param[out]  outSessions             スキャン結果の出力先です。
    * @param[out]  pOutCount               @a outSessions に出力されたスキャン結果の数です。
    * @param[in]   bufferCount             @a outSessions に格納できるスキャン結果の数です。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultDeviceOccupied}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a outSessions != nullptr
    * - @a pOutCount != nullptr
    * - 0 < @a bufferCount
    *
    * @details
    * 周囲のセッションを探索します。
    *
    * この関数は Initialize() で LCS ライブラリを初期化した後、
    * OpenSession() や JoinSession() でセッションに参加する前に実行してください。
    * 1 回のスキャンで発見できるセッションの数は ScanResultCountMax です。
    * @ref State_Initialized 以外の状態でこの関数を使用すると ResultInvalidState を返します。
    *
    * @a pOutCount で取得できる出力数は最大で @ref ScanResultCountMax です。
    *
    * 実際には周囲でセッションが構築されている場合であっても、
    * 周囲の通信環境によっては 1 回の探索では発見できないことがあります。
    * この関数を複数回実行し、結果を統合するなどの対策を検討してください。
    *
    * Scan() で得られる SessionInfo の情報は Scan() を行った時点での情報です。
    * @a nodeCount や、 @a contents に含まれる @a version や、 @a displayVersion は
    * セッションへのクライアントの参加・離脱に応じて随時変化しますので、
    * 定期的に Scan() を行い、更新することが望ましいです。
    *
    * なお、既に StartContentsShare() でコンテンツ配信を開始したセッションは
    * Scan() で発見することができません。
    *
    * この関数はセッションの探索が完了するまで最大 1 秒間程度処理を返しません。
    */
    Result Scan(SessionInfo* outSessions, int* pOutCount, int bufferCount) NN_NOEXCEPT;

    /**
    * @brief       クライアントとしてセッションに参加します。
    *
    * @param[in]    session                 対象のセッションです。
    *
    * @retresult
    *  @handleresult{ResultSuccess}
    *  @handleresult{ResultDeviceOccupied}
    *  @handleresult{ResultWifiOff}
    *  @handleresult{ResultSleep}
    *  @handleresult{ResultInvalidState}
    *  @handleresult{ResultNoSharableContentsSession}
    *  @handleresult{ResultLowerVersion}
    *  @handleresult{ResultHigherVersion}
    *  @handleresult{ResultIncompatibleSystemVersion}
    *  @handleresult{ResultCommunicationError}
    *  @handleresult{ResultSessionNotFound}
    *  @handleresult{ResultConnectionFailed}
    *  @handleresult{ResultNodeCountLimitation}
    *  @handleresult{ResultApplyDeltaTaskExists}
    * @endretresult
    *
    * @pre
    * - GetState() != @ref State_None
    * - @a session が Scan() で取得した SessionInfo であること
    *
    * @post
    * - GetState() == @ref State_Joined
    *
    * @details
    * クライアントとしてセッションに参加します。
    *
    * Scan() で取得した SessionInfo を @a session に指定してください。
    * セッションへの参加に成功するとこの関数が成功し、 @ref State_Joined 状態に遷移します。
    *
    * @ref State_Initialized 以外の状態でこの関数を使用すると ResultInvalidState を、
    * 指定されたセッションが見つからなかった場合や接続に失敗した場合には ResultCommunicationError
    * を、無線コントローラの接続数が多すぎてセッションを開始できない場合には
    * ResultNodeCountLimitation を返して失敗します。
    *
    * この関数が成功すると、@a session で設定されているアプリケーションに対して、
    * 適用待ちとなっているタスクが存在する場合、適用待ちタスクは破棄されます。
    * Open 時、ホストが SessionSettings に設定していないアプリケーションの
    * 適用待ちタスクは破棄されません。
    * 破棄されたタスクは LeaveSession() を使用した際に再登録され、
    * インターネットに接続されると再度ダウンロードが行われます。
    * 但し、破棄されたタスクで得られるパッチと LCS ライブラリで取得できたパッチが
    * 同一バージョンの場合、タスクの再登録はされません。

    * セッションへの参加に成功した後、ホストが StartContentsShare() を実行すると、クライアントは
    * @ref State_Transferring 状態あるいは @ref State_Suspended 状態に自動的に遷移します。
    * @ref State_Suspended 状態は受信に必要なストレージの空きが足りない場合や、
    * 再起動が必要などが発生した場合に遷移します。
    * @ref State_Suspended 状態に遷移したときは GetSuspendedReason()
    * で配信を開始できない理由を確認し、問題を解決してから ResumeContentsShare() や
    * ResumeSession() で配信を再開してください。
    * @ref State_Transferring 状態に遷移した場合、ホスト、クライアントに関係なく、
    * 送信者・受信者の役割が与えられ、配信は自動的に進行します。
    * コンテンツ配信の進捗や現在の役割は GetProgress() で取得できます。
    * 途中で再度 @ref State_Suspended 状態に遷移することもありますので注意してください。
    * 配信が完了すると @ref State_Completed 状態に遷移します。
    * 配信中や、配信完了後は GetDownloadedContents() で受信したコンテンツの情報を取得できます。
    * @ref State_Completed 状態からは LeaveSession() や Finalize() で終了してください。
    *
    * この関数はセッションへの参加が完了するまで最大数秒間程度処理を返しません。
    */
    Result JoinSession(const SessionInfo& session) NN_NOEXCEPT;

    //! @}

}} // namespace nn::lcs
