﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/kpr/kpr_KeyCode.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_BitPack.h>

namespace nn { namespace kpr { namespace detail {

//!< 文字コード変換表のロックキーを表す列挙体です。
enum class KeyCodeMapLockKey
{
    None,       //!< 無し
    CapsLock,   //!< Caps Lock
    NumLock,    //!< Num Lock
};

//!< 文字コード変換表のロック情報のフィールド定義です。
struct KeyCodeMapLockInfoField final
{
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<0,  1, bool>
            Basic;      //!< 基本マップがロック可能か否か
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<1,  1, bool>
            AltGr;      //!< Alt Gr マップがロック可能か否か
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<2,  1, bool>
            Hiragana;   //!< かな入力（ひらがな）マップがロック可能か否か
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<3,  1, bool>
            Katakana;   //!< かな入力（カタカナ）マップがロック可能か否か
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<4,  1, bool>
            Cyrillic;   //!< キリル文字入力マップがロック可能か否か
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<5,  1, bool>
            Hangul;     //!< ハングル入力マップがロック可能か否か
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<6,  1, bool>
            Zhuyin;     //!< 注音入力マップがロック可能か否か
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<7,  1, bool>
            Cangjie;    //!< 倉頡入力マップがロック可能か否か
    typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
                ::Field<12, 4, int>
            LockKey;    //!< ロックキー
};

//!< 文字コード変換表のロック情報を表す型です。
typedef ::nn::util::BitPack<uint16_t, KeyCodeMapLockInfoField>
        KeyCodeMapLockInfo;

const uint16_t BasicSystem = 0x0000;    //!< 制御キー用ロック情報内部値
const uint16_t BasicSymbol = 0x1000;    //!< 記号キー用ロック情報内部値
const uint16_t BasicLetter = 0x1001;    //!< 文字キー用ロック情報内部値
const uint16_t AltGrSymbol = 0x1002;    //!< 記号キー用ロック情報内部値 (Alt Gr)
const uint16_t AltGrLetter = 0x1003;    //!< 文字キー用ロック情報内部値 (Alt Gr)
const uint16_t CyrilSymbol = 0x1010;    //!< 記号キー用ロック情報内部値 (キリル)
const uint16_t CyrilLetter = 0x1011;    //!< 文字キー用ロック情報内部値 (キリル)
const uint16_t BasicKeypad = 0x20FF;    //!< テンキー用ロック情報内部値

//!< 文字コード変換表の属性定義です。
struct KeyCodeMapAttribute final
{
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<0>
            Default;    //!< 既定の変換モードに対応しているか否か
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<1>
            Hiragana;   //!< かな入力（ひらがな）に対応しているか否か
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<2>
            Katakana;   //!< かな入力（カタカナ）に対応しているか否か
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<3>
            Cyrillic;   //!< キリル文字入力に対応しているか否か
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<4>
            Hangul;     //!< ハングル入力に対応しているか否か
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<5>
            Zhuyin;     //!< 注音入力に対応しているか否か
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<6>
            Cangjie;    //!< 倉頡入力に対応しているか否か
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<24>
            AltGr;      //!< Alt Gr に対応しているか否か
    typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute>::Flag<25>
            AltGrShift; //!< Alt Gr + Shift に対応しているか否か
};

//!< 文字コード変換表の属性集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, KeyCodeMapAttribute> KeyCodeMapAttributeSet;

//!< 文字コード変換表の内部表現を表す構造体です。
struct KeyCodeMapStorage final
{
    KeyCodeMapAttributeSet attributes;  //!< 文字コード変換表の属性
    uint32_t mapCount;                  //!< マップの数
    uint32_t keyCount;                  //!< キーの数
    uint32_t altGrMapOffset;            //!< Alt Gr マップのオフセット
    uint64_t _reserved[6];
    uint16_t data[2016];                //!< シリアライズされた文字コード変換表
};

}}} // namespace nn::kpr::detail
