﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Struct and constant declaration for AdaptiveClusteringProcessor
 */

#pragma once

#include <nn/util/util_MathTypes.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>

namespace nn { namespace irsensor {

const int AdaptiveClusteringProcessorObjectCountMax = 48;     //!< Maximum object count.
const int AdaptiveClusteringProcessorStateCountMax = 5;       //!< Maximum number of buffered AdaptiveClusteringProcessorState.

/**
 * @brief Mode of AdaptiveClusteringProcessor.
 */
enum AdaptiveClusteringMode
{
    AdaptiveClusteringMode_StaticFov,     //!< Only camera setting is adaptive.
    AdaptiveClusteringMode_DynamicFov,    //!< Addition to camera settings, change fov adaptively with object tracking.
};

/**
 * @brief Auto camera setting depends on target distance of AdaptiveClusteringProcessor.
 */
enum AdaptiveClusteringTargetDistance
{
    AdaptiveClusteringTargetDistance_Near,     //!< Auto camera setting for near range objects (approx, ~30cm). Robustness for noise is high.
    AdaptiveClusteringTargetDistance_Middle,   //!< Auto camera setting for middle range objects (approx,~1m). Robustness for noise is middle.
    AdaptiveClusteringTargetDistance_Far,      //!< Auto camera setting for far range objects (approx, 1m ~). Robustness for noise is low.
};

/**
 * @brief The accuracy level of the output of AdaptiveClusteringProcessor
 */
enum AdaptiveClusteringAccuracyLevel
{
    AdaptiveClusteringAccuracyLevel_Low,    //!< Low quality frame to checking if there is no new object. In a High quality frame flow, this frame can be ignored.
    AdaptiveClusteringAccuracyLevel_Middle, //!< Middle quality frame because the object is captured with a wide FOV.
    AdaptiveClusteringAccuracyLevel_High,   //!< High quality frame because the object is captured with a narrow FOV.
};

/**
 * @brief Struct for setting of AdaptiveClusteringProcessor.
 */
struct AdaptiveClusteringProcessorConfig
{
    AdaptiveClusteringMode mode;               //!< Adaptive algorithm mode.
    AdaptiveClusteringTargetDistance distance; //!< Target distance setting.
};

/**
 * @brief Struct for output data of each object.
 */
struct AdaptiveClusteringData
{
    ::nn::util::Float2 centroid; //!< Subpixel position of the object in a 320x240. (Value for x can vary from 0 to 320. Value for y can vary from 0 to 240.)
    float area;                  //!< Area as the pixel count of the objects. (Vary between 0.f and 320.f * 240.f = 76800.f)
    bool isIncomplete;           //!< True if the object surface doesn't have complete information like touching border.
};

/**
 * @brief Struct for output data from AdaptiveClusteringProcessor.
 */
struct AdaptiveClusteringProcessorState
{
    int64_t samplingNumber;                                                    //!< Sampling number which increments by getting data from processor.
    ::nn::TimeSpanType timeStamp;                                              //!< Timestamp from starting the processor.
    IrCameraAmbientNoiseLevel ambientNoiseLevel;                               //!< Noise level from surrounding environment.
    AdaptiveClusteringAccuracyLevel accuracyLevel;                             //!< Accuracy level of data.
    int8_t objectCount;                                                        //!< The number of detected objects.
    uint8_t backgroundIntensity;                                               //!< Average pixel intensity of background area except for objects. (Value can vary from 0 to 255.)
    char reserved[6];
    AdaptiveClusteringData objects[AdaptiveClusteringProcessorObjectCountMax]; //!< Detected objects data.
};

}} // namespace nn::irsensor
