﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   コントローラーサポートの UI 呼び出しの起動パラメータの宣言（非公開）
 */

#pragma once

#include <nn/hid/hid_ControllerSupportArg.h>
#include <nn/hid/hid_NpadCommon.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/system/hid_ControllerSupportTypes.h>

namespace nn { namespace hid { namespace system {

// 暗黙のパディングは警告を出す
#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic push
#pragma clang diagnostic warning "-Wpadded"
#endif

/**
* @brief コントローラーサポートのモードを示す列挙子です。
*/
enum ControllerSupportMode : uint8_t
{
    ControllerSupportMode_Default = 0,          //!< デフォルト
    ControllerSupportMode_StrapGuide,           //!< ストラップ着用案内
    ControllerSupportMode_FirmwareUpdate,       //!< ファームウェア更新
    ControllerSupportMode_StrapGuideForPalma,   //!< Palma用のストラップ着用案内
};

/**
 * @brief コントローラーサポートの UI 呼び出しの起動パラメータです。
 */
struct ControllerSupportArgPrivate
{
    //! @brief 構造体全体のサイズです。
    uint32_t argPrivateSize;

    //! @brief アプリが設定する起動パラメータ構造体のサイズです。
    uint32_t argSize;

    //! @brief 最初に接続情報を表示するかを指定します。
    bool showConnectionStatus;

    //! @brief 本体機能からの呼び出しかを保持します。
    bool isSystem;

    //! @brief モードを指定します。
    ControllerSupportMode mode;

    //! @brief 呼び出し元を指定します。
    ControllerSupportCaller caller;

    //! @brief 設定を引き継ぐために、呼び出し元の操作形態を保持します。
    nn::hid::NpadStyleSet styleSet;

    //! @brief 設定を引き継ぐために、呼び出し元の持ち方設定を保持します。
    nn::hid::NpadJoyHoldType holdType;

    //! @brief デフォルト値を設定します。
    void SetDefault() NN_NOEXCEPT
    {
        this->argPrivateSize = sizeof( ControllerSupportArgPrivate );
        this->argSize = sizeof( ControllerSupportArg );
        this->showConnectionStatus = false;
        this->isSystem = false;
        this->mode = ControllerSupportMode_Default;
        this->caller = ControllerSupportCaller_Application;
        this->styleSet = nn::hid::GetSupportedNpadStyleSet();
        this->holdType = nn::hid::GetNpadJoyHoldType();
    }
};

/**
 * @brief コントローラーサポートの結果を示す列挙子です。
 */
enum ControllerSupportResult : uint32_t
{
    ControllerSupportResult_Success,                //!< 成功
    ControllerSupportResult_Cancel,                 //!< キャンセル
    ControllerSupportResult_NotSupportedNpadStyle,  //!< サポート外の操作形態

    // FW 更新用
    ControllerSupportResult_FirmwareUpdateFailed,   //!< 更新に失敗
};

/**
 * @brief コントローラーサポート終了時のパラメータです。
 *
 * @details
 *  値はコントローラーサポート内部で設定されます。
 */
struct ControllerSupportReturnArgPrivate
{
    //! @brief 呼び出し元に渡す値を格納します。
    ControllerSupportResultInfo controllerSupportResultInfo;

    //! @brief 結果を表す内部向けのパラメータです。
    ControllerSupportResult result;
};

#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic pop
#endif


}}} // namespace nn::hid::system

