﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Xpad に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid {

/**
 * @brief       Xpad ID を表す構造体です。
 *
 * @details     基本的な構成を持つ Xpad と見做して参照します。
 */
struct BasicXpadId
{
    int32_t _storage;
};

const int XpadIdCountMax = 16;  //!< GetXpadIds() が一度に返す Xpad ID の最大数

const int XpadStateCountMax = 16;   //!< 内部的に保持される Xpad の入力状態の最大数

/**
 * @brief       基本的な構成を持つ Xpad のデジタルボタン定義です。
 */
struct BasicXpadButton
{
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<0> A;          //!< 基本的な構成を持つ Xpad の a ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<1> B;          //!< 基本的な構成を持つ Xpad の b ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<2> X;          //!< 基本的な構成を持つ Xpad の x ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<3> Y;          //!< 基本的な構成を持つ Xpad の y ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<4> StickL;     //!< 基本的な構成を持つ Xpad の L スティックボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<5> StickR;     //!< 基本的な構成を持つ Xpad の R スティックボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<6> L;          //!< 基本的な構成を持つ Xpad の L ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<7> R;          //!< 基本的な構成を持つ Xpad の R ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<8> ZL;         //!< 基本的な構成を持つ Xpad の ZL ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<9> ZR;         //!< 基本的な構成を持つ Xpad の ZR ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<10> Start;     //!< 基本的な構成を持つ Xpad の Start ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<11> Select;    //!< 基本的な構成を持つ Xpad の Select ボタン
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<12> Left;      //!< 基本的な構成を持つ Xpad の十字ボタン 左
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<13> Up;        //!< 基本的な構成を持つ Xpad の十字ボタン 上
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<14> Right;     //!< 基本的な構成を持つ Xpad の十字ボタン 右
    typedef ::nn::util::BitFlagSet<32, BasicXpadButton>::Flag<15> Down;      //!< 基本的な構成を持つ Xpad の十字ボタン 下
};

/**
 * @brief       基本的な構成を持つ Xpad のデジタルボタンの集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, BasicXpadButton> BasicXpadButtonSet;

/**
 * @brief       基本的な構成を持つ Xpad の入力状態の属性定義です。
 */
struct BasicXpadAttribute
{
    typedef ::nn::util::BitFlagSet<32, BasicXpadAttribute>::Flag<0>
            IsConnected;    //!< 対応するデバイスがシステムと接続状態にあるか否か
};

/**
 * @brief       基本的な構成を持つ Xpad の入力状態の属性集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, BasicXpadAttribute> BasicXpadAttributeSet;

/**
 * @brief       基本的な構成を持つ Xpad の入力状態を表す構造体です。
 */
struct BasicXpadState
{
    int64_t samplingNumber;             //!< 基本的な構成を持つ Xpad の入力状態が更新される度に増加する値です。
    BasicXpadAttributeSet attributes;   //!< 基本的な構成を持つ Xpad の入力状態の属性です。
    BasicXpadButtonSet buttons;         //!< 基本的な構成を持つ Xpad のデジタルボタンの状態です。
    AnalogStickState analogStickL;      //!< 基本的な構成を持つ Xpad の左アナログスティックの状態です。
    AnalogStickState analogStickR;      //!< 基本的な構成を持つ Xpad の右アナログスティックの状態です。
};

//! @name Xpad 関連 API
//! @{

/**
 * @brief       Xpad ID の再解決通知をバインドします。 <b>廃止予定です。</b>
 *
 * @details     Xpad ID の再解決通知を指定のイベントオブジェクトにバインドします。
 *              同時に複数のイベントオブジェクトに対してバインドすることはできません。
 *
 * @deprecated  再解決通知をハンドリングする必要はなくなりました。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *              - *pEvent はシグナル状態
 */
NN_DEPRECATED
void BindXpadIdEvent(::nn::os::SystemEventType* pEvent,
                     ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       Xpad ID の再解決通知のバインドを解除します。 <b>廃止予定です。</b>
 *
 * @details     Xpad ID の再解決通知を指定のイベントオブジェクトから解除します。
 *              通知がバインドされていないイベントオブジェクトが指定された場合は何も行いません。
 *
 * @deprecated  再解決通知をハンドリングする必要はなくなりました。
 *
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は初期化状態
 *              - *pEvent を待機しているスレッドはいない
 * @post
 *              - *pEvent は未初期化状態
 */
NN_DEPRECATED
void UnbindXpadIdEvent(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief       Xpad ID を取得します。
 *
 * @details     取得される Xpad ID の数、値、順番は常に一定です。
 *              Xpad の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *              任意のタイミングで呼び出しに成功します。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[out]  outXpadIds                  Xpad ID を読み出すバッファ（配列）
 * @param[in]   count                       読み出す Xpad ID の数
 *
 * @return      読み出した Xpad ID の数です。
 *
 * @pre
 *              - outXpadIds != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 */
NN_DEPRECATED
int GetXpadIds(BasicXpadId* outXpadIds, int count) NN_NOEXCEPT;

/**
 * @brief       Xpad のプレイヤー番号を取得します。
 *
 * @details     Xpad ID で指定した Xpad のプレイヤー番号を取得します。
 *              プレイヤー番号は 0 オリジンです。
 *              Xpad ID の異なる Xpad が同じ値を返すことはありません。
 *              同じ Xpad ID に対しては常に同じ結果となります。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[in]   xpadId                      Xpad ID
 *
 * @return      Xpad のプレイヤー番号です。
 *
 * @post
 *              - 戻り値 n について、n >= 0
 */
NN_DEPRECATED
int GetXpadPlayerNumber(const BasicXpadId& xpadId) NN_NOEXCEPT;

/**
 * @brief       Xpad を初期化します。
 *
 * @details     Xpad ID で指定した Xpad を初期化します。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[in]   xpadId                      Xpad ID
 */
NN_DEPRECATED
void InitializeXpad(const BasicXpadId& xpadId) NN_NOEXCEPT;

/**
 * @brief       Xpad の最新の入力状態を取得します。
 *
 * @details     指定の Xpad ID と対応する Xpad から GetXpadStates() で 1 つの入力状態を読み出した時と同じ値が返ります。
 *              最低 1 回の入力状態の更新が保証されるため、 InitializeXpad() の呼び出し完了直後から利用可能です。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[out]  outValue                    入力状態を読み出すバッファ
 * @param[in]   xpadId                      Xpad ID
 *
 * @pre
 *              - 指定の Xpad ID と対応する Xpad について InitializeXpad() の呼び出しが完了している
 *              - outValue != nullptr
 * @post
 *              - !(outValue->buttons.Get<BasicXpadButton::Left>() && outValue->buttons.Get<BasicXpadButton::Right>())
 *              - !(outValue->buttons.Get<BasicXpadButton::Up>() && outValue->buttons.Get<BasicXpadButton::Down>())
 *              - アナログスティックの入力状態 outValue->analogStickL と outValue->analogStickR は下記の条件を満たす円の内側（境界含む）に位置
 *                  - 原点は (0, 0)
 *                  - 半径は AnalogStickMax
 */
NN_DEPRECATED
void GetXpadState(BasicXpadState* outValue, const BasicXpadId& xpadId
                  ) NN_NOEXCEPT;

/**
 * @brief       Xpad の入力状態を過去に遡って読み出します。
 *
 * @details     最新のものから過去に遡って利用可能な数だけ順に、指定の Xpad ID と対応する Xpad から入力状態を読み出します。
 *              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
 *
 *              読み出し可能な入力状態の最大数は XpadStateCountMax 個です。
 *              利用可能な入力状態には読み出し済みのものも含まれます。
 *              差分だけを利用したい場合は BasicXpadState::samplingNumber を参照してください。
 *
 *              BasicXpadState::attributes メンバの BasicXpadAttribute::IsConnected フラグが false の場合、
 *              入力状態は無入力（デジタルボタンの押下は無く、アナログスティックはニュートラル位置）となります。
 *
 * @deprecated  Xpad は廃止されました。 Npad を利用してください。
 *
 * @param[out]  outStates                   入力状態を読み出すバッファ（配列）
 * @param[in]   count                       読み出す入力状態の数
 * @param[in]   xpadId                      Xpad ID
 *
 * @return      読み出した入力状態の数を返します。
 *
 * @pre
 *              - 指定の Xpad ID と対応する Xpad について InitializeXpad() の呼び出しが完了している
 *              - outStates != nullptr
 *              - count >= 0
 * @post
 *              - 戻り値 n について、n >= 0
 *              - 読み出した入力状態 outStates[i] について
 *                  - !(outStates[i].buttons.Get<BasicXpadButton::Left>() && outStates[i].buttons.Get<BasicXpadButton::Right>())
 *                  - !(outStates[i].buttons.Get<BasicXpadButton::Up>() && outStates[i].buttons.Get<BasicXpadButton::Down>())
 *                  - アナログスティックの入力状態 outStates[i].analogStickL と outStates[i].analogStickR は下記の条件を満たす円の内側（境界含む）に位置
 *                      - 原点は (0, 0)
 *                      - 半径は AnalogStickMax
 */
NN_DEPRECATED
int GetXpadStates(BasicXpadState* outStates,
                  int count,
                  const BasicXpadId& xpadId) NN_NOEXCEPT;

//! @}

}} // namespace nn::hid
