﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Palma の固有の機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_PalmaTypes.h>

namespace nn { namespace hid {

//! @name Palma の固有の機能に関係する API
//! @{

/**
 * @brief       Npad に割り当てられた PalmaConnectionHandle を取得します。
 *
 * @details     NpadIdType で指定された Npad に対応する PalmaConnectionHandle を取得します。
 *              1つの Npad につき Palma のハンドルは最大 1つまで取得することができます。
 *              Npad に対して Palma が接続されていない場合は、ハンドルの取得に失敗します。
 *
 * @param[out]  pOutHandle                 取得されたハンドル
 * @param[in]   id                         ハンドルを取得する対象の NpadIdType
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaNotConnected,       Palma が接続されていません。}
 * @endretresult
 */
Result GetPalmaConnectionHandle(PalmaConnectionHandle* pOutHandle, const NpadIdType& id) NN_NOEXCEPT;

/**
 * @brief       Palma の固有機能を初期化します。
 *
 * @details     PalmaConnectionHandle で操作する機能を用いる場合は必ず本関数を用いて初期化を行ってください。
 *
 * @param[in]   handle                      Palma のコネクションハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 * @endretresult
 */
Result InitializePalma(const PalmaConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       Palma に対しての各操作が完了したことの通知をイベントオブジェクトにバインドします。
 *
 * @details     Application Section の読み書きなど、Palma に対する操作には時間がかかるものがあります。
 *              これらの処理は、本関数で設定した SystemEventType で処理の完了を取得します。
 *              完了結果は、nn::hid::PalmaOperationInfo で取得することができます。どの処理に対する結果かは、type を参照することで判定することができます。
 *              あるデバイスに対して処理を実行している際は、その関数の実行結果として nn::hid::ResultPalmaBusy が返り他の処理を実行することができません。
 *              異なるデバイス間においては、あるデバイスに対しての処理を別のデバイスに対して実行することができます。
 *
 *              Palma が切断されるなどしてハンドルが無効になった場合は、本関数で割り当てた SystemEventType を nn::os::DestroySystemEvent() で破棄してください。
 *
 *
 * @param[in]   handle                      Palma のコネクションハンドル
 * @param[in]   pEvent                      イベントオブジェクトを指すポインタ
 * @param[in]   clearMode                   イベントオブジェクトのクリアモード
 *
 * @pre
 *              - pEvent != nullptr
 *              - *pEvent は未初期化状態
 * @post
 *              - *pEvent は初期化状態
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 * @endretresult
 */
Result BindPalmaOperationCompleteEvent(const PalmaConnectionHandle& handle,
                                      nn::os::SystemEventType* pEvent,
                                      ::nn::os::EventClearMode clearMode) NN_NOEXCEPT;

/**
 * @brief       Palma に対しての各操作に対する実行結果を取得します。
 *
 * @details     Palma に対して非同期に行われる各操作についての実行結果を取得します。
 *
 *              実行結果の取得には Palma のハンドルを指定します。
 *              実行結果がない場合には、ResultPalmaNoOperationInfo が返ります。
 *              実行結果の種類は pOutValue->type を確認します。
 *
 *              操作に対する実行結果は pOutValue->result に設定されます。pOutValue->result.IsSuccess() == true の場合は処理に成功したことを表します。
 *
 *              各操作の内容に準じた付帯情報も取得することが可能です。
 *              具体的には、 PalmaOperationInfo の中身を確認してください。
 *
 *              各操作は本関数が呼ばれるまで Palma をビジー状態とします。
 *              次の操作を行う場合に必ず 本関数を呼び結果を取得してから次の操作を行うようにしてください。
 *
 * @param[out]  pOutValue               各操作に対する実行結果
 * @param[in]   handle                  Palma のコネクションハンドル
 *
 * @pre
 *              - pOutValue != nullptr
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaNoOperationInfo,    実行結果が存在しません。操作が実行されていないか、実行が完了していません。}
 * @endretresult
 */
Result GetPalmaOperationInfo(PalmaOperationInfo* pOutValue,
                            const PalmaConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       Activity を再生開始要求を発行します。
 *
 * @return      指定された Index に対応する Activity を再生開始します。
 *
 *              再生の要求処理は非同期に行われます。
 *              再生開始の要求が Palma に届き再生が開始すると、 BindPalmaOperationCompleteEvent() で割り当てたシステムイベントがシグナルされます。
 *              一度セットした再生要求が Palma に届き再生開始されるまでは、次の再生要求を発行することができません。
 *              再生開始後(再生中)であれば、新たな Activity の再生開始を要求することができます。
 *              この場合は、現在の再生を中断して新たに指定された Activity を開始します。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   index                   再生する Activity の Index
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result PlayPalmaActivity(const PalmaConnectionHandle& handle, const PalmaActivityIndex& index) NN_NOEXCEPT;

/**
 * @brief       Palma FrMode の動作種別を設定します。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_SetFrModeType が設定されます。
 *
 *              PalmaFrModeType の具体的な動作仕様は、Palma の仕様を確認してください。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   type                    FrMode の動作種別
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result SetPalmaFrModeType(const PalmaConnectionHandle& handle, const PalmaFrModeType& type) NN_NOEXCEPT;

/**
 * @brief       Palma の内部に格納された Step 機能の値を取得します。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_ReadPalmaStep が設定されます。
 *              読み出された UniqueCode は PalmaOperationInfo::readPalmaStep 内部に設定されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result ReadPalmaStep(const PalmaConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       Palma の Step 機能の有効/無効を設定します。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_EnableStep が設定されます。
 *
 *              Palma の Step 機能は、出荷状態で無効になっています。
 *              Step 機能を有効にするためには、本関数で設定を行ってください。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   isEnabled               有効/無効の設定
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result EnablePalmaStep(const PalmaConnectionHandle& handle, bool isEnabled) NN_NOEXCEPT;

/**
 * @brief       Palma の内部に格納された Step 値を 0 で初期化します。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_ResetPalmaStep が設定されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result ResetPalmaStep(const PalmaConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       Palma の Application Section のデータを読み出します。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_ReadApplicationSection が設定されます。
 *              読み出された内容は PalmaOperationInfo::readApplicationSection に設定されます。
 *
 *              Application Section のアドレスは 0 から開始します。
 *              サイズの最大長は PalmaApplicationSectionSize です。
 *              1回あたりの読み出しは、PalmaApplicationSectionAccessUnitSize 以下です。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   address                 読み出し先のアドレス
 * @param[in]   size                    読み出すデータのサイズ
 *
 * @pre
 *              - address >= 0
 *              - size >= 1
 *              - address + size <= PalmaApplicationSectionSize
 *              - size <= PalmaApplicationSectionAccessUnitSizeMax
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result ReadPalmaApplicationSection(const PalmaConnectionHandle& handle, int address, size_t size) NN_NOEXCEPT;

/**
 * @brief       Palma の Application Section の任意のデータを書き込みます。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_WriteApplicationSection が設定されます。
 *              PalmaOperationInfo::writeApplicationSection を参照することで書き込みが行われたアドレスとサイズを参照することが可能です。
 *
 *              Application Section のアドレスは 0 から開始します。
 *              サイズの最大長は PalmaApplicationSectionSize です。
 *              1回あたりの書き込みは、PalmaApplicationSectionAccessUnitSize 以下です。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   address                 書き込む先のアドレス
 * @param[in]   size                    書き込むデータのサイズ
 * @param[in]   buffer                  書き込むデータが格納されたバッファ
 *
 * @pre
 *              - address >= 0
 *              - size >= 1
 *              - address + size <= PalmaApplicationSectionSize
 *              - size <= PalmaApplicationSectionAccessUnitSizeMax
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result WritePalmaApplicationSection(const PalmaConnectionHandle& handle, int address, size_t size, const PalmaApplicationSectionAccessBuffer& buffer) NN_NOEXCEPT;

/**
 * @brief       Palma の内部に格納された UniqueCode を取得します。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_ReadUniqueCode が設定されます。
 *              読み出された UniqueCode は PalmaOperationInfo::readUniqueCode に設定されます。
 *
 *              UniqueCode が無効になっている場合、PalmaOperationInfo::result に ResultPalmaUniqueCodeInvalid が設定されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result ReadPalmaUniqueCode(const PalmaConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       Palma の内部に格納された UniqueCode を無効にします。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_SetUniqueCodeInvalid が設定されます。
 *
 *              UniqueCode が無効にすると、以降同一の Palma に対して ReadUniqueCode をした場合の処理結果が ResultPalmaUniqueCodeInvalid となります。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result SetPalmaUniqueCodeInvalid(const PalmaConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       Activity Entry を Palma の ActivityDataBase の指定した Index に書込みます。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_WriteActivityEntry が設定されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   index                   Activity を書き込む Index
 * @param[in]   pEntry                  書き込む Activity Entry
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result WritePalmaActivityEntry(const PalmaConnectionHandle& handle, const PalmaActivityIndex& index, const PalmaActivityEntry* pEntry) NN_NOEXCEPT;

/**
 * @brief       RGB LED パターンを Palma の ActivityDataBase の指定した Index に書込みます。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_WriteRgbLedPatternEntry が設定されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   index                   RGB LED パターンデータを書き込む Index
 * @param[in]   pRgbLedPatternBuffer    書き込む RGB LED パターンが格納されたバッファ
 * @param[in]   size                    書き込むサイズ
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result WritePalmaRgbLedPatternEntry(const PalmaConnectionHandle& handle, const PalmaRgbLedPatternIndex& index, const void* pRgbLedPatternBuffer, size_t size) NN_NOEXCEPT;

/**
 * @brief       音源データを Palma の ActivityDataBase の指定した音源データベースの Index に書込みます。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_WriteWaveEntry が設定されます。
 *              Buffer と BufferSize は nn::os::MemoryPageSize の整数倍である必要があります。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   waveSet                 音源データベースの種類
 * @param[in]   index                   音源データを書き込む Index
 * @param[in]   pWaveBuffer             書き込む音源データが格納されたバッファ
 * @param[in]   bufferSize              音源データのバッファサイズ
 * @param[in]   waveDataSize            実際の音源データのサイズ
 *
 * @pre
 *  - pWaveBuffer が nn::os::MemoryPageSize の整数倍である。
 *  - bufferSize が nn::os::MemoryPageSize の整数倍である。
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result WritePalmaWaveEntry(const PalmaConnectionHandle& handle, const PalmaWaveSet& waveSet, const PalmaWaveIndex& index, const void* pWaveBuffer, size_t bufferSize, size_t waveDataSize) NN_NOEXCEPT;

/**
 * @brief       Palma ActivityDataBase のデータベース識別バージョンを設定します
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_WriteDataBaseIdentificationVersion が設定されます。
 *
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   version                 データベース識別バージョン
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result SetPalmaDataBaseIdentificationVersion(const PalmaConnectionHandle& handle, int version) NN_NOEXCEPT;

/**
 * @brief       Palma ActivityDataBase のデータベース識別バージョンを取得します
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_ReadDataBaseIdentificationVersion が設定されます。
 *              読み出されたデータベース識別バージョンは PalmaOperationInfo::readDataBaseIdentificationVersion 内部に設定されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result GetPalmaDataBaseIdentificationVersion(const PalmaConnectionHandle& handle) NN_NOEXCEPT;

/**
 * @brief       Palma の Feature Set の機能を一時的に停止/再開を操作します。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_SuspendFeature が設定されます。
 *
 *              SuspendFeatureSet でビットの立っている機能を一時的に停止します。ビットが立っていない機能は再開します。
 *              Palma の Step 機能は、EnablePalmaStep で機能自体が有効になっている状態で本関数を用いてサスペンドした場合に一時的に停止します。
 *              一時的に停止した Feature Set の機能はこの関数でレジュームした場合か、Nx と Palma の通信が切断された場合に再開されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   suspendFeatureSet       停止/再開する Palma の Feature Set
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result SuspendPalmaFeature(const PalmaConnectionHandle& handle, const PalmaFeatureSet& suspendFeatureSet) NN_NOEXCEPT;


/**
 * @brief       Palma の内部に格納された Play Log の値を取得します。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_ReadPlayLog が設定されます。
 *
 *              FieldIndex で指定された値を読み出します。
 *              指定した FieldIndex が存在しない場合、PalmaOperationInfo::result に ResultPalmaInvalidFieldIndex が設定されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   index                   読み出す Play Log のインデックス
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result ReadPalmaPlayLog(const PalmaConnectionHandle& handle, const PalmaPlayLogFieldIndex& index) NN_NOEXCEPT;

/**
 * @brief       Palma の内部に格納された Play Log の値をリセットします。
 *
 * @return      本処理は非同期に行われます。処理の完了は BindPalmaOperationCompleteEvent() で割り当てたシステムイベントへの通知で検出します。
 *              処理の完了後、GetPalmaOperationInfo() で処理結果を取得することが可能です。
 *              PalmaOperationInfo::type には PalmaOperationType_ResetPlayLog が設定されます。
 *
 *              FieldIndex で指定された値をリセットします。
 *              指定した FieldIndex が存在しない場合、PalmaOperationInfo::result に ResultPalmaInvalidFieldIndex が設定されます。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 * @param[in]   index                   リセットする Play Log のインデックス
 *
 * @retresult
 *   @handleresult{nn::ResultSuccess,                      処理に成功しました。}
 *   @handleresult{nn::hid::ResultPalmaInvalidHandle,      無効なハンドルが指定されています。}
 *   @handleresult{nn::hid::ResultPalmaBusy,               他の操作を処理中のため、本操作を開始できません。}
 * @endretresult
 */
Result ResetPalmaPlayLog(const PalmaConnectionHandle& handle, const PalmaPlayLogFieldIndex& index) NN_NOEXCEPT;

/**
 * @brief       接続されている Palma とペアリングを行います。
 *
 * @return      接続されている Palma とのペアリングを行います。
 *              本関数で登録が行われた Palma は、 EnablePairedPalmaConnection() で待ち受けが有効になっている際に接続することが可能になります。
 *
 * @param[in]   handle                  Palma のコネクションハンドル
 */
void PairPalma(const PalmaConnectionHandle& handle)  NN_NOEXCEPT;

/**
 * @brief       Palma の高速通信モードの使用の有無を設定します。
 *
 * @return      WritePalmaWaveEntry() などで高速な書き換えを行いたい場合は、高速通信モードを有効にすることができます。
 *              デフォルトでは高速通信モードは無効になっています。
 *
 */
void SetPalmaBoostMode(bool enabled)  NN_NOEXCEPT;

//! @}

}} // namespace nn::hid
