﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Xpad のデバッグ機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_BatteryLevel.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_Xpad.h>

namespace nn { namespace hid { namespace debug {

/**
 * @brief       基本的な構成を持つ Xpad の自動操作状態を表す構造体です。
 */
struct BasicXpadAutoPilotState
{
    int32_t powerState;             //!< 基本的な構成を持つ Xpad の電源状態です。
    int32_t batteryLevel;           //!< 基本的な構成を持つ Xpad のバッテリー残量です。
    BasicXpadButtonSet buttons;     //!< 基本的な構成を持つ Xpad のデジタルボタンの状態です。
    AnalogStickState analogStickL;  //!< 基本的な構成を持つ Xpad の左アナログスティックの状態です。
    AnalogStickState analogStickR;  //!< 基本的な構成を持つ Xpad の右アナログスティックの状態です。
};

/**
 * @brief       Xpad をファイナライズします。
 *
 * @details     指定の Xpad ID と対応する Xpad について、
                事前に nn::hid::InitializeXpad() を呼び出した回数と同じだけ
 *              呼びだされた際に Xpad の初期化状態を解除します。
 *
 * @param[in]   xpadId                      Xpad ID
 *
 * @pre
 *              - Xpad が初期化状態にある
 */
void FinalizeXpad(const BasicXpadId& xpadId) NN_NOEXCEPT;

/**
 * @brief       Xpad に自動操作状態を設定します。
 *
 * @details     Xpad ID で指定した Xpad に自動操作状態を設定します。
 *
 * @param[in]   xpadId                      Xpad ID
 * @param[in]   state                       Xpad の自動操作状態
 *
 * @pre
 *              - Xpad が初期化状態にある
 */
void SetXpadAutoPilotState(const BasicXpadId& xpadId,
                           const BasicXpadAutoPilotState& value) NN_NOEXCEPT;

/**
 * @brief       Xpad の自動操作状態を解除します。
 *
 * @details     Xpad ID で指定した Xpad の自動操作状態を解除します。
 *
 * @param[in]   xpadId                      Xpad ID
 *
 * @pre
 *              - Xpad が初期化状態にある
 */
void UnsetXpadAutoPilotState(const BasicXpadId& xpadId) NN_NOEXCEPT;

}}} // namespace nn::hid::debug
