﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/gfxTool/gfxTool_Util.h>
#include <nn/gfxTool/gfxTool_ApiCommon.h>
#include <nn/gfxTool/gfxTool_Error.h>

// 正式版に乗り換えるまでの暫定

namespace nn {
namespace gfxTool {

enum class OptionValueType
{
    None,
    One,
    Multi
};

struct CommandLineOptionDefinition
{
    char shortName;
    bool isRequired;
    size_t longNameLength;
    size_t descriptionLength;
    const char* pLongName;
    const char* pDescription;
};

class CommandLineOptionBase
{
public:
    CommandLineOptionBase()
        : m_IsExisting( false )
    {
    }

    CommandLineOptionBase* Define( bool isRequired, char shortName, const char* pDescription )
    {
        return Define( isRequired, shortName, nullptr, pDescription );
    }

    CommandLineOptionBase* Define( bool isRequired, const char* pLongName, const char* pDescription )
    {
        return Define( isRequired, 0, pLongName, pDescription );
    }

    CommandLineOptionBase* Define( bool isRequired, char shortName,
        const char* pLongName, const char* pDescription )
    {
        m_Definition.isRequired = isRequired;
        m_Definition.pDescription = pDescription;
        m_Definition.shortName = shortName;
        m_Definition.pLongName = pLongName;
        m_Definition.descriptionLength = strlen( pDescription );
        m_Definition.longNameLength = pLongName ? strlen( pLongName ) : 0;
        return this;
    }

    const CommandLineOptionDefinition& GetDefinition() const
    {
        return m_Definition;
    }

    virtual OptionValueType GetOptionValueType() const = 0;
    virtual void SetValue( const char* pValue ) = 0;

    bool IsExisting() const
    {
        return m_IsExisting;
    }

    void SetExisting( bool value )
    {
        m_IsExisting = value;
    }

protected:
    CommandLineOptionDefinition m_Definition;

    bool m_IsExisting;
};

template< typename TType = void >
class CommandLineOption
    : public CommandLineOptionBase
{
public:
    typedef TType ValueType;

    virtual OptionValueType GetOptionValueType() const override
    {
        return OptionValueType::One;
    }

    virtual void SetValue( const char* pValue ) override
    {
        SetValue( LexicalCast< ValueType >( pValue ) );
    }

    void SetValue( const ValueType& value )
    {
        m_Value = value;
    }

    const ValueType& GetValue() const
    {
        return m_Value;
    }

private:
    TType m_Value;
};

template<>
class CommandLineOption< void >
    : public CommandLineOptionBase
{
public:
    virtual OptionValueType GetOptionValueType() const override
    {
        return OptionValueType::None;
    }

    virtual void SetValue( const char* ) override
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }
};

template< typename TType >
class CommandLineOption< TType[] >
    : public CommandLineOptionBase
{
public:
    typedef TType ValueType;

    virtual OptionValueType GetOptionValueType() const override
    {
        return OptionValueType::Multi;
    }

    virtual void SetValue( const char* pValue ) override
    {
        SetValue( LexicalCast< ValueType >( pValue ) );
    }

    void SetValue( const ValueType & value )
    {
        m_Value.push_back( value );
    }

    const typename Custom< std::vector< ValueType > >::Type& GetValue() const
    {
        return m_Value;
    }

private:
    typename Custom< std::vector< ValueType > >::Type m_Value;
};

class CommandLineParser
{
public:
    /**
    * @brief CommandLineParser::Parse() で未定義のオプションを検出した際に呼ばれるコールバックの型です。
    * @param[in] argc 文字列配列の個数
    * @param[in] argv 文字列配列
    * @param[in] argIndex 未定義オプションが含まれる argv の index
    * @param[in] pCallbackParam ユーザーポインター
    * @return 未定義オプションを無視する場合は true を、エラーとする場合は false を返します。
    *         無視した場合は、argv[argIndex+1] から次の解析を再開します。
    */
    typedef bool(*UndefinedOptionCallback)(int argc, const char* const* argv, int argIndex, void* pCallbackParam);

    class CommandLineArg
    {
    public:
        explicit CommandLineArg( const nn::util::string_view& commandLine );

        int GetArgC() const
        {
            return StaticCastAuto( m_Args.size() );
        }

        const char* const * GetArgv() const
        {
            return m_Args.size() > 0 ? &m_Args[ 0 ] : nullptr;
        }

    private:
        Custom< std::vector< const char* > >::Type m_Args;
        Custom< std::vector< Custom< std::vector< char > >::Type > >::Type m_ArgStrings;
    };

    CommandLineParser()
        : m_Options()
        , m_pArgs()
        , m_pCallback( nullptr )
        , m_pUndefinedOptionCallback( nullptr )
    {
        SetOption( m_ArgsFile.Define( false, "args-file", "Import arguments from a file." ) );    // TODO: 翻訳の仕組みができたら修正する。"引数を外部ファイルから入力します"
    }

    void SetOption( CommandLineOptionBase* pOption )
    {
        m_Options.push_back( pOption );
    }

    void SetArgs( Custom< std::vector< Custom< std::string >::Type > >::Type* pArgs )
    {
        m_pArgs = pArgs;
    }

    void SetReadFileCallback( nngfxToolReadFileCallback pCallback, void* pCallbackParam )
    {
        m_pCallback = pCallback;
        m_pCallbackParam = pCallbackParam;
    }

    void SetUndefinedOptionCallback(UndefinedOptionCallback pCallback, void* pCallbackParam)
    {
        m_pUndefinedOptionCallback = pCallback;
        m_pUndefinedOptionCallbackParam = pCallbackParam;
    }

    void ResetOptions()
    {
        Custom< std::vector< CommandLineOptionBase* > >::Type().swap( m_Options );
        SetOption( &m_ArgsFile );
    }

    void PrintHelp( const char* pToolName, const char* pToolDescription );

    void Parse( int argc, const char* const* argv );

    void CheckRequiredOption();

private:
    void ExpandExternalFile( const char* pPath );

    Custom< std::vector< CommandLineOptionBase* > >::Type m_Options;
    Custom< std::vector< Custom< std::string >::Type > >::Type* m_pArgs;

    CommandLineOption< Custom< std::string >::Type > m_ArgsFile;

    nngfxToolReadFileCallback m_pCallback;
    void* m_pCallbackParam;

    UndefinedOptionCallback m_pUndefinedOptionCallback;
    void* m_pUndefinedOptionCallbackParam;
};

}
}
