﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace ens {

/**
 * @brief   検索クエリのビルダー
 *
 * @details
 *  検索機能を有する API に指定する検索クエリのビルダーです。
 *
 *  URL エンコードされた状態の文字列を生成します。@n
 *  値に「あいうえお」を指定した場合、 "%E3%81%82%E3%81%84%E3%81%86%E3%81%88%E3%81%8A" という文字列が生成されます。
 *
 *  検索に指定するキーは以下の文字が使用可能です。
 *
 *  - 半角英数
 *  - ハイフン
 *  - アンダースコア
 */
class SearchQueryBuilder
{
private:
    NN_DISALLOW_COPY(SearchQueryBuilder);
    NN_DISALLOW_MOVE(SearchQueryBuilder);

public:
    /**
     * @brief   コンストラクタ
     *
     * @param[in]   pBuffer バッファ
     * @param[in]   size    バッファサイズ
     *
     * @pre
     *  - pBuffer != nullptr
     *  - size > 0
     *
     * @details
     */
    SearchQueryBuilder(char* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief   パラメータを設定します。
     *
     * @param[in]   pKey    キー
     * @param[in]   pValue  値
     *
     * @return  設定できたかどうか
     *
     * @pre
     *  - pKey != nullptr
     *  - pValue != nullptr
     *  - pValue が UTF-8 でエンコードされている
     *
     * @details
     *
     * @attention
     *  同名のキーを複数回指定することはできません。
     *
     * @note
     *  キーや値の範囲については、サーバー側と相談の上決定してください。
     */
    bool Set(const char* pKey, const char* pValue) NN_NOEXCEPT;

    /**
     * @brief   複数の値が指定可能なパラメータを設定します。
     *
     * @param[in]   pKey    キー
     * @param[in]   pValue  値
     *
     * @return  設定できたかどうか
     *
     * @pre
     *  - pKey != nullptr
     *  - pValue != nullptr
     *  - pValue が UTF-8 でエンコードされている
     *
     * @details
     *
     * @note
     *  キーや値の範囲については、サーバー側と相談の上決定してください。
     */
    bool SetMulti(const char* pKey, const char* pValue) NN_NOEXCEPT;

    /**
     * @brief   生成した文字列を取得します。
     *
     * @return  生成した文字列
     *
     * @details
     */
    const char* GetString() const NN_NOEXCEPT;

private:
    //
    char* m_pBuffer;
    //
    size_t m_Size;
    size_t m_Length;

private:
    //
    bool SetImpl(const char* pKey, const char* pValue, bool isMulti) NN_NOEXCEPT;
    //
    bool WriteKey(const char* pKey, bool isMulti) NN_NOEXCEPT;
    bool WriteValue(const char* pValue) NN_NOEXCEPT;
};

}}
