﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/ens/ens_AsyncContext.h>
#include <nn/ens/ens_ResultPublic.h>
#include <nn/ens/ens_Types.h>

namespace nn { namespace ens {

//! @name 通信機能の利用開始／終了
//! @{

/**
 * @brief   拡張ネットワークサービスの利用を開始します。（ブロック関数）
 *
 * @param[in]   pServerKey  利用するサーバーを示すキー文字列
 * @param[in]   pBuffer     ライブラリが利用するバッファ
 * @param[in]   bufferSize  pBuffer のサイズ
 *
 * @pre
 *  - nn::ens::StartServiceLoop() が実行中でない
 *  - スレッドスタックが nn::ens::RequiredStackSize 以上確保されている
 *  - nn::socket::Initialize が呼び出し済み
 *  - libcurl が初期化済み
 *  - pServerKey != nullptr
 *  - pBuffer != nullptr
 *  - bufferSize > nn::ens::RequiredMemorySizeMin
 *
 * @details
 *  拡張ネットワークサービスの利用を開始します。
 *
 *  本関数は nn::ens::StopServiceLoop() の呼び出し完了までブロックし、内部で通信処理を行います。@n
 *  本関数が処理を返すまで、 nn::socket と libcurl を破棄してはいけません。
 *
 *  サーバーキーには、事前に発行された文字列を指定します。
 *
 *  ライブラリが利用するバッファには、一度のリクエストで扱う通信データサイズ（送信サイズまたは受信サイズのより大きい方）に
 *  nn::ens::RequiredMemorySizeMin を加算したサイズ以上のバッファを指定します。
 *
 * @note
 *  socket の並列利用数は 1 です。ただし将来変わる可能性があります。
 *
 * @see nn::ens::StopServiceLoop()
 */
void StartServiceLoop(const char* pServerKey, void* pBuffer, size_t bufferSize) NN_NOEXCEPT;

/**
 * @brief   拡張ネットワークサービスの利用を終了します。
 *
 * @pre
 *  - nn::ens::StartServiceLoop() が実行中である
 *
 * @details
 *  拡張ネットワークサービスの利用を終了します。
 *
 *  本関数の実行によって、ブロック中の nn::ens::StartServiceLoop() が返ります。
 *
 *  完了していない非同期処理があっても本関数を呼ぶことができます。@n
 *  その場合、完了していない非同期処理はキャンセルされます。
 *
 * @see nn::ens::StartServiceLoop()
 */
void StopServiceLoop() NN_NOEXCEPT;

//! @}

//! @name アカウント
//! @{

/**
 * @brief   サービスを利用するためのユーザー ID を含む認証情報を生成します。
 *
 * @param[out]  pOutContext     非同期コンテキスト
 * @param[out]  pOutCredential  認証情報（非同期処理成功後に取得可能）
 * @param[in]   pToken          ネットワークサービスアカウントの ID トークン
 *
 * @pre
 *  - @link about_ens_pre_condition ネットワーク通信を行う API の事前条件 @endlink を満たしている
 *  - pOutContext != nullptr
 *  - pOutCredential != nullptr
 *  - pToken != nullptr
 *
 * @details
 *  サービスを利用するためのユーザー ID を含む認証情報を生成します。
 *
 *  同じユーザー ID に対する認証情報 (nn::ens::Credential) は変化することはありません。@n
 *  生成した認証情報をセーブデータなどに永続化し、次回以降の起動へ引き継ぐことで同じユーザーとしてサービスを利用することができます。
 *
 *  非同期処理の成否は pOutContext の nn::ens::AsyncContext::GetResult() で確認してください。
 */
void GenerateCredential(AsyncContext* pOutContext,
    Credential* pOutCredential, const char* pToken) NN_NOEXCEPT;

//! @}

}}
