﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief  アプリケーション独自のアルバムファイル操作関連機能に関する定義
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/album/album_AlbumFileEntry.private.h>

namespace nn { namespace album {

//! @name アプリケーション独自の動画ファイル関連機能
//! @{

//! @brief 動画ファイルに埋め込むことのできる任意データの最大サイズです。
const size_t AlbumUserDataSizeMax = 1024;


//----------------------------------------------------------------------------
//! @brief アプリケーション独自の動画ファイルのファイルリストを取得します。
//!
//! @param[out] pOutCount     取得した動画ファイル数の格納先ポインタ
//! @param[out] buffer        取得した動画ファイルリストを受け取る配列
//! @param[in]  bufferLength  buffer で渡す配列の長さ（要素数）
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである。
//! @pre bufferLength >= 0
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! アプリケーションが独自に作成・保存した現在の動画ファイルのファイルリストを
//! buffer が指す AlbumFileEntry 配列に取得します。
//!
//! buffer に書き込まれる要素の順番は保証されません。
//! また、bufferLength が取得可能なファイル数よりも大きな値だった場合、
//! 0 から (*pOutCount - 1) 番目の要素に値が書き込まれます。
//!
//! bufferLength が取得可能な動画ファイル数よりも小さな値だった場合、
//! buffer には bufferLength 個の要素だけが書き込まれます。
//!
//! NX Addon 5.1 では、アプリケーションが独自に作成・保存可能な動画ファイルの
//! 最大数は規定されていません。上記 buffer および bufferLength には、
//! アプリケーションにとって都合のよいバッファ容量を渡すようにして下さい。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでユーザに告知できます。
//!
Result GetExtraMovieFileList(int* pOutCount, AlbumFileEntry buffer[], int bufferLength) NN_NOEXCEPT;


//----------------------------------------------------------------------------
//! @brief アプリケーション独自の動画ファイルを削除します。
//!
//! @param[in] entry    削除する対象ファイルの識別子です。
//!
//! @pre nn::album::Initialize() によりアルバムライブラリが初期化済みである
//! @post 成功した場合、指定したファイルは削除されます。
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! アプリケーションが独自に作成・保存した動画ファイルを削除します。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでユーザに告知できます。
//!
Result DeleteExtraMovieFile(const AlbumFileEntry& entry) NN_NOEXCEPT;


//----------------------------------------------------------------------------
//! @brief アプリケーション独自の動画ファイルの静止画の画像データを、任意データと共に読み込みます。
//!
//! @param[out] pOutWidth          読み込んだ静止画の横幅ピクセル数の格納先
//! @param[out] pOutHeight         読み込んだ静止画の高さピクセル数の格納先
//! @param[out] pOutUserDataSize   読み込んだ静止画ファイルの任意データの実サイズ格納先
//! @param[out] pOutUserDataBuffer 読み込んだ静止画ファイルの任意データの格納先
//! @param[in]  userDataBufferSize pOutUserDataBuffer のサイズ（バイト）
//! @param[out] pBuffer            画像データを受け取るバッファ
//! @param[in]  bufferSize         画像データを受け取るバッファサイズ（バイト）
//! @param[in]  pWorkBuffer        作業用バッファ
//! @param[in]  workBufferSize     作業用バッファサイズ（バイト）
//! @param[in]  entry              対象ファイルの識別子
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! 基本的な機能は、
//! nn::album::LoadAlbumImage(int*, int*, void*, size_t, void*, size_t, const AlbumFileEntry& entry)
//! と等価です。返値や事前条件等の詳細はそちらを参照して下さい。
//!
//! 本 API では上記機能に加えて、アルバムファイルに埋め込まれた任意データを
//! pOutUserDataBuffer と userDataBufferSize で指定された領域に格納します。
//! また、任意データの実サイズを pOutUserDataSize に格納します。
//!
//! 任意データの実サイズが userDataBufferSize よりも大きい場合、
//! pOutUserDataSize には任意データの先頭から userDataBufferSize 分だけが
//! 格納されます。任意データを完全に取得するには、
//! nn::album::AlbumUserDataSizeMax バイト以上のバッファを渡して下さい。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでユーザに告知できます。
//!
Result LoadAlbumImage(
    int* pOutWidth,
    int* pOutHeight,
    size_t* pOutUserDataSize,
    void* pOutUserDataBuffer,
    size_t userDataBufferSize,
    void* pBuffer,
    size_t bufferSize,
    void* pWorkBuffer,
    size_t workBufferSize,
    const AlbumFileEntry& entry
    ) NN_NOEXCEPT;


//----------------------------------------------------------------------------
//! @brief アプリケーション独自の動画ファイルのサムネイル画像データを、任意データと共に読み込みます。
//!
//! @param[out] pOutWidth          読み込んだサムネイル画像の横幅ピクセル数の格納先
//! @param[out] pOutHeight         読み込んだサムネイル画像の高さピクセル数の格納先
//! @param[out] pOutUserDataSize   読み込んだ静止画ファイルの任意データの実サイズ格納先
//! @param[out] pOutUserDataBuffer 読み込んだ静止画ファイルの任意データの格納先
//! @param[in]  userDataBufferSize pOutUserDataBuffer のサイズ（バイト）
//! @param[out] pBuffer            サムネイル画像データを受け取るバッファ
//! @param[in]  bufferSize         サムネイル画像データを受け取るバッファサイズ
//! @param[in]  pWorkBuffer        作業用バッファ
//! @param[in]  workBufferSize     作業用バッファの大きさ（バイト）
//! @param[in]  entry              対象ファイルの識別子
//!
//! @return nn::Result 型で結果を返します。
//!
//! @details
//! 基本的な機能は、
//! nn::album::LoadAlbumThumbnailImage(int*, int*, void*, size_t, void*, size_t, const AlbumFileEntry&)
//! と等価です。返値や事前条件等の詳細はそちらを参照して下さい。
//!
//! 本 API では上記機能に加えて、アルバムファイルに埋め込まれた任意データを
//! pOutUserDataBuffer と userDataBufferSize で指定された領域に格納します。
//! また、任意データの実サイズを pOutUserDataSize に格納します。
//!
//! 任意データの実サイズが userDataBufferSize よりも大きい場合、
//! pOutUserDataSize には任意データの先頭から userDataBufferSize 分だけが
//! 格納されます。任意データを完全に取得するには、
//! nn::album::AlbumUserDataSizeMax バイト以上のバッファを渡して下さい。
//!
//! 返値が nn::Result::IsSuccess() == true でない場合は、
//! その値を使ってエラービューアを起動することでユーザに告知できます。
//!
Result LoadAlbumThumbnailImage(
    int* pOutWidth,
    int* pOutHeight,
    size_t* pOutUserDataSize,
    void* pOutUserDataBuffer,
    size_t userDataBufferSize,
    void* pBuffer,
    size_t bufferSize,
    void* pWorkBuffer,
    size_t workBufferSize,
    const AlbumFileEntry& entry
    ) NN_NOEXCEPT;

//! @}

}}  // namespace nn::album
