﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <cstdint>
/**
*   @file
*   @brief  キャプチャデバイスアクセス、画像処理に関する定義
*/


/**
*   @brief  画像データ ID 型
*/
struct ImageIdType
{
    unsigned int id;
};

/**
*   @brief  画像変換フィルターのタイプ
*/
struct FilterIdType
{
    unsigned int id;
};

/**
 * @enum CaptureResult
 * キャプチャデバイスアクセス操作の結果
 */
enum CaptureResult : int32_t
{
    //! 操作に成功したときの値です。
    CaptureResult_Success        = 0,

    //! 不測のエラーが発生したときの値です。
    CaptureResult_Unexpected     = -1,

    //! 不正な画像データ ID が指定された場合の値です。
    CaptureResult_InvalidImageId = -2,

    //! 対象が見つからなかった場合の値です。
    CaptureResult_NotFound = -3,

    //! Target Manager から情報取得に失敗した場合の値です。
    CaptureResult_TargetManagerError = -4,

    //! 未初期化状態で呼び出されたときの値です。
    CaptureResult_NotInitialized = -5,

    //! ファイル作成に失敗した場合の値です。
    CaptureResult_FileCreationFailed = -6,

    //! 不正なフィルター ID が指定された場合の値です。
    CaptureResult_InvalidFilterId = -7,

    //! 不正なフィルターパラメーターが指定された場合の値です。
    CaptureResult_InvalidParameter = -8,

    //! 不正な OCR 結果ハンドルが指定された場合の値です。
    CaptureResult_InvalidOcrResultHandle = -9,

    //! 不正な ラベリング 結果ハンドルが指定された場合の値です。
    CaptureResult_InvalidLabelingResultHandle = -10,
};

/**
 * @enum MatchingResult
 * マッチング操作の結果
 */
enum MatchingResult : int32_t
{
    //! マッチングに成功したときの値です。
    MatchingResult_Success         = 0,

    //! マッチングに失敗したときの値です。
    MatchingResult_Failed          = -1,

    //! 不測のエラーが発生したときの値です。
    MatchingResult_Unexpected      = -2,

    //! 不正な画像データ ID が指定された場合の値です。
    MatchingResult_InvalidImageId  = -3,

    //! 未初期化状態で呼び出されたときの値です。
    MatchingResult_NotInitialized = -4,

    //! 不正なフィルター ID が指定された場合の値です。
    MatchingResult_InvalidFilterId = -5,
};

/**
 * @enum CaptureResolution
 * キャプチャ解像度
 */
enum CaptureResolution : int32_t
{
    //! 720p キャプチャ
    CaptureResolution_720p = 0,

    //! 1080p キャプチャ
    CaptureResolution_1080p = 1,

    CaptureResolution_Count,
};

/**
 * @struct Rgb24Color
 * RGB 値
 */
struct Rgb24Color
{
    //! 赤色
    int red;

    //! 緑色
    int green;

    //! 青色
    int blue;
};

/**
 * @brief CaptureProcessor ライブラリ内で扱う画像のカラーチャンネル数です。
 */
const int ColorChannelCount = 3;

/**
 * @enum MorphologyConversionType
 * モルフォロジー変換の種類
 */
enum MorphologyConversionType : int32_t
{
    //! 収縮
    MorphologyConversionType_Erosion,

    //! 膨張
    MorphologyConversionType_Dilation,

    //! 収縮 -> 膨張
    MorphologyConversionType_Opening,

    //! 膨張 -> 収縮
    MorphologyConversionType_Closing,
};

/**
 * @struct ImageRect
 * 画像の領域
 */
struct ImageRect
{
    //! 左上 X 座標
    int left;
    //! 左上 Y 座標
    int top;
    //! 幅
    int width;
    //! 高さ
    int height;
};

/**
 * @struct LabelingResultHandleType
 * ラベリングの結果ハンドル
 */
struct LabelingResultHandleType
{
    unsigned int handle;
};

/**
 * @struct LabelingResult
 * ラベリング結果領域の情報
 */
struct LabelingResult
{
    //! ラベリング結果領域の外接矩形
    ImageRect rect;
    //! 面積
    int area;
    //! 重心の X 座標
    int centerX;
    //! 重心の Y 座標
    int centerY;
};

/**
 * @struct OcrResultHandleType
 * OCR の結果ハンドル
 */
struct OcrResultHandleType
{
    int handle;
};

/**
 * @enum OcrLanguage
 * OCR の対象言語
 */
enum OcrLanguage : int32_t
{
    //! 日本語
    OcrLanguage_Japanese,
    //! 英語
    OcrLanguage_English
};

/**
 * @enum OcrSeparateLevel
 * OCR の認識単位
 */
enum OcrSeparateLevel : int32_t
{
    //! 単語
    OcrSeparateLevel_Word,
    //! 行
    OcrSeparateLevel_Line,
};
