﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "CppHeaderWriter.h"
#include <cerrno>

#include "util.h"


using namespace std;

const char *CppHeaderWriter::g_pBoilerplateHeader =
"\xef\xbb\xbf/*--------------------------------------------------------------------------------*\n"
"  Copyright (C)Nintendo All rights reserved.\n"
"\n"
"  These coded instructions, statements, and computer programs contain proprietary\n"
"  information of Nintendo and/or its licensed developers and are protected by\n"
"  national and international copyright laws. They may not be disclosed to third\n"
"  parties or copied or duplicated in any form, in whole or in part, without the\n"
"  prior written consent of Nintendo.\n"
"\n"
"  The content herein is highly confidential and should be handled accordingly.\n"
" *--------------------------------------------------------------------------------*/\n"
"\n"
"#pragma once"
"\n";

const char *CppHeaderWriter::g_pAutoGenHeader =
"\n"
"namespace nn { namespace ssl {\n\n";

const char *CppHeaderWriter::g_pAutoGenFooter =
"} }    //  nn::ssl\n";


const char *CppHeaderWriter::g_pEnumValAllFmt = "    %s_All = 0x%8.8X\n";
const char *CppHeaderWriter::g_pEnumStartFmt = "enum %s\n{\n";
const char *CppHeaderWriter::g_pEnumEnd = "};\n\n";
const char *CppHeaderWriter::g_pEnumEntryFmt = "    %s_%s = 0x%8.8X,\n";
const char CppHeaderWriter::g_pSkipChars[] =
{
    ' ', '\n', '\t', '\b',  '(',  ')',  '[',  ']',  '{',  '}',  '!',  '@',
    '#',  '$',  '%',  '^',  '&',  '*',  '_',  '+',  '-',  '=',  '`',  '`',
    ',',  '.',  '/', '\\',  '?',  ';',  '|',  ':', '\'', '\"',  '<',  '>'
};


char CppHeaderWriter::MakeUpper(char in)
{
    char                        ret = in;

    if ((in >= 'a') && (in <= 'a'))
    {
        ret = ret - 32;
    }

    return ret;
}


int CppHeaderWriter::MakeEnumName(char        *pOutBuf,
                                  size_t      bufSize,
                                  const char  *pInName)
{
    int                         ret = -1;
    size_t                      i;
    size_t                      j;
    int                         k;
    bool                        needUpper = false;

    if (bufSize < strlen(pInName))
    {
        LOGE("CppHeaderWriter::MakeEnumName: provided buf size is no large enough; got %u, need %u\n",
             bufSize,
             strlen(pInName));
        goto errExit;
    }

    // Scan the input name for special characters, skip them.
    // The way this works is the outer loop goes through each character in the
    // input name and compares it against the special "skip" characters (which
    // are not valid for C/C++ enum names.)  If a special character is
    // encountered, the 2nd loop breaks out and the 'needUpper' flag is set.
    // The loop will then continue to the next character.  Once no skip
    // characters are encountered, the current character is converted to
    // upper case (to retain some kind of camel case) if needed then copied
    // into the output.
    for (i = k = 0; i < strlen(pInName); i++)
    {
        char                    curChar = pInName[i];

        for (j = 0; j < sizeof(g_pSkipChars); j++)
        {
            if (curChar == g_pSkipChars[j])
            {
                needUpper = true;
                break;
            }
        }

        if (j == sizeof(g_pSkipChars))
        {
            if (needUpper)
            {
                curChar = MakeUpper(curChar);
                needUpper = false;
            }

            pOutBuf[k] = curChar;
            k++;
        }
    }

    //  Don't forget the null termination
    pOutBuf[k] = 0;
    ret = 0;

errExit:
    return ret;
}


int CppHeaderWriter::WriteEnumEntry(FILE             *pOutFile,
                                    const char       *pEnumName,
                                    BuiltinDataInfo  *pInfo)
{
    int                         ret = -1;
    size_t                      bytesWritten;
    size_t                      amtToWrite;
    char                        tmpBuf[g_MaxEnumLineLength];
    char                        nameBuf[g_MaxEnumLineLength];
    const char                  *pName;

    pName = pInfo->GetName();

    if (MakeEnumName(nameBuf, g_MaxEnumLineLength, pName) != 0)
    {
        LOGE("CppHeaderWriter::WriteEnumEntry: failed to convert name \'%s\'\n",
             pName);
        goto errExit;
    }

    LOGD("CppHeaderWriter::WriteEnumEntry: data \'%s\' enum is \'%s\', value 0x%8.8X\n",
         pName,
         nameBuf,
         pInfo->GetId());

    nn_snprintf(tmpBuf,
                g_MaxEnumLineLength,
                g_pEnumEntryFmt,
                pEnumName,
                nameBuf,
                pInfo->GetId());

    amtToWrite = strlen(tmpBuf);
    bytesWritten = fwrite(tmpBuf, 1, amtToWrite, pOutFile);
    if (bytesWritten < amtToWrite)
    {
        ret = ferror(pOutFile);
        LOGE("CppHeaderWriter::WriteEnumEntry: failed to write enum entry: %d\n", ret);
        goto errExit;
    }

    ret = 0;

errExit:
    return ret;
}


int CppHeaderWriter::DoWrite(FILE                              *pOutFile,
                             const char                        *pEnumName,
                             map<uint32_t, BuiltinDataInfo *>  *pBdiMap)
{
    int                         ret = -1;
    int                         status;
    size_t                      bytesWritten;
    size_t                      amtToWrite;
    char                        tmpBuf[g_MaxEnumLineLength];

    //  Seek to the start of the file
    status = fseek(pOutFile, 0, SEEK_SET);
    if (status < 0)
    {
        ret = -errno;
        LOGE("CppHeaderWriter::DoWrite: failed to see to start of file: %d\n", ret);
        goto errExit;
    }

    //  Write out the two headers
    amtToWrite = strlen(g_pBoilerplateHeader);
    bytesWritten = fwrite(g_pBoilerplateHeader, 1, amtToWrite, pOutFile);
    if (bytesWritten < amtToWrite)
    {
        ret = ferror(pOutFile);
        LOGE("CppHeaderWriter::DoWrite: failed to write header comment block: %d\n", ret);
        goto errExit;
    }

    amtToWrite = strlen(g_pAutoGenHeader);
    bytesWritten = fwrite(g_pAutoGenHeader, 1, amtToWrite, pOutFile);
    if (bytesWritten < amtToWrite)
    {
        ret = ferror(pOutFile);
        LOGE("CppHeaderWriter::DoWrite: failed to write auto-gen comment block: %d\n", ret);
        goto errExit;
    }

    //  Start the enum declaration
    nn_snprintf(tmpBuf,
                g_MaxEnumLineLength,
                g_pEnumStartFmt,
                pEnumName);
    amtToWrite = strlen(tmpBuf);
    bytesWritten = fwrite(tmpBuf, 1, amtToWrite, pOutFile);
    if (bytesWritten < amtToWrite)
    {
        ret = ferror(pOutFile);
        LOGE("CppHeaderWriter::DoWrite: failed to write enum start: %d\n", ret);
        goto errExit;
    }

    //  Write out each cert name and id
    for (map<uint32_t, BuiltinDataInfo *>::iterator it = pBdiMap->begin();
         it != pBdiMap->end();
         ++it)
    {
        BuiltinDataInfo         *pCurInfo = it->second;

        status = WriteEnumEntry(pOutFile, pEnumName, pCurInfo);
        if (status != 0)
        {
            goto errExit;
        }
    }

    //  Write out the "all" entry, which must be LAST
    nn_snprintf(tmpBuf,
                g_MaxEnumLineLength,
                g_pEnumValAllFmt,
                pEnumName,
                BuiltinDataInfo::g_DataIdAll);
    amtToWrite = strlen(tmpBuf);
    bytesWritten = fwrite(tmpBuf, 1, amtToWrite, pOutFile);
    if (bytesWritten < amtToWrite)
    {
        ret = ferror(pOutFile);
        LOGE("CppHeaderWriter::DoWrite: failed to write ALL enum entry: %d\n", ret);
        goto errExit;
    }

    //  Write out the enum ending
    amtToWrite = strlen(g_pEnumEnd);
    bytesWritten = fwrite(g_pEnumEnd, 1, amtToWrite, pOutFile);
    if (bytesWritten < amtToWrite)
    {
        ret = ferror(pOutFile);
        LOGE("CppHeaderWriter::DoWrite: failed to write enum end: %d\n", ret);
        goto errExit;
    }

    //  Finish out with the namspace footer
    amtToWrite = strlen(g_pAutoGenFooter);
    bytesWritten = fwrite(g_pAutoGenFooter, 1, amtToWrite, pOutFile);
    if (bytesWritten < amtToWrite)
    {
        ret = ferror(pOutFile);
        LOGE("CppHeaderWriter::DoWrite: failed to write auto-gen footer block: %d\n", ret);
        goto errExit;
    }

    //  Done, change ret to 0 and get out
    ret = 0;

errExit:
    return ret;
}
