﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SettingsManager
{
    using System.Collections.Generic;
    using System.Diagnostics;
    using Properties;

    /// <summary>
    /// プログラムを実行します。
    /// </summary>
    internal sealed class Executer
    {
        private readonly object syncObject = new object();

        private readonly Platform platform;

        private readonly List<string> args = new List<string>();

        private string path = string.Empty;

        private ErrorCodeParser errorCodeParser = null;

        /// <summary>
        /// Executer クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="platform">プラットフォームです。</param>
        internal Executer(Platform platform)
        {
            this.platform = platform;
        }

        /// <summary>
        /// 開発機を指定します。
        /// </summary>
        internal string Target { private get; set; } = string.Empty;

        /// <summary>
        /// 設定マネージャエンジンを駆動します。
        /// </summary>
        /// <param name="enginePath">設定マネージャエンジンのファイルパスです。</param>
        /// <param name="args">設定マネージャエンジンの引数です。</param>
        internal void Ignite(string enginePath, string[] args)
        {
            switch (this.platform)
            {
                case Platform.Horizon:
                    MakeCommandForHorizon(enginePath, args);
                    break;

                case Platform.Windows:
                default:
                    MakeCommandForWindows(enginePath, args);
                    break;
            }

            using (var process = new Process())
            {
                process.StartInfo = new ProcessStartInfo()
                {
                    FileName = this.path,
                    Arguments = string.Join(" ", this.args),
                    UseShellExecute = false,
                    CreateNoWindow = true,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true
                };

                var handler = new DataReceivedEventHandler(this.ReceiveOutput);

                process.OutputDataReceived += handler;

                process.ErrorDataReceived += handler;

                this.errorCodeParser = new ErrorCodeParser();

                process.Start();

                process.BeginOutputReadLine();

                process.BeginErrorReadLine();

                process.WaitForExit();

                if (this.errorCodeParser.Detects)
                {
                    throw new SettingsManagerException(
                        this.errorCodeParser.Message);
                }

                if (process.ExitCode != 0)
                {
                    throw new SettingsManagerException(
                        string.Format(
                            Resources.ErrorExitCode, process.ExitCode));
                }
            }
        }

        private void MakeCommandForWindows(string enginePath, string[] args)
        {
            this.path = enginePath;

            this.args.Clear();

            this.args.AddRange(args);
        }

        private void MakeCommandForHorizon(string enginePath, string[] args)
        {
            this.path = ResourceManager.RunOnTargetPath;

            this.args.Clear();

            if (!string.IsNullOrEmpty(this.Target))
            {
                this.args.AddRange(new[] { "--target", this.Target });
            }

            var failurePattrens = new[]
            {
                @"\bAssertion Failure:",
                @"\bPrecondition not met:",
                @"\bUnexpected Default:",
                @"\bAbort:",
                @"\bUserException handler is called at pid=",
            };

            foreach (string pattern in failurePattrens)
            {
                this.args.AddRange(
                    new[] { "--pattern-failure-exit", $"\"{pattern}\"" });
            }

            this.args.Add($"\"{enginePath}\"");

            if (args.Length > 0)
            {
                this.args.Add("--");

                this.args.AddRange(args);
            }
        }

        private void ReceiveOutput(object sender, DataReceivedEventArgs args)
        {
            if (string.IsNullOrWhiteSpace(args.Data))
            {
                return;
            }

            lock (this.syncObject)
            {
                this.errorCodeParser.Parse(args.Data);
            }
        }
    }
}
