﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using Nintendo.Foundation.IO;
using System.Xml;
using System.Xml.Linq;
using System.Diagnostics;
using MakeInitialImage;
using System.Text.RegularExpressions;

namespace MakeSystemUpdateMeta
{
    public class MakeSystemUpdateMetaArguments
    {
        [CommandLineOption('o', "output", Description = " ", IsRequired = true)]
        public string OutputXmlPath { get; set; }

        [CommandLineOption('i', "input", Description = " ")]
        public string[] InputNspList { get; set; }

        [CommandLineOption("input-directory", DefaultValue = null, Description = " ")]
        public string InputNspDirectory { get; set; }

        [CommandLineOption("system-meta-id", Description = " ", IsRequired = true)]
        public string Id { get; set; }

        [CommandLineOption("system-meta-version", Description = " ", IsRequired = true)]
        public string Version { get; set; }

        [CommandLineOption("input-text-file", DefaultValue = null, Description = " ")]
        public string InputTextFilePath { get; set; }
    }

    class Program
    {
        static void Main(string[] args)
        {
#if !DEBUG
            try
            {
#endif
            MakeSystemUpdateMetaArguments parameters = new MakeSystemUpdateMetaArguments();
            if (CommandLineParser.Default.ParseArgs<MakeSystemUpdateMetaArguments>(args, out parameters))
            {
                ValidateParameters(parameters);

                var inputNsps = GetInputNspFileInfoList(parameters);

                using (var streamWriter = MakeTextWriter(parameters.OutputXmlPath))
                {
                    MakeSystemUpdateMeta(streamWriter, inputNsps, parameters.Id, parameters.Version);
                }
            }
            else
            {
                return;
            }
#if !DEBUG
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("[Error] {0}", exception.Message);
                Console.Error.WriteLine("{0}", exception.StackTrace);
                Environment.Exit(1);
            }
#endif
        }

        private static void ValidateParameters(MakeSystemUpdateMetaArguments parameters)
        {
            if (!Regex.IsMatch(parameters.Version, "^[0-9]+$"))
            {
                throw new Exception(string.Format("Invalid system meta version: {0}", parameters.Version));
            }

            if (!Regex.IsMatch(parameters.Id, "^0x([0-9a-fA-F]){16}$"))
            {
                throw new Exception(string.Format("Invalid system meta id: {0}", parameters.Id));
            }

            // そもそもインプットファイル用の引数を指定しているかの確認
            if ((parameters.InputNspList == null || parameters.InputNspList.Length == 0)
                && parameters.InputNspDirectory == null
                && parameters.InputTextFilePath == null)
            {
                throw new Exception("No input argument (--input, --input-directory, --input-text-file)");
            }

            // ディレクトリ(--input-directory)指定された場合のチェック処理
            if (parameters.InputNspDirectory != null)
            {
                // 指定パスの存在チェック
                if (Directory.Exists(parameters.InputNspDirectory) == false)
                {
                    throw new Exception(string.Format("No exist or Not Directory path: --input-directory={0}", parameters.InputNspDirectory));
                }

                // 指定パスに nsp ファイルが含まれているかどうかの簡易チェック
                var nspDirInfo = new DirectoryInfo(parameters.InputNspDirectory);
                if (nspDirInfo.GetFiles("*.nsp").Length == 0)
                {
                    throw new Exception(string.Format("No nsp files in directory: --input-directory={0}", parameters.InputNspDirectory));
                }
            }

            // テキストファイル(--input-text-file)指定された場合のチェック処理
            if (parameters.InputTextFilePath != null)
            {
                // 指定パスの存在チェック
                if (File.Exists(parameters.InputTextFilePath) == false)
                {
                    throw new Exception(string.Format("No exist or Not file path: --input-text-file={0}", parameters.InputTextFilePath));
                }
            }
        }

        private static List<FileInfo> GetInputNspFileInfoList(MakeSystemUpdateMetaArguments parameters)
        {
            // input-text-file オプションで指定された nsp ファイル
            if (parameters.InputTextFilePath != null)
            {
                // ひとまず、パス文字列のリスト(Set)のみをファイルから抽出する
                var filePathSet = new HashSet<string>();
                using (var textFile = new StreamReader(parameters.InputTextFilePath))
                {
                    string line;
                    while ((line = textFile.ReadLine()) != null)
                    {
                        if (Regex.IsMatch(line, ".+\\.nsp") && File.Exists(line))
                        {
                            // nsp ファイル かつ ファイルパスとして存在すればリストに保持
                            // (Setコレクションなので重複文字列は入らないはず)
                            filePathSet.Add(line);
                        }
                    }
                }

                if (filePathSet.Count == 0)
                {
                    // 念のため有効なファイルパスが見つからなかった場合は例外を出す
                    throw new Exception(string.Format("No available file path in text: --input-text-file={0}", parameters.InputTextFilePath));
                }

                // このオプションは特別に --input や --input-directory オプションの指定を無視する形とする
                // List<FileInfo> 形式で返す
                return (from path in filePathSet select new FileInfo(path)).ToList();
            }

            // --input オプションで指定された nsp ファイル
            var nspFileInfoList = new List<FileInfo>();
            if (parameters.InputNspList != null)
            {
                nspFileInfoList = (from inputPath in parameters.InputNspList select new FileInfo(inputPath)).ToList();
            }

            // --input-directory オプションで指定された nsp ファイル
            var dirNspFileInfoList = new List<FileInfo>();
            if (parameters.InputNspDirectory != null)
            {
                // ディレクトリ内部の nsp ファイル指定を追加する
                var nspDirInfo = new DirectoryInfo(parameters.InputNspDirectory);
                foreach (var nspFileInfo in nspDirInfo.GetFiles("*.nsp"))
                {
                    bool isAddList = true;
                    foreach (var info in nspFileInfoList)
                    {
                        // 重複指定があればリストに追加しない
                        if (info.FullName == nspFileInfo.FullName)
                        {
                            isAddList = false;
                            break;
                        }
                    }

                    if (isAddList)
                    {
                        dirNspFileInfoList.Add(nspFileInfo);
                    }
                }
            }

            // 2つのオプションで指定されたリストを連結して返す
            return nspFileInfoList.Concat(dirNspFileInfoList).ToList();
        }

        private static StreamWriter MakeTextWriter(string pathOrStdout)
        {
            if (pathOrStdout == "-")
            {
                return new StreamWriter(Console.OpenStandardOutput());
            }
            else
            {
                return new FileInfo(pathOrStdout).CreateText();
            }
        }

        private static string MakeXmlText(XElement xml)
        {
            var settings = new XmlWriterSettings()
            {
                Encoding = Encoding.UTF8,
                Indent = true,
                IndentChars = "  "
            };

            using (var stream = new MemoryStream())
            {
                using (var writer = XmlWriter.Create(stream, settings))
                {
                    xml.Save(writer);
                }

                stream.Position = 0;

                using (var reader = new System.IO.StreamReader(stream))
                {
                    return reader.ReadToEnd();
                }
            }
        }

        private static XElement GetContentMetaFromNsp(FileInfo nsp, string systemMetaId)
        {
            using (var tempHolder = new TemporaryFileHolder("MakeSystemUpdateMeta"))
            {
                var extractedDirectory = tempHolder.CreateTemporaryDirectory("ExtractedNsp");

                ExtractNsp(extractedDirectory, nsp);

                var filename = FindSingleContentMetaXml(extractedDirectory);

                XElement contentMetaXml = XElement.Load(filename);

                // --system-meta-id で指定された ID と同じ ID を持つ nsp ファイルが指定された場合はツールの実行エラーとする
                if (contentMetaXml.Element("Id").Value == systemMetaId)
                {
                    throw new InvalidDataException(string.Format("Include Invalid Nsp File\nPath=\n{0}", nsp.FullName));
                }

                return new XElement("ContentMeta",
                            contentMetaXml.Element("Type"),
                            contentMetaXml.Element("Id"),
                            contentMetaXml.Element("Version"),
                            contentMetaXml.Elements("ContentMetaAttribute"));
            }
        }

        private static string FindSingleContentMetaXml(DirectoryInfo extractedDirectory)
        {
            return (from fileInfo in extractedDirectory.GetFiles() where fileInfo.FullName.EndsWith(".cnmt.xml") select fileInfo.FullName).Single();
        }

        private static void ExtractNsp(DirectoryInfo extractedDirectory, FileInfo nsp)
        {
            string arguments = string.Empty;

            arguments += "extractnsp ";
            arguments += "-o " + "\"" + extractedDirectory.FullName + "\"" + " ";
            arguments += "\"" + nsp.FullName + "\"";

            var toolPath = FindToolPath(
                "AuthoringTool",
                "AuthoringTool/AuthoringTool.exe",
                "AuthoringTool/AuthoringTool.exe");

            using (var process = Process.Start(new ProcessStartInfo()
            {
                FileName = toolPath.FullName,
                Arguments = arguments,
                CreateNoWindow = true,
                UseShellExecute = false,
                RedirectStandardError = true,
                RedirectStandardOutput = true
            }))
            {
                process.OutputDataReceived += OutputDataReceived;
                process.ErrorDataReceived += ErrorDataReceived;
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
                process.WaitForExit();

                if (process.ExitCode != 0)
                {
                    throw new InvalidDataException(string.Format("Failed to extract nsp. Arguments={0}", string.Join(",", arguments)));
                }
            }
        }

        private static void ErrorDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data != string.Empty && e.Data != null)
            {
                Console.Error.WriteLine(e.Data);
            }
        }

        private static void OutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (e.Data != string.Empty && e.Data != null)
            {
                Console.WriteLine(e.Data);
            }
        }

        private static void MakeSystemUpdateMeta(StreamWriter outputStream, List<FileInfo> inputNsps, string id, string version)
        {
            var xml = new XElement("Meta",
                new XElement("SystemUpdate",
                    new XElement("ContentMeta",
                        new XElement("Type", new XText("SystemUpdate")),
                        new XElement("Id", new XText(id)),
                        new XElement("Version", new XText(version)),
                        from nsp in inputNsps select GetContentMetaFromNsp(nsp, id)
                    )
                )
            );

            outputStream.WriteLine(MakeXmlText(xml));
        }

        public static FileInfo FindToolPath(string toolName, string relativePath = null, string rootRelativePath = null, string specifiedTool = null)
        {
            var currentApplicationPosition = System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);

            // specifiedPosition
            if (specifiedTool != null)
            {
                if (File.Exists(specifiedTool))
                {
                    return new FileInfo(specifiedTool);
                }
            }

            // samePosition
            {
                var targetApplicationPath = Path.Combine(currentApplicationPosition, toolName);

                if (File.Exists(targetApplicationPath))
                {
                    return new FileInfo(targetApplicationPath);
                }
            }

            // relativePosition
            if (relativePath != null)
            {
                var targetApplicationPath = Path.Combine(currentApplicationPosition, relativePath);

                if (File.Exists(targetApplicationPath))
                {
                    return new FileInfo(targetApplicationPath);
                }
            }

            // rootRelativePath
            if (rootRelativePath != null)
            {
                var targetApplicationPath = Path.Combine(FindSdkRoot(currentApplicationPosition), "Tools", "CommandLineTools", rootRelativePath);

                if (File.Exists(targetApplicationPath))
                {
                    return new FileInfo(targetApplicationPath);
                }
            }

            throw new FileNotFoundException(string.Format("{0} is not found.", toolName));
        }

        private static string FindSdkRoot(string path)
        {
            string startPath = path;
            string currentDirectory = startPath;
            string rootDirectory = Path.GetPathRoot(currentDirectory);

            while (rootDirectory != currentDirectory)
            {
                string sigloRootMarkFilePath = System.IO.Path.Combine(currentDirectory, SdkRootMarkFileName);
                if (System.IO.File.Exists(sigloRootMarkFilePath))
                {
                    return currentDirectory;
                }

                currentDirectory = System.IO.Path.GetFullPath(System.IO.Path.Combine(currentDirectory, ".."));
            }

            throw new InvalidDataException(string.Format("Not found NintendoSdkRootMark\nPath=\n{0}", startPath));
        }

        private const string SdkRootMarkFileName = "NintendoSdkRootMark";
    }
}
