﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using CommandUtility;

namespace MakeQspiBootImage
{
    public class RecoveryWriterParameter
    {
        private const long SectorSize = 512;
        private const long FirstOffset = 0x15000;
        private const long ParameterSize = 4 * 8;
        private const long ParameterRegionSize = 3 * 1024;
        private const long QspiSize = 16 * 1024 * 1024;
        private const long AlignmentOfMakeFatImage = 1 * 1024 * 1024;

        private long BctOffset;
        private long BctSize;
        private long BootLoaderOffset;
        private long BootLoaderSize;
        private long Package2Offset;
        private long Package2Size;
        public long SystemPartitionOffset { get; private set; }
        public long SystemPartitionSize { get; private set; }

        public RecoveryWriterParameter(FileInfo recoveryWriterProgram, FileInfo gptPartition, FileInfo package1, FileInfo package2)
        {
            BctOffset = FirstOffset;
            BctSize = 0x100000;
            BootLoaderOffset = BctOffset + BctSize;
            BootLoaderSize = AlignSector(package1.Length - BctSize);
            Package2Offset = BootLoaderOffset + BootLoaderSize;
            Package2Size = AlignSector(package2.Length);
            SystemPartitionOffset = BinaryUtility.AlignUp(Package2Offset + Package2Size, AlignmentOfMakeFatImage);
            SystemPartitionSize = QspiSize - SystemPartitionOffset;
        }

        public RecoveryWriterParameter(FileInfo recoveryWriterProgram, FileInfo gptPartition, FileInfo package1, FileInfo package2, long padding, long systemPartitionSize)
        {
            BctOffset = FirstOffset;
            BctSize = 0x100000;
            BootLoaderOffset = BctOffset + BctSize;
            BootLoaderSize = AlignSector(package1.Length - BctSize);
            Package2Offset = BootLoaderOffset + BootLoaderSize;
            Package2Size = AlignSector(package2.Length);
            SystemPartitionOffset = padding;
            SystemPartitionSize = AlignSector(systemPartitionSize);
        }

        public FileInfo MakeParameterFile(FileInfo outputPath)
        {
            MakeParameterFile(
                outputPath,
                BctOffset, BctSize,
                BootLoaderOffset, BootLoaderSize,
                Package2Offset, Package2Size,
                SystemPartitionOffset, SystemPartitionSize);

            return outputPath;
        }

        public static void MakeParameterFile(
            FileInfo outputPath,
            long bctOffset, long bctSize,
            long bootLoaderOffset, long bootLoaderSize,
            long package2Offset, long package2Size,
            long systemPartitionOffset, long systemPartitionSize)
        {
            using (var output = outputPath.OpenWrite())
            {
                BinaryUtility.WriteBinary<uint>(output, (uint)bctOffset);
                BinaryUtility.WriteBinary<uint>(output, (uint)bctSize);
                BinaryUtility.WriteBinary<uint>(output, (uint)bootLoaderOffset);
                BinaryUtility.WriteBinary<uint>(output, (uint)bootLoaderSize);
                BinaryUtility.WriteBinary<uint>(output, (uint)package2Offset);
                BinaryUtility.WriteBinary<uint>(output, (uint)package2Size);
                BinaryUtility.WriteBinary<uint>(output, (uint)systemPartitionOffset);
                BinaryUtility.WriteBinary<uint>(output, (uint)systemPartitionSize);

                var paddingSize = ParameterRegionSize - ParameterSize;
                var padding = new byte[paddingSize];
                output.Write(padding, 0, (int)paddingSize);
            }
        }

        private static long AlignSector(long value)
        {
            return BinaryUtility.AlignUp(value, SectorSize);
        }
    }
}
