﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using Nintendo.Foundation.IO;
using System.Text.RegularExpressions;
using CommandUtility;
using System.Threading;
using System.Globalization;

namespace ReplaceFileInNsp
{
    public class ReplaceFileInNspArgument
    {
        [CommandLineOption('i', "input",
            Description = "set input nsp file.",
            IsRequired = true)]
        public string InputFile { get; set; }

        [CommandLineOption("overwrite",
            Description = "overwrite to input file.",
            IsRequired = false)]
        public bool OverWrite { get; set; }

        [CommandLineOption('o', "output",
            Description = "set output nsp path.",
            IsRequired = false)]
        public string OutputFile { get; set; }

        [CommandLineOption("target",
            Description = "set target file needs replace in nsp.",
            IsRequired = true)]
        public string TargetFile { get; set; }

        [CommandLineOption("source",
            Description = "set source file to replace.",
            IsRequired = true)]
        public string SourceFile { get; set; }

        [CommandLineOption("desc",
            Description = "set desc file if input nsp is a Program.",
            IsRequired = false)]
        public string DescFile { get; set; }

    }

    class Program
    {
        static void Main(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);
#if !DEBUG
            try
            {
#endif
            ReplaceFileInNspArgument parameters = new ReplaceFileInNspArgument();
            if (CommandLineParser.Default.ParseArgs<ReplaceFileInNspArgument>(args, out parameters))
            {
                using (var tempHolder = new TemporaryFileHolder("ReplaceFileInNsp"))
                {
                    var inputNsp = FileUtility.MakeExistedFileInfo("Input Nsp", parameters.InputFile);
                    var outputTemp = tempHolder.CreateTemporaryFilePath("replacedNsp");
                    var descFile = String.IsNullOrEmpty(parameters.DescFile) ? null : FileUtility.MakeExistedFileInfo("Desc File", parameters.DescFile);
                    if (IsMultipleReplace(parameters.SourceFile))
                    {
                        var sourceFiles = parameters.SourceFile.Split('|')
                            .Select(i => FileUtility.MakeExistedFileInfo("Source File", i));
                        var targetFiles = parameters.TargetFile.Split('|');
                        if(sourceFiles.Count() != targetFiles.Count())
                        {
                            throw new Exception(string.Format("File number in --target \"{0}\" and in --source \"{1}\" are imcompatible", parameters.TargetFile, parameters.SourceFile));
                        }
                        ReplaceFileInNsp(outputTemp, inputNsp, targetFiles, sourceFiles, tempHolder, descFile);

                    }
                    else
                    {
                        var sourceFile = FileUtility.MakeExistedFileInfo("Source File", parameters.SourceFile);
                        ReplaceFileInNsp(outputTemp, inputNsp, parameters.TargetFile, sourceFile, tempHolder, descFile);
                    }


                    if (parameters.OverWrite)
                    {
                        File.Copy(outputTemp.FullName, parameters.InputFile);
                    }
                    else
                    {
                        File.Copy(outputTemp.FullName, parameters.OutputFile);
                    }
                }
            }
            else
            {
                return;
            }
#if !DEBUG
            }
            catch (Exception exception)
            {
                Console.Error.WriteLine("Error: {0}", exception.Message);
                Console.Error.WriteLine("{0}", exception.StackTrace);
                Environment.Exit(1);
            }
#endif
        }

        private static bool IsMultipleReplace(string arg)
        {
            return arg.Contains("|");
        }

        private static void ReplaceFileInNsp(FileInfo outputNsp, FileInfo inputNsp, string targetFile, FileInfo sourceFile, TemporaryFileHolder tempHolder, FileInfo descFile)
        {
            var outputText = new StringBuilder();

            GetNspInfo(inputNsp, outputText);

            var text = outputText.ToString();
            var match = Regex.Match(text, string.Format("([0-9a-fA-F]+.nca/.*{0})\\s*", targetFile));
            if (match.Success)
            {
                var outputDir = tempHolder.CreateTemporaryDirectory("ReplacedNsp");
                var targetFilePath = match.Groups[1].Value;
                if (descFile != null)
                {
                    ReplaceNspApplication(inputNsp, descFile, outputDir, targetFilePath, sourceFile.FullName);
                }
                else
                {
                    ReplaceNsp(inputNsp, outputDir, targetFilePath, sourceFile.FullName);
                }

                File.Copy(outputDir.EnumerateFiles().Single().FullName, outputNsp.FullName);
            }
            else
            {
                Console.Error.WriteLine("file list:");
                Console.Error.WriteLine(text);
                throw new Exception(string.Format("found no files: {0}", targetFile));
            }
        }

        private static void ReplaceFileInNsp(FileInfo outputNsp, FileInfo inputNsp, string[] targetFile, IEnumerable<FileInfo> sourceFile, TemporaryFileHolder tempHolder, FileInfo descFile)
        {
            var outputText = new StringBuilder();

            GetNspInfo(inputNsp, outputText);

            var text = outputText.ToString();
            var matches = targetFile.Select(i => Regex.Match(text, string.Format("([0-9a-fA-F]+.nca/.*{0})\\s*", i)));
            if(matches.All(i => i.Success))
            {
                var outputDir = tempHolder.CreateTemporaryDirectory("ReplacedNsp");
                var targetFilePaths = string.Join("|", matches.Select(i => i.Groups[1].Value));
                var sourceFilePaths = string.Join("|", sourceFile.Select(i => i.FullName));
                if (descFile != null)
                {
                    ReplaceNspApplication(inputNsp, descFile, outputDir, targetFilePaths, sourceFilePaths);
                }
                else
                {
                    ReplaceNsp(inputNsp, outputDir, targetFilePaths, sourceFilePaths);
                }

                File.Copy(outputDir.EnumerateFiles().Single().FullName, outputNsp.FullName);
            }
            else
            {
                Console.Error.WriteLine("file list:");
                Console.Error.WriteLine(text);
                var notFound = string.Join("\n", targetFile.Where(i => !Regex.IsMatch(text, string.Format("([0-9a-fA-F]+.nca/.*{0})\\s*", i))));
                throw new Exception(string.Format("found no files: {0}", notFound));
            }
        }

        private static void ReplaceNsp(FileInfo inputNsp, DirectoryInfo outputDir, string targetFilePaths, string sourceFilePaths)
        {
            SdkTool.Execute(SdkPath.FindToolPath("AuthoringTool.exe"),
                new string[]
                {
                        "replace",
                        "--keyconfig", Path.Combine(SdkPath.FindSdkRoot(), "Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml"),
                        "-o", outputDir.FullName,
                        inputNsp.FullName,
                        targetFilePaths,
                        sourceFilePaths

                }
            );
        }

        private static void ReplaceNspApplication(FileInfo inputNsp, FileInfo descFile, DirectoryInfo outputDir, string targetFilePaths, string sourceFilePaths)
        {
            SdkTool.Execute(SdkPath.FindToolPath("AuthoringTool.exe"),
                new string[]
                {
                        "replace",
                        "--keyconfig", Path.Combine(SdkPath.FindSdkRoot(), "Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml"),
                        "-o", outputDir.FullName,
                        inputNsp.FullName,
                        targetFilePaths,
                        sourceFilePaths,
                        "--desc", descFile.FullName

                }
            );
        }

        private static void GetNspInfo(FileInfo inputNsp, StringBuilder outputText)
        {
            SdkTool.Execute(SdkPath.FindToolPath("AuthoringTool.exe"),
                new string[]
                {
                    "list",
                    "--keyconfig", Path.Combine(SdkPath.FindSdkRoot(), "Tools/CommandLineTools/AuthoringTool/AuthoringTool.repository.keyconfig.xml"),
                    inputNsp.FullName
                },
                output: outputText
            );
        }
    }
}
