﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;

namespace MakeInitialImage
{
    /// <summary>
    /// Fatイメージの中にあるセーブデータを取り出して、
    /// 別の場所にコピーするプリプロセッサ
    /// <see cref="InstalledFatImage"/> で使用する
    /// </summary>
    public class CopyFilesPreprocessor : IFatImagePreprocessor
    {
        public List<Tuple<string, string>> CopiedFileInfos { get; private set; }

        public CopyFilesPreprocessor(string[] rawParameter)
        {
            this.CopiedFileInfos = MakeCopiedFileInfos(rawParameter);
        }

        public CopyFilesPreprocessor(List<Tuple<string, string>> copiedFileInfos)
        {
            this.CopiedFileInfos = copiedFileInfos;
        }

        private static List<Tuple<string, string>> MakeCopiedFileInfos(string[] copiedFiles)
        {
            if(copiedFiles == null || copiedFiles.Count() <= 0)
            {
                return new List<Tuple<string, string>>();
            }

            return (from rawFileInfo in copiedFiles select MakeCopiedFileInfo(rawFileInfo)).ToList();
        }

        private static Tuple<string, string> MakeCopiedFileInfo(string rawFileInfo)
        {
            var match = Regex.Match(rawFileInfo, "^(.+),(.+)$");

            if (!match.Success)
            {
                throw new Exception($"Invalida argument: {rawFileInfo}");
            }

            return new Tuple<string, string>(match.Groups[1].Value, match.Groups[2].Value);
        }

        public void Process(DirectoryInfo storageDirectory)
        {
            var combined = from infos in CopiedFileInfos
                           select new Tuple<FileInfo, FileInfo>(
                               new FileInfo(Path.Combine(storageDirectory.FullName, infos.Item1)),
                               new FileInfo(Path.Combine(storageDirectory.FullName, infos.Item2)));

            foreach (var target in combined)
            {
                var sourceInfo = target.Item1;
                var destinationInfo = target.Item2;

                if(!sourceInfo.Exists)
                {
                    throw new Exception($"Input file not found: {sourceInfo.FullName}");
                }

                if(destinationInfo.Exists)
                {
                    throw new Exception($"Output path is already exist: {destinationInfo.FullName}");
                }

                sourceInfo.CopyTo(destinationInfo.FullName);

                if(!File.Exists(destinationInfo.FullName))
                {
                    throw new Exception($"Found no output. path={destinationInfo.FullName}");
                }
            }
        }
    }
}
