﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;

namespace MakeInitialImage
{
    public enum PartitionAddressType
    {
        Absolute,
        Relative,
        Keep
    }

    public class RelativePartitionAddress
    {
        public int BasePartitionIndex { get; set; }
        public long Address { get; set; }
    }

    public class AbsolutePartitionAddress
    {
        public long Address { get; set; }
    }

    public class PartitionAddress
    {
        public PartitionAddressType Type { get; set; }
        public RelativePartitionAddress RelativeAddress { get; set; }
        public AbsolutePartitionAddress AbsoluteAddress { get; set; }
        public long Size { get; set; }

        public static PartitionAddress MakeAbsoluteAddress(long address, long partitionSize, long sectorSize)
        {
            if (!Utility.IsAligned(address, sectorSize))
            {
                throw new InvalidDataException(string.Format("Invalid alignment: value = {0}", address));
            }

            return new PartitionAddress()
            {
                Type = PartitionAddressType.Absolute,
                AbsoluteAddress = new AbsolutePartitionAddress()
                {
                    Address = address,
                },
                Size = partitionSize
            };
        }

        public static PartitionAddress MakeRelativeAddress(int targetIndex, long address, long partitionSize, long sectorSize)
        {
            if (!Utility.IsAligned(address, sectorSize))
            {
                throw new InvalidDataException(string.Format("Invalid alignment: value = {0}", address));
            }

            return new PartitionAddress()
            {
                Type = PartitionAddressType.Relative,
                RelativeAddress = new RelativePartitionAddress()
                {
                    BasePartitionIndex = targetIndex,
                    Address = address
                },
                Size = partitionSize
            };
        }

        public static PartitionAddress MakeAsIsAddress(int targetIndex)
        {
            return new PartitionAddress()
            {
                Type = PartitionAddressType.Keep,
                RelativeAddress = new RelativePartitionAddress()
                {
                    BasePartitionIndex = targetIndex,
                    Address = 0
                },
                Size = 0
            };
        }

        public PartitionAddress ForwardAddress(Partition forwardPartition, long sectorSize)
        {
            if (forwardPartition.AllocationType != PartitionAllocationType.Keep)
            {
                if (!Utility.IsAligned(forwardPartition.Size.Bytes, sectorSize))
                {
                    throw new InvalidDataException(string.Format("Invalid alignment: value = {0}", forwardPartition.Size.Bytes));
                }
            }

            switch (Type)
            {
                case PartitionAddressType.Absolute:
                    return new PartitionAddress()
                    {
                        Type = PartitionAddressType.Absolute,
                        AbsoluteAddress = new AbsolutePartitionAddress()
                        {
                            Address = this.AbsoluteAddress.Address + forwardPartition.Size.Bytes
                        }
                    };
                case PartitionAddressType.Relative:
                    return new PartitionAddress()
                    {
                        Type = PartitionAddressType.Relative,
                        RelativeAddress = new RelativePartitionAddress()
                        {
                            BasePartitionIndex = this.RelativeAddress.BasePartitionIndex,
                            Address = this.RelativeAddress.Address + forwardPartition.Size.Bytes
                        }
                    };
                case PartitionAddressType.Keep:
                    return new PartitionAddress()
                    {
                        Type = PartitionAddressType.Relative,
                        RelativeAddress = new RelativePartitionAddress()
                        {
                            BasePartitionIndex = this.RelativeAddress.BasePartitionIndex,
                            Address = 0
                        }
                    };
                default:
                    throw new InvalidDataException();
            }
        }

        public bool IsOverlapped(PartitionAddress nextStartAddress)
        {
            if (nextStartAddress.Type == PartitionAddressType.Absolute && this.Type == PartitionAddressType.Absolute)
            {
                return nextStartAddress.AbsoluteAddress.Address < this.AbsoluteAddress.Address;
            }

            return false;
        }

        public PartitionAddress MakeNextPartitionAddress(Partition partition, int currentIndex, long sectorSize)
        {
            switch (partition.AllocationType)
            {
                case PartitionAllocationType.Fixed:
                    var address = PartitionAddress.MakeAbsoluteAddress(partition.AddressAsNumber, partition.Size.Bytes, sectorSize);
                    if (this.IsOverlapped(address))
                    {
                        throw new InvalidDataException(
                            string.Format(
                                "Address is overlapped.\n" +
                                "    PartitionName =    {0}\n" +
                                "    UsedEndAddress =   0x{1}\n" +
                                "    FailedNewAddress = 0x{2}",
                                partition.Name,
                                Convert.ToString(this.AbsoluteAddress.Address, 16),
                                Convert.ToString(address.AbsoluteAddress.Address, 16)));
                    }
                    return address;

                case PartitionAllocationType.Sequential:
                    switch (this.Type)
                    {
                        case PartitionAddressType.Absolute:
                            return PartitionAddress.MakeAbsoluteAddress(
                                this.AbsoluteAddress.Address,
                                partition.Size.Bytes,
                                sectorSize);
                        case PartitionAddressType.Relative:
                            return PartitionAddress.MakeRelativeAddress(
                                this.RelativeAddress.BasePartitionIndex,
                                this.RelativeAddress.Address,
                                partition.Size.Bytes,
                                sectorSize);
                        default:
                            throw new InvalidDataException();
                    }

                case PartitionAllocationType.Keep:
                    return PartitionAddress.MakeAsIsAddress(currentIndex);

                default:
                    throw new InvalidDataException("AllocationType is invalid. PartitionName=" + partition.Name);
            }
        }

        public static List<PartitionAddress> MakePartitionAddress(StorageDefinition storageDefinition)
        {
            var ret = new List<PartitionAddress>();
            long sectorSize = storageDefinition.SectorSize.Bytes;

            var currentAddress = new PartitionAddress()
            {
                Type = PartitionAddressType.Absolute,
                AbsoluteAddress = new AbsolutePartitionAddress()
                {
                    Address = CalculateGptSize(storageDefinition)
                }
            };

            Utility.Enumerate(
                storageDefinition.Partitions,
                (partition, currentIndex) =>
                {
                    var address = currentAddress.MakeNextPartitionAddress(partition, currentIndex, sectorSize);
                    ret.Add(address);
                    currentAddress = address.ForwardAddress(partition, sectorSize);
                });

            return ret;
        }

        private static long CalculateGptSize(StorageDefinition storageDefinition)
        {
            long ret = 0;
            long sectorSize = storageDefinition.SectorSize.Bytes;

            // ProtectiveMBR
            ret += sectorSize;

            // GptHeader
            ret += sectorSize;

            // GptPartitionEntries
            ret += Utility.AlignUp(Math.Min(storageDefinition.Partitions.Count, GptDefinition.ReservedPartitions) * GptDefinition.DefaultPartitionSize, sectorSize);

            return ret;
        }
    }
}
