﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Threading;
using System.IO;

namespace Nintendo.HtcTools.Htclow
{
    internal sealed class Mux : IDisposable
    {
        // 送信タスクを起床させるためのイベント
        private ManualResetEventSlim m_AddSendPacketEvent = new ManualResetEventSlim(true);

        private IDictionary<Channel, ChannelResource> m_ChannelResources = new Dictionary<Channel, ChannelResource>();

        // 下位通信路の切断を示す CancellationToken
        private CancellationToken m_ConnectionCancel;

        public bool RetransmitEnable { get; set; } = true;

        public Mux(CancellationToken connectionCancel)
        {
            m_ConnectionCancel = connectionCancel;
        }

        /// <summary>
        /// 通信を行うために必要なチャネルのリソースを確保します。
        /// </summary>
        public void Open(Channel channel)
        {
            lock (m_ChannelResources)
            {
                if (m_ChannelResources.ContainsKey(channel))
                {
                    throw new HtclowChannelException(channel, $"Channel ({channel}) is already opened.");
                }

                m_ChannelResources.Add(channel, new ChannelResource() { RetransmitEnable = RetransmitEnable });
            }
        }

        /// <summary>
        /// チャネルのリソースを確保し、ターゲットとの通信を確立します。
        /// この API は、通信が確立するまでブロックします。
        /// </summary>
        public void Connect(Channel channel)
        {
            ChannelResource resource;

            lock (m_ChannelResources)
            {
                if (!m_ChannelResources.ContainsKey(channel))
                {
                    throw new HtclowChannelException(channel, $"Channel ({channel}) is not opened.");
                }

                resource = m_ChannelResources[channel];

                switch (resource.State)
                {
                    case ChannelState.Opened:
                        resource.State = ChannelState.SynWait;
                        break;
                    case ChannelState.SynReceived:
                        resource.State = ChannelState.Established;

                        resource.SendBuffer.AddAckPacket(resource.PacketFactory.MakeSynAckPacket(channel));
                        m_AddSendPacketEvent.Set();
                        break;
                    default:
                        throw new HtclowChannelException(channel, $"Unpredicted channel state ({resource.State}).");
                }
            }

            try
            {
                resource.Established.Wait(m_ConnectionCancel);
            }
            catch (OperationCanceledException)
            {
                throw new HtclowException("Htclow connection cancelled.");
            }
        }

        /// <summary>
        /// ターゲットにデータを送信します。
        /// </summary>
        /// <param name="channel">チャンネル</param>
        /// <param name="buffer">送信バッファ</param>
        public void Send(Channel channel, byte[] buffer, int offset, int length)
        {
            lock (m_ChannelResources)
            {
                if (!m_ChannelResources.ContainsKey(channel))
                {
                    throw new HtclowChannelException(channel, $"Channel ({channel}) is not opened.");
                }

                var resource = m_ChannelResources[channel];
                if (!(resource.State == ChannelState.Established || resource.State == ChannelState.CloseWait))
                {
                    throw new HtclowChannelException(channel, $"Unpredicted channel state ({resource.State}).");
                }

                if (buffer.Length == 0)
                {
                    return;
                }

                // 最大ボディサイズを考慮して、データを分割しながらパケットを作成
                // TODO: バッファコピーの削減を検討 (Phase 2 以降)
                int totalSendSize = 0;
                while (totalSendSize < length)
                {
                    var bodySize = Math.Min(length - totalSendSize, Packet.MaxBodySize);

                    var packet = resource.PacketFactory.MakeDataPacket(channel, buffer, offset + totalSendSize, bodySize);
                    resource.SendBuffer.AddPacket(packet);

                    totalSendSize += bodySize;
                }

                m_AddSendPacketEvent.Set();
            }
        }

        /// <summary>
        /// ターゲットからデータを受信します。
        /// </summary>
        /// <param name="channel">チャンネル</param>
        /// <param name="buffer">受信バッファ</param>
        /// <param name="offset">受信バッファにおいてデータの格納する開始オフセット</param>
        /// <param name="length">受信するバイト数</param>
        /// <returns>受信したデータの長さ</returns>
        public int Receive(Channel channel, byte[] buffer, int offset, int length)
        {
            if (buffer == null)
            {
                throw new ArgumentNullException();
            }

            if (offset < 0 || buffer.Length < offset)
            {
                throw new ArgumentException();
            }

            if (length < 0 || buffer.Length < offset + length)
            {
                throw new ArgumentException();
            }

            if (length == 0)
            {
                return 0;
            }

            while (true)
            {
                ManualResetEventSlim receiveReady;

                lock (m_ChannelResources)
                {
                    if (!m_ChannelResources.ContainsKey(channel))
                    {
                        throw new HtclowChannelException(channel, $"Channel ({channel}) is not opened.");
                    }

                    var resource = m_ChannelResources[channel];
                    if (resource.State == ChannelState.Opened || resource.State == ChannelState.SynReceived || resource.State == ChannelState.SynWait)
                    {
                        throw new HtclowChannelException(channel, $"Unpredicted channel state ({resource.State}).");
                    }

                    var readableSize = resource.ReceiveBuffer.GetReadableBodySize();

                    if (length <= readableSize)
                    {
                        // 受信バッファのデータが十分なら、ブロックせずに返る
                        resource.ReceiveBuffer.ReadBody(buffer, offset, length);
                        return length;
                    }

                    if (m_ConnectionCancel.IsCancellationRequested ||
                        resource.State == ChannelState.CloseWait ||
                        resource.State == ChannelState.LastAck ||
                        resource.State == ChannelState.Closing ||
                        resource.State == ChannelState.Closed)
                    {
                        // 新しい受信データが来る見込みがない場合
                        if (readableSize > 0)
                        {
                            // バッファにデータがあればそれを返す
                            resource.ReceiveBuffer.ReadBody(buffer, offset, readableSize);
                            return readableSize;
                        }
                        else
                        {
                            // バッファにデータが無ければ例外を投げる
                            throw new HtclowChannelException(channel, "Htclow channel closed.");
                        }
                    }

                    // 新しいデータが来る見込みがある場合はイベント待ち
                    resource.ReceiveBuffer.ReceiveReadyThreshold = length;
                    receiveReady = resource.ReceiveBuffer.ReceiveReady;
                }

                try
                {
                    receiveReady.Wait(m_ConnectionCancel);
                }
                catch (OperationCanceledException)
                {
                    throw new HtclowException("Htclow connection cancelled.");
                }
            }
        }

        /// <summary>
        /// ハンドシェークを実行してターゲットとの通信を終了します。
        /// </summary>
        public void Shutdown(Channel channel)
        {
            try
            {
                ManualResetEventSlim closed;
                ManualResetEventSlim sendCompleted;

                lock (m_ChannelResources)
                {
                    if (!m_ChannelResources.ContainsKey(channel))
                    {
                        throw new HtclowChannelException(channel, $"Channel ({channel}) is not opened.");
                    }

                    var resource = m_ChannelResources[channel];

                    switch (resource.State)
                    {
                        case ChannelState.Established:
                            resource.State = ChannelState.FinWait1;
                            break;
                        case ChannelState.CloseWait:
                            resource.State = ChannelState.LastAck;
                            break;
                        default:
                            throw new HtclowChannelException(channel, $"Unpredicted channel state ({resource.State}).");
                    }

                    resource.SendBuffer.AddPacket(resource.PacketFactory.MakeFinPacket(channel));
                    m_AddSendPacketEvent.Set();

                    closed = resource.Closed;
                    sendCompleted = resource.SendCompleted;
                }

                closed.Wait(m_ConnectionCancel);
                sendCompleted.Wait(m_ConnectionCancel);
            }
            catch (Exception e) when (e is HtclowChannelException || e is OperationCanceledException)
            {
                // Shutdown は例外を投げない
                Log.Info($"Exception suppressed in Shutdown(). ({e})");
            }
        }

        /// <summary>
        /// チャネルのリソースを解放します。
        /// </summary>
        public void Close(Channel channel)
        {
            lock (m_ChannelResources)
            {
                m_ChannelResources.Remove(channel);
            }
        }

        /// <summary>
        /// 受信したパケットを処理する、Worker 向けのメソッドです。
        /// </summary>
        /// <param name="packet"></param>
        public void ProcessReceive(Packet packet)
        {
            // 受信したパケットを処理
            switch (packet.PacketType)
            {
                case PacketType.Data:
                    ProcessReceiveData(packet);
                    break;
                case PacketType.DataAck:
                    ProcessReceiveDataAck(packet);
                    break;
                case PacketType.Syn:
                    ProcessReceiveSyn(packet);
                    break;
                case PacketType.SynAck:
                    // パケットを破棄
                    Log.IgnorePacket(packet);
                    break;
                case PacketType.Fin:
                    ProcessReceiveFin(packet);
                    break;
                case PacketType.FinAck:
                    ProcessReceiveFinAck(packet);
                    break;
                default:
                    // パケットを破棄
                    Log.IgnorePacket(packet);
                    break;
            }
        }

        private void ProcessReceiveData(Packet packet)
        {
            try
            {
                lock (m_ChannelResources)
                {
                    if (!m_ChannelResources.ContainsKey(packet.Channel))
                    {
                        // パケットを破棄
                        Log.IgnorePacket(packet);
                        return;
                    }

                    var resource = m_ChannelResources[packet.Channel];
                    resource.CheckPacketVersion(packet);
                    resource.ReceiveBuffer.AddPacket(packet);

                    var ack = resource.PacketFactory.MakeDataAckPacket(packet.Channel, packet.SequenceId);
                    resource.SendBuffer.AddAckPacket(ack);
                    m_AddSendPacketEvent.Set();
                }
            }
            catch (ChannelSequenceIdException e)
            {
                // パケットを破棄
                Log.IgnorePacket(packet, e);
            }
        }

        private void ProcessReceiveDataAck(Packet packet)
        {
            lock (m_ChannelResources)
            {
                var resource = m_ChannelResources[packet.Channel];
                resource.CheckPacketVersion(packet);

                resource.SendBuffer.RemovePacket(packet.SequenceId);
            }
        }

        private void ProcessReceiveSyn(Packet packet)
        {
            lock (m_ChannelResources)
            {
                if (!m_ChannelResources.ContainsKey(packet.Channel))
                {
                    // パケットを破棄
                    Log.IgnorePacket(packet);
                    return;
                }

                var resource = m_ChannelResources[packet.Channel];

                // セッションで使用するバージョンの決定
                resource.Version = Math.Min(PacketFactory.MaxVersion, packet.Version);

                switch (resource.State)
                {
                    case ChannelState.Opened:
                        resource.State = ChannelState.SynReceived;
                        break;
                    case ChannelState.SynWait:
                        resource.State = ChannelState.Established;

                        resource.SendBuffer.AddAckPacket(resource.PacketFactory.MakeSynAckPacket(packet.Channel));
                        m_AddSendPacketEvent.Set();
                        break;
                    default:
                        // パケットを破棄
                        Log.IgnorePacket(packet);
                        return;
                }
            }
        }

        private void ProcessReceiveFin(Packet packet)
        {
            lock (m_ChannelResources)
            {
                if (!m_ChannelResources.ContainsKey(packet.Channel))
                {
                    // パケットを破棄
                    Log.IgnorePacket(packet);
                    return;
                }

                var resource = m_ChannelResources[packet.Channel];
                resource.CheckPacketVersion(packet);

                switch (resource.State)
                {
                    case ChannelState.Established:
                        resource.State = ChannelState.CloseWait;
                        break;
                    case ChannelState.FinWait1:
                        resource.State = ChannelState.Closing;
                        break;
                    case ChannelState.FinWait2:
                        resource.State = ChannelState.Closed;
                        break;
                    default:
                        // パケットを破棄
                        Log.IgnorePacket(packet);
                        return;
                }

                resource.SendBuffer.AddAckPacket(resource.PacketFactory.MakeFinAckPacket(packet.Channel, packet.SequenceId));
                m_AddSendPacketEvent.Set();
            }
        }

        private void ProcessReceiveFinAck(Packet packet)
        {
            lock (m_ChannelResources)
            {
                if (!m_ChannelResources.ContainsKey(packet.Channel))
                {
                    // パケットを破棄
                    Log.IgnorePacket(packet);
                    return;
                }

                var resource = m_ChannelResources[packet.Channel];
                resource.CheckPacketVersion(packet);

                switch (resource.State)
                {
                    case ChannelState.FinWait1:
                        resource.State = ChannelState.FinWait2;
                        break;
                    case ChannelState.Closing:
                    case ChannelState.LastAck:
                        resource.State = ChannelState.Closed;
                        break;
                    default:
                        // パケットを破棄
                        Log.IgnorePacket(packet);
                        return;
                }

                resource.SendBuffer.RemovePacket(packet.SequenceId);
            }
        }

        public Packet QuerySendPacket()
        {
            while (true)
            {
                m_AddSendPacketEvent.Reset();

                lock (m_ChannelResources)
                {
                    // 全チャンネルの SendBuffer を調べて次に送るべきパケットを探す
                    // TODO: 最適な送信順序の検討 (ex. ack 優先) (Phase 2)
                    foreach (var resource in m_ChannelResources.Values)
                    {
                        var packet = resource.SendBuffer.GetNextPacket();
                        if (packet != null)
                        {
                            return packet;
                        }
                    }
                }

                // TODO: 最適なタイムアウト時間を計算 (Phase 2)
                m_AddSendPacketEvent.Wait(Packet.RetransmitTimeoutMs, m_ConnectionCancel);
            }
        }

        public void Dispose()
        {
            m_ChannelResources.Clear();

            m_AddSendPacketEvent.Dispose();
        }
    }
}
