﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.h"
#include "HotBridgeAPI.h"
#include "CrossBar.h"
#include "ConfigDb.h"

#include "ConsolePrintUtil.h"
#include "BridgeCallbackHandler.h"
#include "BridgeUtil.h"

using namespace tics;

static const int DEFAULT_TIMEOUT = 15000;
static const int WRITE_PARAM_TIMEOUT = 35000; // Writing to entire EEPROM takes about 25 seconds.
static const int WRITE_FIRMWARE_TIMEOUT = 30000; // Writing FW to NAND taks about 15 seconds.


void CrossBarThreadFunc( void* ctx )
{
    ((tics::CrossBar*)ctx)->Run();
}

BridgeUtil::BridgeUtil()
{
    m_BridgeHandle = NULL;
    m_bTakenOver = false;
    m_CrossBar = NULL;

}

BridgeUtil::~BridgeUtil()
{
    if ( m_bTakenOver == true )
    {
        Release();
    }
}

bool BridgeUtil::Enumerate( int timeout )
{
    BridgeEnumCallbackHandler enumCallbackHandler;
    BridgeEnumerate( AliveBridges, timeout, &enumCallbackHandler, BridgeEnumCallbackHandler::BridgeEnumCallback );
    return true;
}

bool BridgeUtil::Find( const tics::portability::stl::string& name, int timeout, tics::portability::stl::string* ip, int* port )
{
    BridgeEnumCallbackHandler enumCallbackHandler;
    BridgeEnumerate( AliveBridges, timeout, &enumCallbackHandler, BridgeEnumCallbackHandler::BridgeEnumCallback );

    for( unsigned int i = 0; i < enumCallbackHandler.discoveredBridgeNames.size(); i++ )
    {
        if( enumCallbackHandler.discoveredBridgeNames[i].compare( name ) == 0 )
        {
            if( ip != NULL )
            {
                *ip = enumCallbackHandler.discoveredBridgeIPs[i];
                *port = enumCallbackHandler.discoveredBridgePorts[i];
            }
            return true;
        }
    }

    return false;
}

bool BridgeUtil::Takeover( const tics::portability::stl::string& ipPort )
{
    BridgeCallbackHandlerBase takeoverHandler;
    takeoverHandler.Reset();

    m_BridgeHandle = BridgeTakeoverByIP( ipPort.c_str(), true, 0, DEFAULT_TIMEOUT, &takeoverHandler, BridgeCallbackHandlerBase::OnOperationComplete );

    m_CrossBar = GetBridgeCrossBar( m_BridgeHandle );
    m_CrossBar->create_thread( CrossBarThreadFunc, m_CrossBar );

    bool completed = takeoverHandler.GetAutoResetEvent()->WaitOne( DEFAULT_TIMEOUT );
    if( !completed )
    {
        PRINT_ERROR_MSG( "Timeout at BridgeTakeoverByIP.\n" );
        return false;
    }
    int takeoverResult = takeoverHandler.GetLastOperationResult();
    if( takeoverResult != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Bridge takeover callback returned with operation result: %08X\n", takeoverHandler.GetLastOperationResult() );
        return false;
    }

    m_bTakenOver = true;

    return true;
}

bool BridgeUtil::Release()
{
    if( m_BridgeHandle == NULL )
    {
        return true;
    }

    bool released = false;

    BridgeCallbackHandlerBase releaseHandler;
    int result = BridgeRelease( m_BridgeHandle, false, &releaseHandler, BridgeCallbackHandlerBase::OnOperationComplete );
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Failed to call BridgeRelease.\n", result );
    }
    else
    {
        bool completed = releaseHandler.GetAutoResetEvent()->WaitOne( DEFAULT_TIMEOUT );
        if( !completed )
        {
            PRINT_ERROR_MSG( "Timeout at BridgeRelease.\n" );
        }
        else if( releaseHandler.GetLastOperationResult() != BRIDGE_ERROR_SUCCESS )
        {
            PRINT_ERROR_MSG( "BridgeRelease Failed: %d.\n", result );
        }
        else
        {
            released = true;
        }
    }

    if ( m_CrossBar != nullptr )
    {
        m_CrossBar->BeginShutdown( CrossBar::ExitDone );
    }

    BridgeHandleFree( m_BridgeHandle );
    m_BridgeHandle = NULL;
    m_CrossBar = NULL;
    m_bTakenOver = false;
    return released;
}

bool BridgeUtil::SetBridgeConfig( const tics::portability::stl::string& ipPort, const tics::portability::stl::string& attr, const tics::portability::stl::string& key )
{
    if( !m_bTakenOver )
    {
        if( ipPort.empty() )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
        else if( !Takeover( ipPort ) )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
    }

    BridgeCallbackHandlerBase setConfigsectionCallbackHandler;
    setConfigsectionCallbackHandler.Reset();

    int result = BridgeConfigSectionAttrSet( m_BridgeHandle, ConfigDb::SECTION_NAME_BRIDGE_CONFIG, attr.c_str(), key.c_str(), true, &setConfigsectionCallbackHandler, BridgeCallbackHandlerBase::OnOperationComplete );
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Failed to call BridgeConfigSectionContentsGet.\n" , result );
        return false;
    }

    bool completed = setConfigsectionCallbackHandler.GetAutoResetEvent()->WaitOne( DEFAULT_TIMEOUT );
    if( !completed )
    {
        PRINT_ERROR_MSG( "Timeout at setConfigsectionCallbackHandler.\n" );
        return false;
    }

    result = setConfigsectionCallbackHandler.GetLastOperationResult();
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "BridgeConfigSectionAttrSet Failed: %d.\n", result );
        return false;
    }

    return true;
}

bool BridgeUtil::GetBridgeConfig( const tics::portability::stl::string& ipPort, tics::portability::stl::string* configText )
{
    if( !m_bTakenOver )
    {
        if( ipPort.empty() )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
        else if( !Takeover( ipPort ) )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
    }

    GetConfigSectionCallbackHandler configSectionContentsGetCallbackHandler;
    configSectionContentsGetCallbackHandler.Reset();

    int result = BridgeConfigSectionContentsGet( m_BridgeHandle, "BridgeConfig", &configSectionContentsGetCallbackHandler, GetConfigSectionCallbackHandler::OnGetConfigSection );
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Failed to call BridgeConfigSectionContentsGet.\n" , result );
        return false;
    }

    bool completed = configSectionContentsGetCallbackHandler.GetAutoResetEvent()->WaitOne( DEFAULT_TIMEOUT );
    if( !completed )
    {
        PRINT_ERROR_MSG( "Timeout at BridgeConfigSectionContentsGet.\n" );
        return false;
    }

    result = configSectionContentsGetCallbackHandler.GetLastOperationResult();
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "BridgeConfigSectionContentsGet Failed: %d.\n", result );
        return false;
    }
    if( configText != NULL )
    {
        // [Note] In GetConfigSectionCallbackHandler::OnGetConfigSection, only configText is stored as operationResources.
        // BridgeCallbackInfo* info = static_cast<BridgeCallbackInfo*>( configSectionContentsGetCallbackHandler.GetLastOperationResources() );
        // *configText = const_cast<char*>( info->zstrInfo );

        *configText = static_cast<char*>( configSectionContentsGetCallbackHandler.GetLastOperationResources() );
    }

    return true;
}

bool BridgeUtil::UpdateBridgeFirmware( const tics::portability::stl::string& ipPort, const tics::portability::stl::string& filePath )
{
    if( !m_bTakenOver )
    {
        if( ipPort.empty() )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
        else if( !Takeover( ipPort ) )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
    }

    BridgeCallbackHandlerBase callbackHandler;
    callbackHandler.Reset();
    int result = BridgeFirmwareUpdate( m_BridgeHandle, filePath.c_str(), &callbackHandler, BridgeCallbackHandlerBase::OnOperationComplete );
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Failed to call BridgeHostBridgeFirmwareUpload.\n" , result );
        return false;
    }

    bool completed = callbackHandler.GetAutoResetEvent()->WaitOne( WRITE_FIRMWARE_TIMEOUT );
    if( !completed )
    {
        PRINT_ERROR_MSG( "Timeout at BridgeHostBridgeFirmwareUpload.\n" );
        return false;
    }

    result = callbackHandler.GetLastOperationResult();
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "BridgeHostBridgeFirmwareUpload Failed: %d.\n", result );
        return false;
    }
    return true;
}

bool BridgeUtil::GetBridgeFirmwareVersion( const tics::portability::stl::string& ipPort, unsigned int* packageVersion)
{
    if( !m_bTakenOver )
    {
        if( ipPort.empty() )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
        else if( !Takeover( ipPort ) )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
    }

    GetBridgeFirmwareVersionCallbackHandler callbackHandler;
    callbackHandler.Reset();

    int result = BridgeFirmwareGetVersion( m_BridgeHandle, &callbackHandler, GetBridgeFirmwareVersionCallbackHandler::OnGetBridgeFirmwareVersion );
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Failed to call BridgeFirmwareGetVersion.\n" , result );
        return false;
    }

    bool completed = callbackHandler.GetAutoResetEvent()->WaitOne( DEFAULT_TIMEOUT );
    if( !completed )
    {
        PRINT_ERROR_MSG( "Timeout at BridgeFirmwareGetVersion.\n" );
        return false;
    }

    result = callbackHandler.GetLastOperationResult();
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "BridgeHostBridgeFirmwareUpload Failed: %d.\n", result );
        return false;
    }

    if(packageVersion != NULL)
    {
        *packageVersion = ((BridgeFirmwareVersion*)callbackHandler.GetLastOperationResources())->packageVersion;
    }

    return true;
}

bool BridgeUtil::ControlTargetPower( const tics::portability::stl::string& ipPort, BridgeTargetPowerState powerState, bool force )
{
    if( !m_bTakenOver )
    {
        if( ipPort.empty() )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
        else if( !Takeover( ipPort ) )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
    }

    BridgeTargetStateCallbackHandler callbackHandler;
    callbackHandler.Reset();

    int result = BridgeTargetPower( m_BridgeHandle, powerState, force, &callbackHandler, BridgeTargetStateCallbackHandler::OnGetTargetStatus );
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Failed to call BridgeTargetPower.\n" , result );
        return false;
    }

    bool completed = callbackHandler.GetAutoResetEvent()->WaitOne( DEFAULT_TIMEOUT );
    if( !completed )
    {
        PRINT_ERROR_MSG( "Timeout at BridgeTargetPower.\n" );
        return false;
    }

    result = callbackHandler.GetLastOperationResult();
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "BridgeTargetPower Failed: %d.\n", result );
        return false;
    }
    return true;
}

bool BridgeUtil::SetTargetStatus( const tics::portability::stl::string& ipPort, ETargetStatusFlag flag, bool state )
{
    if( !m_bTakenOver )
    {
        if( ipPort.empty() )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
        else if( !Takeover( ipPort ) )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
    }

    BridgeTargetStateCallbackHandler callbackHandler;
    callbackHandler.Reset();

    int result = BridgeSetTargetStatus( m_BridgeHandle, flag, state, &callbackHandler, BridgeTargetStateCallbackHandler::OnGetTargetStatus );
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Failed to call BridgeSetTargetStatus.\n" , result );
        return false;
    }

    bool completed = callbackHandler.GetAutoResetEvent()->WaitOne( DEFAULT_TIMEOUT );
    if( !completed )
    {
        PRINT_ERROR_MSG( "Timeout at BridgeTargetPower.\n" );
        return false;
    }

    result = callbackHandler.GetLastOperationResult();
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "BridgeTargetPower Failed: %d.\n", result );
        return false;
    }
    return true;
}

bool BridgeUtil::GetTargetStatus( const tics::portability::stl::string& ipPort)
{
    if( !m_bTakenOver )
    {
        if( ipPort.empty() )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
        else if( !Takeover( ipPort ) )
        {
            PRINT_ERROR_MSG("Failed to Takeover Bridge %s.\n", ipPort.c_str() );
            return false;
        }
    }

    BridgeTargetStateCallbackHandler callbackHandler;
    callbackHandler.Reset();

    // Calling BridgeSetTargetStatus with EFlag_None returns current status without changing the current status (I believe)
    int result = BridgeSetTargetStatus( m_BridgeHandle, EFlag_None, false, &callbackHandler, BridgeTargetStateCallbackHandler::OnGetTargetStatus );
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "Failed to call BridgeSetTargetStatus.\n" , result );
        return false;
    }

    bool completed = callbackHandler.GetAutoResetEvent()->WaitOne( DEFAULT_TIMEOUT );
    if( !completed )
    {
        PRINT_ERROR_MSG( "Timeout at BridgeTargetPower.\n" );
        return false;
    }

    result = callbackHandler.GetLastOperationResult();
    if( result != BRIDGE_ERROR_SUCCESS )
    {
        PRINT_ERROR_MSG( "BridgeTargetPower Failed: %d.\n", result );
        return false;
    }
    return true;
}
