﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.IO;

namespace ErrorMessageDatabaseConverter
{
    using System.Text.RegularExpressions;
    using static ErrorMessageDatabaseConverter.InputXmlDataModel;

    /// <summary>
    /// システムデータのデータ構造を定義します。
    /// </summary>
    public class SystemDataModel
    {
        public class ErrorCode : IEquatable<ErrorCode>
        {
            public static readonly ErrorCode InvalidErrorCode = new ErrorCode(0, 0);

            private string m_Text;
            public string Text
            {
                get
                {
                    return m_Text;
                }
                set
                {
                    if (m_Text != value)
                    {
                        var tmp = value.Split('-');
                        if(tmp.Length != 2)
                        {
                            throw new ArgumentException("ErrorCode text format must be XXXX-XXXX where X must be numeric characters.");
                        }
                        if(!UInt32.TryParse(tmp[0], out m_Category))
                        {
                            throw new ArgumentException("ErrorCode text format must be XXXX-XXXX where X must be numeric characters.");
                        }
                        if(!UInt32.TryParse(tmp[1], out m_Number))
                        {
                            throw new ArgumentException("ErrorCode text format must be XXXX-XXXX where X must be numeric characters.");
                        }

                        m_Text = $"{m_Category:0000}-{m_Number:0000}";
                    }
                }
            }

            private UInt32 m_Category;
            public UInt32 Category
            {
                get
                {
                    return m_Category;
                }
                set
                {
                    m_Category = value;
                    m_Text = $"{m_Category:0000}-{m_Number:0000}";
                }
            }

            private UInt32 m_Number;
            public UInt32 Number
            {
                get
                {
                    return m_Number;
                }
                set
                {
                    m_Number = value;
                    m_Text = $"{m_Category:0000}-{m_Number:0000}";
                }
            }

            public ErrorCode(string errorCode)
            {
                Text = errorCode;
            }

            public ErrorCode(UInt32 category, UInt32 number)
            {
                Category = category;
                Number = number;
            }

            public static implicit operator ErrorCode(string errorCodeText)
            {
                return new ErrorCode(errorCodeText);
            }

            public bool Equals(ErrorCode other)
            {
                return (Category == other.Category) && (Number == other.Number);
            }

            public override int GetHashCode()
            {
                // TORIAEZU : nn::Result ベースの場合 Category (Module) は 9bit, 独自の場合は 10bitなので以下の式で。
                return (int)(Category | (Number << 10));
            }
        }

        /// <summary>
        /// エラーメッセージ毎に保存する、言語間で共通のデータのデータ構造。ボタン数など。
        /// </summary>
        [Serializable, StructLayout(LayoutKind.Sequential, Pack = 1)]
        public class ErrorMessageCommonData
        {
            public const int StructSize = 56;
            public const int ButtonCountMax = 3;
            public const int ServerCodeLength = 8;
            public const int PaddingSize = 3;

            public byte DialogViewButtonNum;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = ButtonCountMax)]
            public byte[] DialogViewButtonAction;
            public byte FullScreenViewButtonNum;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = ButtonCountMax)]
            public byte[] FullScreenViewButtonAction;
            public byte ServerFailure;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = ServerCodeLength)]
            public byte[] ServerCode;

            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 3)]
            public byte[] PADDING;

            public UInt32 DialogViewMessageReferenceCategory;
            public UInt32 DialogViewMessageReferenceNumber;
            public UInt32 DialogViewButtonMessageReferenceCategory;
            public UInt32 DialogViewButtonMessageReferenceNumber;
            public UInt32 FullScreenViewMessageReferenceCategory;
            public UInt32 FullScreenViewMessageReferenceNumber;
            public UInt32 FullScreenViewButtonMessageReferenceCategory;
            public UInt32 FullScreenViewButtonMessageReferenceNumber;

            public UInt32 Attribute;

            public static ErrorMessageCommonData MakeFromErrorElement(ErrorElement ee)
            {
                var ecd = new ErrorMessageCommonData();
                ecd.DialogViewButtonAction = new byte[ErrorMessageCommonData.ButtonCountMax];
                ecd.FullScreenViewButtonAction = new byte[ErrorMessageCommonData.ButtonCountMax];
                ecd.ServerCode = new byte[ErrorMessageCommonData.ServerCodeLength];
                ecd.PADDING = new byte[ErrorMessageCommonData.PaddingSize];

                ecd.DialogViewButtonNum = (byte)ee.DialogViewButtonActions.Count();
                for (int i = 0; i < ecd.DialogViewButtonNum; i++)
                {
                    ecd.DialogViewButtonAction[i] = ee.DialogViewButtonActions[i];
                }

                ecd.FullScreenViewButtonNum = (byte)ee.FullScreenViewButtonActions.Count();
                for (int i = 0; i < ecd.FullScreenViewButtonNum; i++)
                {
                    ecd.FullScreenViewButtonAction[i] = ee.FullScreenViewButtonActions[i];
                }

                ecd.ServerFailure = (byte)(ee.ServerFailure ? 1 : 0);
                if (ee.ServerCode != null)
                {
                    Encoding.UTF8.GetBytes(ee.ServerCode).CopyTo(ecd.ServerCode, 0);
                }

                ecd.DialogViewMessageReferenceCategory = ee.DialogViewMessageReference.Category;
                ecd.DialogViewMessageReferenceNumber = ee.DialogViewMessageReference.Number;
                ecd.DialogViewButtonMessageReferenceCategory = ee.DialogViewButtonMessageReference.Category;
                ecd.DialogViewButtonMessageReferenceNumber = ee.DialogViewButtonMessageReference.Number;
                ecd.FullScreenViewMessageReferenceCategory = ee.FullScreenViewMessageReference.Category;
                ecd.FullScreenViewMessageReferenceNumber = ee.FullScreenViewMessageReference.Number;
                ecd.FullScreenViewButtonMessageReferenceCategory = ee.FullScreenViewButtonMessageReference.Category;
                ecd.FullScreenViewButtonMessageReferenceNumber = ee.FullScreenViewButtonMessageReference.Number;

                ecd.Attribute = ee.Attribute;

                return ecd;
            }

            public byte[] GetAsBytes()
            {
                var size = Marshal.SizeOf(typeof(ErrorMessageCommonData));
                var ptr = Marshal.AllocHGlobal(size);
                Marshal.StructureToPtr(this, ptr, false);
                byte[] bytes = new byte[size];
                Marshal.Copy(ptr, bytes, 0, size);
                Marshal.FreeHGlobal(ptr);

                Debug.Assert(bytes.Length == ErrorMessageCommonData.StructSize);

                return bytes;
            }
        }

        /// <summary>
        /// エラーメッセージ毎に保存し、かつ言語毎に保存するデータ。メッセージの類。
        /// Message[メッセージの種類][言語] = メッセージの内容（タグを変換した結果の byte 配列が入るため、string ではなく byte[]）
        /// </summary>
        public class ErrorMessageLanguageData
        {
            public Dictionary<LanguageElement.MessageElementKey, Dictionary<Language, byte[]>> Message;
            public ErrorMessageLanguageData()
            {
                Message = new Dictionary<LanguageElement.MessageElementKey, Dictionary<Language, byte[]>>();
            }
        }

        /// <summary>
        /// エラーメッセージ毎に保存するデータ
        /// </summary>
        public class ErrorMessageData
        {
            public ErrorMessageCommonData CommonData { get; set; }
            public ErrorMessageLanguageData LangData { get; set; }
        }

        /// <summary>
        /// エラーコードに紐付かず単体で持つメッセージのデータ。
        /// メッセージ指定でエラーを表示する際に、言語に合わせてボタンのメッセージを設定するために使用する。
        /// Message[キー][言語] = メッセージの内容
        /// </summary>

        /// <summary>
        /// システムデータ自身の情報を表すデータ
        /// </summary>
        public class DatabaseInfo
        {
            public const string FileName = "DatabaseInfo";

            public UInt16 MajorVersion { get; set; } = 1;
            public UInt16 MinorVersion { get; set; } = 0;

            public void Write(string rootDir)
            {
                using (var writer = new BinaryWriter(new FileStream(Path.Combine(rootDir, FileName), FileMode.Create)))
                {
                    writer.Write(MajorVersion);
                    writer.Write(MinorVersion);
                }
            }
        }

        /// <summary>
        /// システムデータ全体を表すデータ
        /// </summary>
        public class SystemData
        {
            public DatabaseInfo DatabaseInfo { get; set; }

            public Dictionary<ErrorCode, ErrorMessageData> ErrorMessageData { get; set; }

            /// <summary>
            /// エラーコードに紐付かず単体で持つメッセージのデータ。
            /// メッセージ指定でエラーを表示する際に、言語に合わせてボタンのメッセージを設定するために使用する。
            /// </summary>
            public Dictionary<string, Dictionary<Language, byte[]>> Message { get; set; }

            public SystemData()
            {
                DatabaseInfo = new DatabaseInfo();
                ErrorMessageData = new Dictionary<ErrorCode, SystemDataModel.ErrorMessageData>();
                Message = new Dictionary<string, Dictionary<Language, byte[]>>();
            }

            /// <summary>
            /// AuthroingTool の入力として指定するためのディレクトリを作成します。
            /// </summary>
            /// <param name="rootDir">出力先のルートディレクトリ</param>
            /// <param name="noOverwrite">上書きするかどうか</param>
            /// <param name="showProgress">進捗を出力するかどうか</param>
            /// <returns>成功した場合は true。何らかの理由で失敗した場合は false</returns>
            public bool CreateDirectoryForAuthoring(string rootDir, bool noOverwrite, bool showProgress)
            {
                Util.ProgressPrinter pp = null;
                if (showProgress)
                {
                    Console.WriteLine(">> システムデータ作成用ディレクトリ構造の書き込み開始");
                    pp = new Util.ProgressPrinter(ErrorMessageData.Count, 200);
                }

                try
                {
                    Directory.CreateDirectory(rootDir);
                }
                catch(Exception e)
                {
                    Util.PrintError($"ディレクトリ {rootDir} の作成に失敗しました。処理を中止します。\n{e.ToString()}");
                    return false;
                }

                DatabaseInfo.Write(rootDir);

                // メッセージの情報
                var messageDir = Path.Combine(rootDir, "Messages");
                Directory.CreateDirectory(messageDir);
                foreach(var message in Message)
                {
                    foreach(var m in message.Value)
                    {
                        string messageFilePath = Path.Combine(messageDir, GetSystemLaugnageName(m.Key) + "_" + message.Key);
                        using (var writer = new BinaryWriter(new FileStream(messageFilePath, FileMode.Create)))
                        {
                            writer.Write(m.Value);
                        }
                    }
                }

                int processedErrorCount = 0;

                // エラーメッセージ毎の情報
                foreach (var errData in ErrorMessageData)
                {
                    try
                    {
                        pp?.PrintProgress(processedErrorCount++);

                        string errorCodeDir = CreateErrorCodeDirectory(rootDir, errData.Key.Text);
                        string commonFilePath = Path.Combine(errorCodeDir, "common");

                        if (noOverwrite && File.Exists(commonFilePath))
                        {
                            continue;
                        }

                        using (var writer = new BinaryWriter(new FileStream(commonFilePath, FileMode.Create)))
                        {
                            var bytes = errData.Value.CommonData.GetAsBytes();
                            writer.Write(bytes);
                        }

                        if (errData.Value.LangData != null)
                        {
                            foreach (var propertyEntry in errData.Value.LangData.Message)
                            {
                                foreach (var langEntry in propertyEntry.Value)
                                {
                                    string langFilePath = Path.Combine(errorCodeDir, GetSystemLaugnageName(langEntry.Key) + "_" + s_SystemDataLanguageFileSuffixDictionary[propertyEntry.Key]);
                                    using (BinaryWriter writer = new BinaryWriter(new FileStream(langFilePath, FileMode.Create)))
                                    {
                                        writer.Write(langEntry.Value);
                                    }
                                }
                            }
                        }
                    }
                    catch (Exception exception)
                    {
                        Util.PrintError($"{errData.Key.Text} のシステムデータの書き込みに失敗しました。処理を中止します。\n{exception.ToString()}");
                        return false;
                    }
                }

                if (showProgress)
                {
                    Console.WriteLine(">> システムデータ作成用ディレクトリ構造の書き込み完了");
                }
                return true;
            }

            /// <summary>
            /// "XXXX-YYYY" という形式の errorCode を入力として "$(rootDir)/XXXX/YYYY" というディレクトリを作成しそのパスを返します。
            /// </summary>
            /// <param name="rootDir">ディレクトリのパス</param>
            /// <param name="errorCodeText">エラーコード</param>
            /// <returns>作成したディレクトリのパス</returns>
            private static string CreateErrorCodeDirectory(string rootDir, string errorCodeText)
            {
                string[] s = errorCodeText.Split('-');
                string errorCodeDir = Path.Combine(rootDir, s[0], s[1]);
                Directory.CreateDirectory(errorCodeDir);
                return errorCodeDir;
            }

            private static Dictionary<LanguageElement.MessageElementKey, string> s_SystemDataLanguageFileSuffixDictionary = new Dictionary<LanguageElement.MessageElementKey, string>
            {
                { LanguageElement.MessageElementKey.DlgMsg, "DlgMsg" },
                { LanguageElement.MessageElementKey.DlgBtn, "DlgBtn" },
                { LanguageElement.MessageElementKey.FlvMsg, "FlvMsg" },
                { LanguageElement.MessageElementKey.FlvBtn, "FlvBtn" }
            };

            /// <summary>
            /// 変換元の xml での言語名と SDK（Programs\Eris\Sources\Libraries\settings\detail\settings_SettingsServerByDfc.cpp） で定義された言語名の対応
            /// </summary>
            private static readonly Dictionary<Language, string> s_SystemLanguageNameDictionary = new Dictionary<Language, string>
            {
                { Language.JPja, "ja" },       // Japanese
                { Language.EUen, "en-GB" },    // British English
                { Language.EUfr, "fr" },       // French
                { Language.EUes, "es" },       // Spanish
                { Language.EUde, "de" },       // German
                { Language.EUit, "it" },       // Italian
                { Language.EUnl, "nl" },       // Dutch (Netherlands)
                { Language.EUpt, "pt" },       // Portuguese
                { Language.EUru, "ru" },       // Russian
                { Language.USen, "en-US" },    // American English
                { Language.USfr, "fr-CA" },    // Canadian French
                { Language.USes, "es-419" },   // Latin American Spanish
                { Language.CNzh, "zh-Hans" },  // Simple Chinese (zh-CN でないのは意図的なもの)
                { Language.KRko, "ko" },       // Korean
                { Language.TWzh, "zh-Hant" },  // Traditional Chinese (zh-TW でないのは意図的なもの)
            };

            public static Language GetLanguageFromSystemName(string langName)
            {
                return s_SystemLanguageNameDictionary.Where(kv => kv.Value == langName).First().Key;
            }

            public static string GetSystemLaugnageName(Language lang)
            {
                return s_SystemLanguageNameDictionary[lang];
            }
        }
    }
}
