﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Text.RegularExpressions;

namespace ErrorMessageDatabaseConverter
{
    using static ErrorMessageDatabaseConverter.InputXmlDataModel;
    using static ErrorMessageDatabaseConverter.SystemDataModel;

    /// <summary>
    /// XMLから読み込んだ変換元のデータをシステムデータ用のデータ構造に変換するクラス
    /// </summary>
    public class Converter
    {
        public static System.Text.Encoding OutputEncoding { get; } = System.Text.Encoding.GetEncoding("UTF-16LE");

        private Dictionary<string, ErrorElement> m_ErrorElementDict;

        /// <summary>
        ///数値表現できない参照専用のエラーコード（"001-comm"など）を実体のあるエラーコードにマップするための辞書。
        // 例えば "001-0000" と "001-0001" が両方 "001-comm" を参照していた場合、
        // "001-000"（先に処理された方）に "001-comm" の値を持たせ、"001-comm" の参照先は "001-0000" であることをこの辞書に登録する。
        // 次に処理される "001-0001" はその辞書を見て "001-0000" を参照する
        /// </summary>
        private Dictionary<LanguageElement.MessageElementKey, Dictionary<string, ErrorCode>> m_MessageReferenceDict;

        /// <summary>
        /// message の内容が参照を表す文字列かどうかを調べます。
        /// "$(XXXX-XXXX)" という形の文字列であれば参照とみなします。
        /// </summary>
        /// <param name="referenceErrorCodeText">参照だった場合、参照先のエラーコード。参照でなければ空の文字列。</param>
        /// <param name="message">参照かどうかを調べる文字列</param>
        /// <returns>参照であった場合は true、そうでなければ false</returns>
        public static bool IsReferenceMessage(out string referenceErrorCodeText, string message)
        {
            if(message != null)
            {
                Match m = Regex.Match(message, @"^\$\((.+?-.+?)\)$");
                if (m.Success)
                {
                    referenceErrorCodeText = m.Groups[1].Value;
                    return true;
                }
            }
            referenceErrorCodeText = string.Empty;
            return false;
        }

        /// <summary>
        /// message 中の URL挿入タグを置換した結果の文字列を返します。
        /// URL挿入タグは "$(URL_KEY)" という形で、KEY の箇所が置換先の URL によって変わります。
        /// <param name="message">置換前の文字列</param>
        /// <param name="lang">入力メッセージの言語</param>
        /// <param name="urlElementList">置換先の URL を表す辞書</param>
        /// <returns>URL挿入タグを置換した結果の文字列</returns>
        public static string ReplaceUrlTags(string message, Language lang, IEnumerable<UrlElement> urlElementList)
        {
            if (urlElementList == null || urlElementList.Count() == 0)
            {
                return message;
            }
            // $(URL_XXX) にマッチする箇所を検索し、XXX 部分を Group[1] で取得
            MatchCollection matchCollection = Regex.Matches(message, @"\$\(URL_(.+?)\)");
            StringBuilder replacedString = new StringBuilder(message.Length);
            int currentIndex = 0;
            foreach (Match match in matchCollection)
            {
                // URL挿入タグ前までを出力に Append
                replacedString.Append(message.Substring(currentIndex, match.Index - currentIndex));
                // URL挿入タグの置換結果を出力 Append
                string url = urlElementList.Where(e => e.Id == match.Groups[1].Value).SelectMany(e => e.LanguageElements).Where(l => l.Language == lang).SingleOrDefault()?.Url;
                if (url != null)
                {
                    replacedString.Append(url);
                }
                else
                {
                    Util.PrintWarning($"メッセージ中にURL挿入タグが見つかりましたが、対応するURLが見つかりませんでした。\nキーの値：{match.Groups[1].Value}\n言語：{lang}\nメッセージ：{message}");
                }
                currentIndex = match.Index + match.Length;
            }
            replacedString.Append(message.Substring(currentIndex));
            return replacedString.ToString();
        }

        private class MessageStudioTag
        {
            // Little Endian で決め打ち
            private static readonly byte[] Start = new byte[] { 0x0E, 0x00 };
            private static readonly byte[] End = new byte[] { 0x0F, 0x00 };
            private static readonly byte[] Group = new byte[] { 0x01, 0x00 };
            private static readonly byte[] NoParam = new byte[]{ 0x00, 0x00 };

            public string TagString { get; set; }
            public List<byte> TagBytes { get; set; }

            public static MessageStudioTag MakeStartTag(string tagString, byte tagId)
            {
                MessageStudioTag tag = new MessageStudioTag();
                tag.TagString = tagString;
                tag.TagBytes = new List<byte>();
                tag.TagBytes.AddRange(Start);
                tag.TagBytes.AddRange(Group);
                tag.TagBytes.Add(tagId);
                tag.TagBytes.Add(0x00);
                tag.TagBytes.AddRange(NoParam);
                return tag;
            }

            public static MessageStudioTag MakeStartTag(string tagString, byte tagId, float param)
            {
                MessageStudioTag tag = new MessageStudioTag();
                tag.TagString = tagString;
                tag.TagBytes = new List<byte>();
                tag.TagBytes.AddRange(Start);
                tag.TagBytes.AddRange(Group);
                tag.TagBytes.Add(tagId);
                tag.TagBytes.Add(0x00);
                tag.TagBytes.Add(0x04); // float のパラメータサイズをリトルエンディアンで決め打ち
                tag.TagBytes.Add(0x00);
                tag.TagBytes.AddRange(BitConverter.GetBytes(param));
                return tag;
            }

            public static MessageStudioTag MakeEndTag(string tagString, byte tagId)
            {
                MessageStudioTag tag = new MessageStudioTag();
                tag.TagString = tagString;
                tag.TagBytes = new List<byte>();
                tag.TagBytes.AddRange(End);
                tag.TagBytes.AddRange(Group);
                tag.TagBytes.Add(tagId);
                tag.TagBytes.Add(0x00);
                return tag;
            }
        }

        private static readonly MessageStudioTag[] s_MessageStudioTags = new MessageStudioTag[]
        {
            MessageStudioTag.MakeStartTag("<nobr>", 0x04),
            MessageStudioTag.MakeEndTag("</nobr>", 0x04),

            MessageStudioTag.MakeStartTag("<indent>", 0x06),
            MessageStudioTag.MakeEndTag("</indent>", 0x06),

            MessageStudioTag.MakeStartTag("<italic>", 0x05),
            MessageStudioTag.MakeEndTag("</italic>", 0x05),

            MessageStudioTag.MakeStartTag("<paragraph/>", 0x00, 0.6f),
        };

        /// <summary>
        /// message 中の特定のタグを対応するバイナリ配列に変換します。
        /// </summary>
        /// <param name="message">入力メッセージ</param>
        /// <returns>タグをバイナリに変換し、テキストを出力のエンコーディングでエンコードしたバイト配列</returns>
        public static byte[] ReplaceMessageStudioTags(string message)
        {
            StringBuilder regexPattern = new StringBuilder();
            regexPattern.Append("(");
            foreach (var tag in s_MessageStudioTags)
            {
                regexPattern.Append(tag.TagString);
                regexPattern.Append("|");
            }
            regexPattern.Remove(regexPattern.Length - 1, 1);
            regexPattern.Append(")");

            List<byte> buffer = new List<byte>();

            MatchCollection matchCollection = Regex.Matches(message, regexPattern.ToString());
            int currentIndex = 0;
            foreach (Match match in matchCollection)
            {
                buffer.AddRange(OutputEncoding.GetBytes(message.Substring(currentIndex, match.Index - currentIndex)));
                buffer.AddRange(s_MessageStudioTags.Where(t => t.TagString == match.Groups[1].Value).Single().TagBytes);
                currentIndex = match.Index + match.Length;
            }
            buffer.AddRange(OutputEncoding.GetBytes(message.Substring(currentIndex)));

            return buffer.ToArray();
        }

        /// <summary>
        /// 参照先を値を持つエラーまで辿ります。
        /// </summary>
        /// <param name="referenceError">参照先のエラー。参照でなければ自分自身。</param>
        /// <param name="srcError">参照を探すエラー。</param>
        /// <param name="propertyKey">参照を探す項目。ダイアログ用メッセージ、ダイアログボタン用メッセージ、全画面用メッセージ、全画面ボタン用のメッセージのどれか。</param>
        /// <returns>参照であれば true、参照でなければ false</returns>
        private bool GetActualReference(out ErrorElement referenceError, ErrorElement srcError, LanguageElement.MessageElementKey propertyKey)
        {
            switch (propertyKey)
            {
                case LanguageElement.MessageElementKey.DlgMsg:
                    break;
                case LanguageElement.MessageElementKey.FlvMsg:
                    break;
                case LanguageElement.MessageElementKey.DlgBtn:
                case LanguageElement.MessageElementKey.DlgBtn0:
                case LanguageElement.MessageElementKey.DlgBtn1:
                case LanguageElement.MessageElementKey.DlgBtn2:
                    propertyKey = LanguageElement.MessageElementKey.DlgBtn0; // ボタンの参照は最初のボタンで探す
                    break;
                case LanguageElement.MessageElementKey.FlvBtn:
                case LanguageElement.MessageElementKey.FlvBtn0:
                case LanguageElement.MessageElementKey.FlvBtn1:
                case LanguageElement.MessageElementKey.FlvBtn2:
                    propertyKey = LanguageElement.MessageElementKey.FlvBtn0; // ボタンの参照は最初のボタンで探す
                    break;
                default:
                    // 上記以外のプロパティは参照を持たない。
                    throw new ArgumentException("key must be selected from predefined values", nameof(propertyKey));
            }

            // 言語によって参照先が変わる事はないので日本語で確認する
            string entryValue = srcError.GetLanguageElement(Language.JPja)[propertyKey];
            string referenceErrorCode;
            if (IsReferenceMessage(out referenceErrorCode, entryValue))
            {
                // 参照先がさらに参照している場合があるので、再帰的に参照を探す
                try
                {
                    var nextSrcError = m_ErrorElementDict[referenceErrorCode];
                    GetActualReference(out referenceError, nextSrcError, propertyKey);
                    return true;
                }
                catch (KeyNotFoundException e)
                {
                    Util.PrintError($"Reference {referenceErrorCode} used by {srcError.ErrorCodeText} not found. Input xml(s) may be incomplete.\n{e.ToString()}\n");
                    // Environment.Exit(-1);
                }
            }
            referenceError = srcError;
            return false;
        }

        /// <summary>
        /// システムデータ上で利用する参照先を設定します。参照専用のエラーを参照していた場合、適宜参照先を変更します。
        /// 参照でない場合は ErrorCode.InvalidErrorCode を設定します。
        /// </summary>
        /// <param name="srcError">参照先を設定するエラー</param>
        /// <param name="propertyKey">参照先を設定する項目</param>
        private void SetMessageReference(ErrorElement srcError, LanguageElement.MessageElementKey propertyKey)
        {
            ErrorElement referenceError;
            if (GetActualReference(out referenceError, srcError, propertyKey))
            {
                if (referenceError.IsReferenceOnly)
                {
                    if (m_MessageReferenceDict[propertyKey].ContainsKey(referenceError.ErrorCodeText))
                    {
                        // 参照先が参照専用のエラーであり、既にそのエラーを参照したエラーがあった場合
                        srcError.SetLanguageElementReference(propertyKey, m_MessageReferenceDict[propertyKey][referenceError.ErrorCodeText]);
                        return;
                    }
                    else
                    {
                        // 参照先が参照専用のエラーであり、初めてそのエラーを参照した場合
                        referenceError.CopyLanguageValuesTo(srcError, propertyKey);
                        srcError.SetLanguageElementReference(propertyKey, ErrorCode.InvalidErrorCode);
                        m_MessageReferenceDict[propertyKey].Add(referenceError.ErrorCodeText, srcError.ErrorCode);
                        return;
                    }
                }
                else
                {
                    // 参照先が通常のエラーである場合
                    srcError.SetLanguageElementReference(propertyKey, referenceError.ErrorCode);
                    return;
                }
            }
            else
            {
                // 参照でない場合
                srcError.SetLanguageElementReference(propertyKey, ErrorCode.InvalidErrorCode);
                return;
            }
        }

        /// <summary>
        /// システムデータへの変換を実行します。
        /// </summary>
        /// <param name="showProgress">進捗をコンソールに出力するかどうか</param>
        /// <returns>処理が最後まで成功した場合は true、それ以外の場合は false</returns>
        public bool Convert(IEnumerable<ErrorElement> errorElementList,
                            IEnumerable<UrlElement> urlElementList,
                            IEnumerable<MessageElement> messageElementList,
                            DatabaseInfoElement databaseInfoElement,
                            out SystemData systemData,
                            bool showProgress)
        {
            if (errorElementList == null || errorElementList.Count() == 0)
            {
                Util.PrintError("エラーメッセージデータがひとつも読み込まれていません。");
                systemData = null;
                return false;
            }

            systemData = new SystemData();

            if (databaseInfoElement != null)
            {
                systemData.DatabaseInfo.MajorVersion = databaseInfoElement.MajorVersion;
                systemData.DatabaseInfo.MinorVersion = databaseInfoElement.MinorVersion;
            }

            if (messageElementList != null)
            {
                foreach (var messageElement in messageElementList)
                {
                    var messageValues = new Dictionary<Language, byte[]>();
                    foreach (var p in messageElement.LanguageElements)
                    {
                        messageValues.Add(p.Language, OutputEncoding.GetBytes(p.Message));
                    }
                    systemData.Message.Add(messageElement.Id, messageValues);
                }
            }

            m_ErrorElementDict = new Dictionary<string, ErrorElement>();
            foreach (var err in errorElementList)
            {
                m_ErrorElementDict[err.ErrorCodeText] = err;
            }

            m_MessageReferenceDict = new Dictionary<LanguageElement.MessageElementKey, Dictionary<string, ErrorCode>>();
            m_MessageReferenceDict.Add(LanguageElement.MessageElementKey.DlgMsg, new Dictionary<string, ErrorCode>());
            m_MessageReferenceDict.Add(LanguageElement.MessageElementKey.DlgBtn, new Dictionary<string, ErrorCode>());
            m_MessageReferenceDict.Add(LanguageElement.MessageElementKey.FlvMsg, new Dictionary<string, ErrorCode>());
            m_MessageReferenceDict.Add(LanguageElement.MessageElementKey.FlvBtn, new Dictionary<string, ErrorCode>());

            int processCount = 0;
            Util.ProgressPrinter pp = null;

            if (showProgress)
            {
                Console.WriteLine(">> データ変換開始");
                pp = new Util.ProgressPrinter(errorElementList.Count(), 200);
            }

            foreach (var err in errorElementList)
            {
                try
                {
                    pp?.PrintProgress(processCount++);

                    // 参照専用のエラーは変換しない
                    if (err.IsReferenceOnly)
                    {
                        continue;
                    }

                    SetMessageReference(err, LanguageElement.MessageElementKey.DlgMsg);
                    SetMessageReference(err, LanguageElement.MessageElementKey.DlgBtn);
                    SetMessageReference(err, LanguageElement.MessageElementKey.FlvMsg);
                    SetMessageReference(err, LanguageElement.MessageElementKey.FlvBtn);

                    var errorMessageData = new ErrorMessageData();

                    var errCommonData = ErrorMessageCommonData.MakeFromErrorElement(err);
                    errorMessageData.CommonData = errCommonData;

                    var langData = new ErrorMessageLanguageData();

                    // ダイアログ用メッセージ
                    if (err.DialogViewMessageReference == ErrorCode.InvalidErrorCode)
                    {
                        var messageData = new Dictionary<Language, byte[]>();
                        foreach (LanguageElement l in err.LanguageElements)
                        {
                            if (l.DialogViewMessage != null && l.DialogViewMessage.Length > 0)
                            {
                                var message = ReplaceUrlTags(l.DialogViewMessage, l.Language, urlElementList);
                                messageData.Add(l.Language, ReplaceMessageStudioTags(message));
                            }
                        }
                        langData.Message.Add(LanguageElement.MessageElementKey.DlgMsg, messageData);
                    }

                    // ダイアログ用ボタンメッセージ
                    if (err.DialogViewButtonMessageReference == ErrorCode.InvalidErrorCode)
                    {
                        var messageData = new Dictionary<Language, byte[]>();
                        foreach (LanguageElement l in err.LanguageElements)
                        {
                            if (l.DialogViewButtonMessage.Count > 0)
                            {
                                messageData.Add(l.Language, OutputEncoding.GetBytes(l[LanguageElement.MessageElementKey.DlgBtn]));
                            }
                        }
                        langData.Message.Add(LanguageElement.MessageElementKey.DlgBtn, messageData);
                    }

                    // 全画面用メッセージ
                    if (err.FullScreenViewMessageReference == ErrorCode.InvalidErrorCode)
                    {
                        var messageData = new Dictionary<Language, byte[]>();
                        foreach (LanguageElement l in err.LanguageElements)
                        {
                            if (l.FullScreenViewMessage != null && l.FullScreenViewMessage.Length > 0)
                            {
                                var message = ReplaceUrlTags(l.FullScreenViewMessage, l.Language, urlElementList);
                                messageData.Add(l.Language, ReplaceMessageStudioTags(message));
                            }
                        }
                        langData.Message.Add(LanguageElement.MessageElementKey.FlvMsg, messageData);
                    }

                    // 全画面用ボタンメッセージ
                    if (err.FullScreenViewButtonMessageReference == ErrorCode.InvalidErrorCode)
                    {
                        var messageData = new Dictionary<Language, byte[]>();
                        foreach (LanguageElement l in err.LanguageElements)
                        {
                            if (l.FullScreenViewButtonMessage.Count > 0)
                            {
                                messageData.Add(l.Language, OutputEncoding.GetBytes(l[LanguageElement.MessageElementKey.FlvBtn]));
                            }
                        }
                        langData.Message.Add(LanguageElement.MessageElementKey.FlvBtn, messageData);
                    }

                    // 1つでも参照ではないメッセージを持っていた場合にのみ結果に追加する
                    if (langData.Message.Count > 0)
                    {
                        errorMessageData.LangData = langData;
                    }
                    systemData.ErrorMessageData.Add(err.ErrorCode, errorMessageData);
                }
                catch (Exception exception)
                {
                    Util.PrintError($"{err.ErrorCodeText} の変換処理に失敗しました。処理を中止します。\n{exception.ToString()}");
                    return false;
                }
            }

            if (showProgress)
            {
                Console.WriteLine("<< データ変換完了");
            }
            return true;
        }

        public static void CreateVersionText(string path, MessageListElement messageList, Dictionary<string, MetaInfo> errorListMetaInfoDict)
        {
            // ファイルの中身のフォーマット（prefix : nx- など）は現在 Ocean が作成しているものに合わせる。
            using (var writer = new StreamWriter(path))
            {
                var metas = from pair in errorListMetaInfoDict orderby pair.Key ascending select pair;
                if (metas != null)
                {
                    foreach (var m in metas)
                    {
                        if (m.Value != null)
                        {
                            writer.WriteLine($"nx-{m.Key} : {m.Value.Ocean}");
                        }
                    }
                }
                if (messageList?.MetaInfo != null)
                {
                    writer.WriteLine($"system-nx_btn : {messageList.MetaInfo.Ocean}");
                }
            }
        }

    }
}
