﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using Nintendo.Foundation.Audio;
using Nintendo.Foundation.IO;

namespace OpusEncoder
{
    internal class Program
    {
        internal class ConsoleApplication
        {
            private class Settings
            {
                [CommandLineValue(0, IsRequired = true, ValueName = "input-filepath", Description = "CommandLine_InputDescription", DescriptionConverterName = "LocalizeDescription")]
                public string InputFilePath { get; set; }

                [CommandLineOption('o', "output", ValueName = "output-filepath", Description = "CommandLine_OutputDescription", DescriptionConverterName = "LocalizeDescription")]
                public string OutputFilePath { get; set; }

                public const int BitRateDefaultValue = 96000;
                public const int BitRateMaxValue = 510000;
                public const int BitRateMinValue = 6000;
                public const double FrameSizeDefaultValue = 20;

                [CommandLineOption("bitrate", ValueName = "bitrate", Description = "CommandLine_BitrateDescription", DescriptionConverterName = "LocalizeDescription")]
                public int BitRate
                {
                    get { return bitRate; }
                    set
                    {
                        bitRate = Math.Min(Math.Max(value, BitRateMinValue), BitRateMaxValue);
                    }
                }
                private int bitRate;

                [CommandLineOption("framesize", ValueName = "framesize", Description = "CommandLine_FrameSizeDescription", DescriptionConverterName = "LocalizeDescription")]
                public double FrameSize { get; set; }

                [CommandLineOption("bitrate-control", ValueName = "bitrate-control", Description = "CommandLine_BitrateControlDescription", DescriptionConverterName = "LocalizeDescription")]
                public string BitrateControl { get; set; }

                [CommandLineOption("coding-mode", ValueName = "coding-mode", Description = "CommandLine_CodingModeDescription", DescriptionConverterName = "LocalizeDescription")]
                public string CodingMode { get; set; }

                [CommandLineOption('v', "verbose", ValueName = "verbose", Description = "CommandLine_VerboseDescription", DescriptionConverterName = "LocalizeDescription")]
                public bool IsVerbose { get; set; }

                public Settings()
                {
                    InputFilePath = null;
                    OutputFilePath = null;
                    BitRate = BitRateDefaultValue;
                    FrameSize = FrameSizeDefaultValue;
                    BitrateControl = null;
                    CodingMode = null;
                    IsVerbose = false;
                }

                public static string LocalizeDescription(string description, string valueName)
                {
                    return Resources.MessageResource.ResourceManager.GetString(description, Resources.MessageResource.Culture);
                }
            }

            private Settings settings = null;

            public bool Run(string[] args)
            {
                try
                {
                    CommandLineParserSettings commandLineParserSettings = new CommandLineParserSettings()
                    {
                        ApplicationDescription = Resources.MessageResource.CommandLine_ApplicationDescription,
                    };
                    if (new CommandLineParser(commandLineParserSettings).ParseArgs(args, out this.settings) == false)
                    {
                        return true;
                    }
                }
                catch
                {
                    return false;
                }

                if (string.IsNullOrEmpty(settings.InputFilePath))
                {
                    System.Console.Error.WriteLine(Resources.MessageResource.Message_InputFilenameNotSpecified);
                    return false;
                }
                // 絶対パス化
                settings.InputFilePath = System.IO.Path.GetFullPath(settings.InputFilePath);
                if (string.IsNullOrEmpty(settings.OutputFilePath))
                {
                    settings.OutputFilePath = System.IO.Path.ChangeExtension(settings.InputFilePath, ".opus");
                }
                // 絶対パス化
                settings.OutputFilePath = System.IO.Path.GetFullPath(settings.OutputFilePath);
                if (settings.IsVerbose)
                {
                    System.Console.WriteLine("InputFilePath: {0}", settings.InputFilePath);
                    System.Console.WriteLine("OutputFilePath: {0}", settings.OutputFilePath);
                    System.Console.WriteLine("FrameSize: {0} [ms]", settings.FrameSize);
                    System.Console.WriteLine("BitRate: {0} [bps]", settings.BitRate);
                    System.Console.WriteLine("BitRateControl: {0}", string.IsNullOrEmpty(settings.BitrateControl) ? "cvbr" : settings.BitrateControl);
                    System.Console.WriteLine("CodingMode: {0}", string.IsNullOrEmpty(settings.CodingMode) ? "celt" : settings.CodingMode);
                }

                if (!System.IO.File.Exists(settings.InputFilePath))
                {
                    System.Console.Error.WriteLine(Resources.MessageResource.Message_InputFileNotFound, settings.InputFilePath);
                    return false;
                }
                var OutputDirectory = System.IO.Path.GetDirectoryName(settings.OutputFilePath);
                if (!System.IO.Directory.Exists(OutputDirectory))
                {
                    System.Console.Error.WriteLine(Resources.MessageResource.Message_OutputDirectoryNotFound, OutputDirectory);
                    return false;
                }

                var waveFile = WaveFileReader.ReadWaveFile(settings.InputFilePath);
                var sampleRate = waveFile.Info.WaveFormat.SamplingRate;
                var channelCount = waveFile.Info.WaveFormat.ChannelCount;
                var channelMask = waveFile.Info.WaveFormat.ChannelMask;
                // もしチャンネルマスクがなければ、チャンネル数分だけ下位からビットを立てる。
                if (0 == channelMask)
                {
                    channelMask = (1u << channelCount) - 1;
                }
                if (settings.IsVerbose)
                {
                    System.Console.WriteLine("SampleRate: {0} [Hz]", waveFile.Info.WaveFormat.SamplingRate);
                    System.Console.WriteLine("ChannelCount: {0}", waveFile.Info.WaveFormat.ChannelCount);
                    System.Console.WriteLine("ChannelMask: " + channelMask.ToString("x"));
                }

                if (!(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000))
                {
                    System.Console.Error.WriteLine(Resources.MessageResource.Message_InvalidSampleRate, sampleRate);
                    return false;
                }
                if (!(settings.FrameSize == 2.5 || settings.FrameSize == 5 || settings.FrameSize == 10 || settings.FrameSize == 20))
                {
                    System.Console.Error.WriteLine(Resources.MessageResource.Message_InvalidFrameSize);
                    return false;
                }
                Nintendo.CodecTool.OpusEncoder opusEncoder = new Nintendo.CodecTool.OpusEncoder(sampleRate, channelCount, channelMask);

                opusEncoder.BitRate = settings.BitRate;

                try
                {
                    Nintendo.CodecTool.OpusEncoder.OpusBitRateControl bitRateControl;
                    string bitRateControlString = string.IsNullOrEmpty(settings.BitrateControl) ? "cvbr" : settings.BitrateControl;
                    bitRateControlString = bitRateControlString.ToLower();
                    if (bitRateControlString.Equals("cvbr"))
                    {
                        bitRateControl = Nintendo.CodecTool.OpusEncoder.OpusBitRateControl.ConstrainedVariable;
                    }
                    else if (bitRateControlString.Equals("cbr"))
                    {
                        bitRateControl = Nintendo.CodecTool.OpusEncoder.OpusBitRateControl.Constant;
                    }
                    else if (bitRateControlString.Equals("vbr"))
                    {
                        bitRateControl = Nintendo.CodecTool.OpusEncoder.OpusBitRateControl.Variable;
                    }
                    else
                    {
                        System.Console.Error.WriteLine(Resources.MessageResource.Message_InvalidBitrateControl);
                        return false;
                    }
                    opusEncoder.BitRateControl = bitRateControl;
                }
                catch (System.ArgumentException)
                {
                    System.Console.Error.WriteLine(Resources.MessageResource.Message_InvalidWaveFileFormat);
                    return false;
                }

                try
                {
                    Nintendo.CodecTool.OpusEncoder.OpusCodingMode codingMode;
                    string codingModeString = string.IsNullOrEmpty(settings.CodingMode) ? "celt" : settings.CodingMode;
                    codingModeString = codingModeString.ToLower();
                    if (codingModeString.Equals("celt"))
                    {
                        codingMode = Nintendo.CodecTool.OpusEncoder.OpusCodingMode.Celt;
                    }
                    else if (codingModeString.Equals("silk"))
                    {
                        codingMode = Nintendo.CodecTool.OpusEncoder.OpusCodingMode.Silk;
                    }
                    else
                    {
                        System.Console.Error.WriteLine(Resources.MessageResource.Message_InvalidCodingMode);
                        return false;
                    }
                    opusEncoder.CodingMode = codingMode;
                }
                catch (System.ArgumentException)
                {
                    System.Console.Error.WriteLine(Resources.MessageResource.Message_InvalidWaveFileFormat);
                    return false;
                }

                using (var outstream = new System.IO.FileStream(settings.OutputFilePath, System.IO.FileMode.Create))
                using (var writer = new System.IO.BinaryWriter(outstream))
                {
                    int OpusFrameSampleCount = (int)(sampleRate * settings.FrameSize / 1000);

                    var input = new short[OpusFrameSampleCount * channelCount];
                    var sampleCount = waveFile.Data.SampleCount;
                    var dataInfo = new Nintendo.CodecTool.OpusDataInfo();
                    var basicInfo = new Nintendo.CodecTool.OpusBasicInfo();
                    long dataInfoOffset = 0;
                    int sampleOffset = 0;

                    int bitsPerSample = waveFile.Info.WaveFormat.BitPerSample;

                    while (sampleCount >= 0)
                    {
                        var count = Math.Min(OpusFrameSampleCount, sampleCount);
                        for (var i = 0; i < count; ++i)
                        {
                            for (var c = 0; c < channelCount; ++c)
                            {
                                var data = waveFile.Data.Samples[c][sampleOffset + i];
                                if (bitsPerSample > 16)
                                {
                                    data >>= bitsPerSample - 16;
                                }
                                else if (bitsPerSample < 16)
                                {
                                    data <<= 16 - bitsPerSample;
                                }
                                input[i * channelCount + c] = (short)data;
                            }
                        }
                        sampleOffset += count;
                        for (var i = count; i < OpusFrameSampleCount; ++i)
                        {
                            for (var c = 0; c < channelCount; ++c)
                            {
                                input[i * channelCount + c] = 0;
                            }
                        }
                        byte[] output;
                        try
                        {
                            output = opusEncoder.Encode(input, OpusFrameSampleCount);
                        }
                        catch (System.ArgumentException)
                        {
                            System.Console.Error.WriteLine(Resources.MessageResource.Message_InvalidFrameSize);
                            return false;
                        }
                        catch (Nintendo.CodecTool.OpusEncoder.UnexpectedCodingModeException)
                        {
                            System.Console.Error.WriteLine(Resources.MessageResource.Message_UnexpectedCodingMode, channelCount);
                            return false;
                        }
                        catch (Nintendo.CodecTool.OpusEncoder.UnexpectedInternalErrorException)
                        {
                            System.Console.Error.WriteLine(Resources.MessageResource.Message_UnexpectedInternalError);
                            return false;
                        }

                        if (outstream.Length == 0)
                        {
                            // 基本情報の書き込み
                            basicInfo = opusEncoder.GetBasicInfo();
                            basicInfo.frameDataSize = (short)((opusEncoder.BitRateControl == Nintendo.CodecTool.OpusEncoder.OpusBitRateControl.Constant) ? output.Length + 8 : 0);  // +8 は後述の「opus_demo に挙動を合わせるための調整」のため
                            basicInfo.preskipSampleCount = (short)(opusEncoder.PreSkipSampleCount);
                            basicInfo.WriteHeader(writer);

                            // マルチストリームデータ向けヘッダの挿入
                            if (opusEncoder.TotalStreamCount > 1)
                            {
                                var multiStreamInfo = new Nintendo.CodecTool.OpusMultiStreamInfo();
                                multiStreamInfo = opusEncoder.GetMultiStreamInfo();
                                multiStreamInfo.WriteHeader(writer);
                            }

                            // データ情報の書き込み
                            // * サイズはこの時点ではわからないので、空書き込みでオフセットをずらすだけ
                            // * 後で書き込むために、オフセットを記憶しておく
                            dataInfoOffset = outstream.Position;
                            dataInfo.WriteHeader(writer);
                        }

                        dataInfo.size += output.Length;

                        // TODO: opus_demo に挙動を合わせるための調整、後日削除するかも
                        byte[] size = BitConverter.GetBytes(output.Length);
                        Array.Reverse(size);
                        UInt32[] finalRanges = new UInt32[opusEncoder.TotalStreamCount];
                        finalRanges = opusEncoder.FinalRanges;
                        byte[] tmp = BitConverter.GetBytes(finalRanges[0]);
                        Array.Reverse(tmp);
                        dataInfo.size += 8;
                        writer.Write(size);
                        writer.Write(tmp);

                        writer.Write(output);

                        if (0 == sampleCount)
                        {
                            break;
                        }
                        sampleCount -= OpusFrameSampleCount;
                    }

                    // 基本情報の更新
                    outstream.Position = 0;
                    basicInfo.dataInfoOffset = (int)dataInfoOffset;
                    basicInfo.WriteHeader(writer);

                    // データ情報の書き込み
                    outstream.Position = dataInfoOffset;
                    dataInfo.WriteHeader(writer);
                }
                return true;
            }
        }

        private static int Main(string[] args)
        {
            return (new ConsoleApplication().Run(args) ? 0 : 1);
        }
    }
}
