﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using System.Xml.Linq;
using BezelEditor.Mvvm.Messages;
using Livet.Messaging;
using Livet.Messaging.IO;
using Microsoft.WindowsAPICodePack.Dialogs;
using Nintendo.Authoring.AuthoringEditor.Controls;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor
{
    public static class ProjectFileHelper
    {
        public static void CreateNewProject(App app, ContentMetaType type)
        {
            app.CreateNewProject();
        }

        public static bool Open(
            Container diContainer,
            InteractionMessenger messenger,
            ImportableFileType type,
            Config config,
            string filePath,
            string originalNspFilePath,
            Action<string, string> onFileSelected,
            Action<string> onFileOpened)
        {
            return
                OpenInternal(
                    diContainer,
                    messenger,
                    type,
                    config,
                    filePath,
                    originalNspFilePath,
                    onFileSelected,
                    onFileOpened);
        }

        public static bool CheckTargetModified(
            ImportableFileType type,
            InteractionMessenger messenger,
            App app,
            AppModeType appMode,
            string operation)
        {
            if (app.IsChangedTarget == false)
                return true;

            var targetFileName =
                string.IsNullOrEmpty(app.Project.TargetFilePath)
                    ? GuiConstants.ImportableFileDict[type].NoName
                    : Path.GetFileName(app.Project.TargetFilePath);

            var choiceItems = new List<DialogMessage.ChoiceItem>();
            {
                // ファイル保存が可能なケース
                // * Project を編集している
                // * Nmeta を編集していて、かつエラーがない
                if (type == ImportableFileType.Project ||
                    (type == ImportableFileType.Meta && app.HasErrors == false))
                {
                    choiceItems.Add(
                        new DialogMessage.ChoiceItem(
                            string.Format(Resources.SaveAndFileOp, targetFileName, operation),
                            FileModifyCheckType.Save)
                    );
                    choiceItems.Add(
                        new DialogMessage.ChoiceItem(
                            string.Format(Resources.SaveAsAndFileOp, targetFileName, operation),
                            FileModifyCheckType.SaveAs));
                }
                choiceItems.Add(
                    new DialogMessage.ChoiceItem(
                        string.Format(Resources.WithoutSavingAndFileOp, targetFileName, operation),
                        FileModifyCheckType.WithoutSaving)
                );
                choiceItems.Add(
                    new DialogMessage.ChoiceItem(Resources.Cancel,
                        FileModifyCheckType.Cancel)
                );
            }

            var m = new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.None,
                Caption = Resources.Confirmation,
                Text = string.Format(Resources.Modified, targetFileName),
                StandardButtons = DialogMessage.StandardButtonsType.None,
                ChoiceItems = choiceItems
            };

            var r = messenger.GetResponse(m);

            var result = (FileModifyCheckType) r.Response.SelectedChiceItemTag;
            switch (result)
            {
                case FileModifyCheckType.Save:
                    if (Save(type.ToExportableFileType(appMode), messenger, app) == false)
                        return false;
                    break;

                case FileModifyCheckType.SaveAs:
                    if (SaveAs(type.ToExportableFileType(appMode), messenger, app) == false)
                        return false;
                    break;

                case FileModifyCheckType.WithoutSaving:
                    break;

                case FileModifyCheckType.Cancel:
                    return false;

                default:
                    throw new ArgumentOutOfRangeException();
            }

            return true;
        }

        public static bool Save(ExportableFileType type, InteractionMessenger messenger, App app)
        {
            return app.Project.TargetFilePath.IsNullOrEmpty
                ? SaveAs(type, messenger, app)
                : app.Save(type);
        }

        public static bool SaveAs(ExportableFileType type, InteractionMessenger messenger, App app)
        {
            var message = messenger.GetResponse(new SavingFileSelectionMessage(GuiConstants.MessageKey_FileSave)
            {
                Filter = GuiConstants.ImportableFileDict[type.ToImportableFileType()].DialogFilter,
                InitialDirectory = app.Project.TargetFilePath.IsNullOrEmpty
                    ? string.Empty
                    : app.Project.TargetFilePath.Parent.ToString()
            });

            var filePath = message?.Response?.FirstOrDefault();

            return filePath != null && app.Save(type, filePath);
        }

        private enum FileModifyCheckType
        {
            Save,
            SaveAs,
            WithoutSaving,
            Cancel
        }

        #region internal

        private static bool OpenInternal(
            Container diContainer,
            InteractionMessenger messenger,
            ImportableFileType type,
            Config config,
            string filePath,
            string originalNspFilePath,
            Action<string, string> onFileSelected,
            Action<string> onFileOpened
        )
        {
            var dialogFilter = GuiConstants.ImportableFileDict[type].DialogFilter;

            var isDirectory = dialogFilter == null;
            if (isDirectory)
                throw new NotImplementedException(); //todo:実装する

            if (filePath == null)
            {
                var message = messenger.GetResponse(new OpeningFileSelectionMessage(GuiConstants.MessageKey_FileOpen)
                {
                    Filter = dialogFilter
                });

                filePath = message?.Response?.FirstOrDefault();

                if (filePath == null)
                    return false;
            }

            if (OpenProjectFile(diContainer, messenger, onFileOpened, onFileSelected, config, type, filePath, originalNspFilePath) == false)
                return false;

            config.AddToMruList(type, filePath);
            return true;
        }

        private static void OpenRemoveFromMru(
            InteractionMessenger messenger, Config config, ImportableFileType type, string filePath)
        {
            var m = config.FirstOrDefault(type, filePath);
            if (m == null)
                return;

            if (ShowFileOpenErrorRemoveFromFileDialog(messenger, filePath))
                config.RemoveFromMruList(type, m);
        }

        private static bool OpenProjectFile(Container diContainer, InteractionMessenger messenger,
            Action<string> onFileOpened,
            Action<string, string> onFileSelected,
            Config config, ImportableFileType type,
            string filePath,
            string originalNspFilePath)
        {
            string errorMessage;
            bool isOpenRemoveFromMru = true;

            RETRY_OPENING:

            using (new InPreparationBlock())
            {
                try
                {
                    var currentAppMode = diContainer.GetInstance<AppProfile>().AppMode;
                    if (currentAppMode != AppModeType.Startup)
                    {
                        var nextAppMode = TypeHelper.ToAppMode(filePath);
                        if (nextAppMode != currentAppMode)
                            throw new CanNotLoadOnCurrentModeException();
                    }

                    onFileSelected(filePath, originalNspFilePath);
                    CheckOpenedProject(diContainer, messenger, filePath);
                    onFileOpened?.Invoke(filePath);
                    return true;
                }
                catch (NspHandleErrorException e)
                {
                    var stringHelper = diContainer.GetInstance<StringHelper>();

                    if (e.Result == NspHandleResult.NeedOriginalApplicationNspError)
                    {
                        var selectedNspFile = GetOriginalAppNspFilePathCandidate(messenger, stringHelper, filePath);
                        if (string.IsNullOrEmpty(selectedNspFile) == false)
                        {
                            // ファイルオープンを再度試みる
                            originalNspFilePath = selectedNspFile;
                            goto RETRY_OPENING;
                        }
                        // ファイル選択ダイアログをキャンセルした場合はファイルオープン自体も取り消し
                        return false;
                    }
                    errorMessage =
                        $"{stringHelper.GetNspHandleResultMessage(e.Result)}\n\n{filePath}\n----\n{string.Join("\n", e.ErrorMessages)}";
                }
                catch (FileNotFoundException)
                {
                    errorMessage = string.Format(Resources.DialogMessage_FileNotFound, filePath);
                }
                catch (CanNotLoadOnCurrentModeException)
                {
                    errorMessage = string.Format(Resources.DialogMessage_CanNotLoadOnCurrentMode, filePath);
                    isOpenRemoveFromMru = false;
                }
                catch (InvalidApplicationMetaItemException e)
                {
                    errorMessage = string.Format(Resources.DialogMessage_UnsupportedMetaItem, e.Name, e.Value);
                }
                catch (Exception)
                {
                    errorMessage = string.Format(Resources.DialogMessage_CannotLoad, filePath);
                }
            }

            messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Error,
                Caption = Resources.Error,
                Text = errorMessage,
                StandardButtons = DialogMessage.StandardButtonsType.Ok
            });

            if (isOpenRemoveFromMru)
                OpenRemoveFromMru(messenger, config, type, filePath);

            return false;
        }

        private static string GetOriginalAppNspFilePathCandidate(InteractionMessenger messenger, StringHelper stringHelper, string filePath)
        {
            messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Information,
                Caption = Resources.Confirmation,
                Text = Resources.DialogMessage_NeedOriginalApplicationNsp,
                StandardButtons = DialogMessage.StandardButtonsType.Ok
            });

            RETRY_FILE_SELECTION:
            var message = messenger.GetResponse(new OpeningFileSelectionMessage(GuiConstants.MessageKey_FileOpen)
            {
                Filter = Resources.DialogFilter_Nsp
            });
            var originalNspFilePath = message?.Response?.FirstOrDefault();
            if (originalNspFilePath != null)
            {
                var r = Task.Run(async () =>
                    await NspBuilder.IsValidPatchAndOriginalApplication(
                        filePath,
                        originalNspFilePath
                    ).ConfigureAwait(false)
                ).Result;
                if (r == NspHandleResult.Ok)
                    return originalNspFilePath;

                messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                {
                    Icon = TaskDialogStandardIcon.Error,
                    Caption = Resources.Error,
                    Text = string.Format(Resources.DialogMessage_RetrySelectOriginalAppNspFile,
                        stringHelper.GetNspHandleResultMessage(r)),
                    StandardButtons = DialogMessage.StandardButtonsType.Ok
                });

                goto RETRY_FILE_SELECTION;
            }
            return null;
        }

        private class CanNotLoadOnCurrentModeException : Exception
        {
        }

        private static bool ShowFileOpenErrorRemoveFromFileDialog(InteractionMessenger messenger, string filePath)
        {
            var message =
                string.Format(
                    File.Exists(filePath)
                        ? Resources.DialogMessage_CannotLoad_RemoveFromMruList
                        : Resources.DialogMessage_RemoveFromMruList,
                    filePath);

            var r = messenger.GetResponse(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Information,
                Caption = Resources.Information,
                Text = message,
                StandardButtons = DialogMessage.StandardButtonsType.YesNo
            });

            return r?.Response?.DialogResult == DialogMessage.DialogResultType.Yes;
        }

        private static void ShowIncompatibleSdkVersionDialog(InteractionMessenger messenger)
        {
            messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Warning,
                Caption = Resources.DialogTitle_IncompatibleSdkVersion,
                Text = Resources.DialogMessage_IncompatibleSdkVersion,
                StandardButtons = DialogMessage.StandardButtonsType.Ok
            });
        }

        private static void ShowHasFileFormatErrorsDialog(InteractionMessenger messenger)
        {
            messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Error,
                Caption = Resources.DialogTitle_HasFileFormatError,
                Text = Resources.DialogMessage_HasFileFormatError,
                StandardButtons = DialogMessage.StandardButtonsType.Ok
            });
        }

        private static void CheckOpenedProject(Container diContainer, InteractionMessenger messenger, string filePath)
        {
            var app = diContainer.GetInstance<App>();
            var project = app.Project;
            var isCompatibleSdk = project.IsUseCompatibleSdk();

            switch (TypeHelper.ToAppMode(project))
            {
                case AppModeType.ApplicationNsp:
                    CheckProjectFileExtension(filePath, project);
                    if (isCompatibleSdk == false)
                    {
                        // SDK のメジャーバージョンが異なる
                        project.Meta.IsReadOnly = true;
                    }
                    EnsureSupportBcat(project);
                    if (isCompatibleSdk == false)
                        ShowIncompatibleSdkVersionDialog(messenger);
                    if (project.Meta.FileFormatErrors.IsEmpty() == false)
                        ShowHasFileFormatErrorsDialog(messenger);
                    break;
                // AoC とパッチの nsp は編集をサポートしない
                case AppModeType.AocNsp:
                case AppModeType.PatchNsp:
                    project.Meta.IsReadOnly = true;
                    break;
                case AppModeType.ApplicationMeta:
                    EnsureVideoCaptureValue(project);
                    break;
            }
        }

        private static void CheckProjectFileExtension(string filePath, Project project)
        {
            var extension = Path.GetExtension(filePath);
            switch (extension?.ToLowerInvariant())
            {
                // .nspu 形式は製品化処理された nsp なので編集をサポートしない
                case ".nspu":
                    project.Meta.IsReadOnly = true;
                    break;
                // .xml 形式は 3PT からダウンロードしてきた nsp のアプリケーション情報を含むフォルダーで、編集操作は行えない
                case ".xml":
                    project.Meta.IsReadOnly = true;
                    break;
            }
        }

        private static void EnsureSupportBcat(Project project)
        {
            if (project.Capability.IsSupportBcat == false)
            {
                // .nsp の作成 SDK が Bcat 未サポートの SDK なら、無効な要素が含まれていないかをチェックする
                DetectInvalidApplicationBcatElements(project.Meta);
                // ApplicationMeta のフォーマットにエラーがある場合、読み取り専用で開く
                if (project.Meta.FileFormatErrors.IsEmpty() == false)
                    project.Meta.IsReadOnly = true;
            }
        }

        private static void EnsureVideoCaptureValue(Project project)
        {
            if (project.AppCapability.IsSupportVideoCapture)
            {
                var videoCapture = project.Meta.Application.VideoCapture;
                switch (videoCapture)
                {
                    case VideoCaptureType.Allow:
                    case VideoCaptureType.Deny:
                        throw new InvalidApplicationMetaItemException(nameof(videoCapture), videoCapture.ToString());
                }
            }
        }

        // Bcat 未サポート環境 (NX Addon 0.12 系, 1.x 系) において Bcat 関連要素の設定状況をチェックする
        private static void DetectInvalidApplicationBcatElements(ApplicationMeta appMeta)
        {
            var e = appMeta.Application.ContainsBcatElement;

            var errors = appMeta.FileFormatErrors;

            if (e.HasFlag(ApplicationMetaBcatElement.BcatPassphrase))
            {
                errors.Add(new ApplicationMetaError
                {
                    TitleTag = nameof(Resources.Error_BcatPassphrase_Title),
                    DescriptionTag = nameof(Resources.Error_BcatPassphrase_Description)
                });
            }

            if (e.HasFlag(ApplicationMetaBcatElement.BcatDeliveryCacheStorageSize))
            {
                errors.Add(new ApplicationMetaError
                {
                    TitleTag = nameof(Resources.Error_BcatDeliveryCacheStorageSize_Title),
                    DescriptionTag = nameof(Resources.Error_BcatDeliveryCacheStorageSize_Description)
                });
            }

            if (e.HasFlag(ApplicationMetaBcatElement.BcatSaveDataSize))
            {
                errors.Add(new ApplicationMetaError
                {
                    TitleTag = nameof(Resources.Error_BcatSaveDataSize_Title),
                    DescriptionTag = nameof(Resources.Error_BcatSaveDataSize_Description)
                });
            }
        }

        #endregion
    }
}
