﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reactive.Linq;
using System.Threading;
using System.Threading.Tasks;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Foundation.Utilities;
using BezelEditor.Mvvm;
using BezelEditor.Mvvm.Messages;
using Livet.Messaging.IO;
using Livet.Messaging.Windows;
using Microsoft.WindowsAPICodePack.Dialogs;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;

namespace Nintendo.Authoring.AuthoringEditor.NspEntriesWindow
{
    public class NspEntriesWindowVm : ViewModelBase
    {
        public ReactiveCommand CloseCommand { get; }
        public ReactiveCommand<FileEntry> OpenByAssociationAppCommand { get; }
        public ReactiveCommand<FileEntry> OpenSourceByAssociationAppCommand { get; }
        public ReactiveCommand<FileEntry> OpenTargetByAssociationAppCommand { get; }

        public FileEntry Root { get; set; }
        public ObservableCollection<FileEntry> SelectedItems { get; } = new ObservableCollection<FileEntry>();

        public ReactiveProperty<bool> IsExtracting { get; }
        public ReactiveProperty<bool> IsInMakeFileTree { get; }

        // ダミー。検索系コントロールのために用意
        public ReactiveProperty<string> DelaySearchWord { get; set; }

        public ReactiveProperty<ContentMetaType> Type { get; }
        public ReactiveProperty<bool> IsDisplayOnlyAppAndPatchDiff { get; }

        public ReactiveCommand DisplayOnlyDifferenceBetweenPatchCommand { get; }

        public ReactiveCommand OpenComparePatchFilePath { get; }

        // 現在開いているファイルがパッチなら true
        public bool IsPatchNsp { get; }
        // パッチ間差分の表示機能を提供できるなら true
        public bool CapableDisplayOnlyBetweenPatchDifference { get; }

        public ReactiveProperty<string> DiffPatchFilePath { get; }
        public ReadOnlyReactiveProperty<string> DiffPatchFilePathLocationLabel { get; }
        public ReadOnlyReactiveProperty<bool> IsDisplayOnlyBetweenPatchDiff { get; }

        public string ExtractCaption => Resources.ContextMenu_Extract;
        public string ExtractSourceCaption => string.Format(Resources.ContextMenu_ExtractSource, _nspFile.NspPath.FileName);
        public string ExtractTargetCaption => string.Format(Resources.ContextMenu_ExtractTarget,
            Path.GetFileName(DiffPatchFilePath.Value));

        public string OpenByAssociatedAppSourceCaption => string.Format(Resources.OpenByAssociationApp_Source,
            _nspFile.NspPath.FileName);
        public string OpenByAssociatedAppTargetCaption => string.Format(Resources.OpenByAssociationApp_Target,
            Path.GetFileName(DiffPatchFilePath.Value));

        public ReactiveCommand ExtractCommand { get; }
        public ReactiveCommand ExtractSourceCommand { get; }
        public ReactiveCommand ExtractTargetCommand { get; }

        public ReactiveProperty<string> StatusText { get; }

        private readonly ManualResetEventSlim _cancelSync = new ManualResetEventSlim(false);
        private readonly INspFile _nspFile;
        private readonly Project _project;

        private CancellationTokenSource _cancel;
        private INspFileEnumerable _nspFileEnumerable;
        private volatile bool _disposed;

        private enum ExtractionType
        {
            Application,
            SourcePatch,
            TargetPatch
        }

        public NspEntriesWindowVm(Project project)
        {
            _project = project;
            _nspFile = project.NspFile;

            CompositeDisposable.Add(() => _disposed = true);

            var helper = new NspEntriesHelper(project.NspFile).AddTo(CompositeDisposable);

            OpenComparePatchFilePath = new ReactiveCommand().AddTo(CompositeDisposable);
            OpenComparePatchFilePath.Subscribe(_ =>
            {
                WindowsUtility.OpenFileByExplorer(DiffPatchFilePath.Value);
            }).AddTo(CompositeDisposable);

            DisplayOnlyDifferenceBetweenPatchCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            DisplayOnlyDifferenceBetweenPatchCommand.Subscribe(async _ => await OpenDiffPatchFile()).AddTo(CompositeDisposable);

            IsPatchNsp = project.PatchContentMeta != null;
            CapableDisplayOnlyBetweenPatchDifference = IsPatchNsp && project.AppCapability.IsSupportPatchDifference;

            DiffPatchFilePath = new ReactiveProperty<string>(null, ReactivePropertyMode.RaiseLatestValueOnSubscribe).AddTo(CompositeDisposable);
            DiffPatchFilePathLocationLabel = DiffPatchFilePath
                .Select(x => string.Format(Resources.DiffPatchFilePath, Path.GetFileName(x)))
                .ToReadOnlyReactiveProperty().AddTo(CompositeDisposable);
            IsDisplayOnlyBetweenPatchDiff = DiffPatchFilePath.Select(x => !string.IsNullOrEmpty(x))
                .ToReadOnlyReactiveProperty().AddTo(CompositeDisposable);

            CompositeDisposable.Add(() =>
            {
                (_nspFileEnumerable as NspFileEnumerable)?.CancelEnumeration();
                _cancel?.Cancel();
                _cancelSync.Wait();
            });

            DelaySearchWord = new ReactiveProperty<string>().AddTo(CompositeDisposable);
            IsExtracting = new ReactiveProperty<bool>().AddTo(CompositeDisposable);

            CloseCommand = new ReactiveCommand().AddTo(CompositeDisposable);
            CloseCommand
                .Subscribe(_ => Messenger.Raise(new WindowActionMessage(WindowAction.Close, "WindowAction")))
                .AddTo(CompositeDisposable);

            OpenByAssociationAppCommand = IsExtracting.Inverse().ToReactiveCommand<FileEntry>().AddTo(CompositeDisposable);
            OpenByAssociationAppCommand
                .Subscribe(e =>
                {
                    if (e == null)
                        return;
                    DoFileExtraction(async () =>
                    {
                        var extractedFilePath = await helper.ExtractEntryToTempDir(e);
                        helper.OpenByAssociationAppAsync(extractedFilePath);
                    });
                })
                .AddTo(CompositeDisposable);

            OpenSourceByAssociationAppCommand =
                IsExtracting.Inverse().ToReactiveCommand<FileEntry>().AddTo(CompositeDisposable);
            OpenSourceByAssociationAppCommand.Subscribe(e =>
            {
                if (e == null)
                    return;
                DoFileExtraction(async () =>
                {
                    await ExtractFileFromPatches(
                        _nspFile,
                        ExtractionType.SourcePatch,
                        new Dictionary<string, FileEntry> { { e.RawPath, e } },
                        helper.TempDir.RootPath);
                    helper.OpenByAssociationAppAsync(Path.Combine(helper.TempDir.RootPath, e.FullPath));
                });
            }).AddTo(CompositeDisposable);

            OpenTargetByAssociationAppCommand =
                IsExtracting.Inverse().ToReactiveCommand<FileEntry>().AddTo(CompositeDisposable);
            OpenTargetByAssociationAppCommand.Subscribe(e =>
            {
                if (e == null)
                    return;
                DoFileExtraction(async () =>
                {
                    await ExtractFileFromPatches(
                        new NspFile(DiffPatchFilePath.Value) { OriginalNspFile = _nspFile.OriginalNspFile },
                        ExtractionType.TargetPatch,
                        new Dictionary<string, FileEntry> { { e.RawPath, e } },
                        helper.TempDir.RootPath);
                    helper.OpenByAssociationAppAsync(Path.Combine(helper.TempDir.RootPath, e.FullPath));
                });
            }).AddTo(CompositeDisposable);

            Root = new FileEntry().AddTo(CompositeDisposable);
            IsInMakeFileTree = new ReactiveProperty<bool>().AddTo(CompositeDisposable);
            Type = new ReactiveProperty<ContentMetaType>(ContentMetaType.Unknown).AddTo(CompositeDisposable);
            IsDisplayOnlyAppAndPatchDiff = new ReactiveProperty<bool>().AddTo(CompositeDisposable);

            Observable
                .Merge(IsDisplayOnlyAppAndPatchDiff.ToUnit(), DiffPatchFilePath.ToUnit())
                .Subscribe(async _ =>
                {
                    if (IsInMakeFileTree.Value)
                        return;
                    IsInMakeFileTree.Value = true;
                    var nspFileEnumerable = GetNspFileEnumerable();
                    var targetDiffPatchNsp = GetNspFileEnumerationType() == NspFileEnumeration.DiffPatch
                        ? new NspFile(DiffPatchFilePath.Value)
                        : null;
                    try
                    {
                        await BeginMakeFileTree(nspFileEnumerable, targetDiffPatchNsp);
                    }
                    catch (OperationCanceledException)
                    {
                        // ignored
                    }
                    if (_disposed)
                        return;
                    IsInMakeFileTree.Value = false;
                })
                .AddTo(CompositeDisposable);

            var collectionNotEmptyObservable =
                Observable.Merge(
                    IsExtracting.ToUnit(),
                    SelectedItems.ToCollectionChanged().ToUnit())
                .Select(x => SelectedItems.Any(y => !y.IsDirectory) && IsExtracting.Value == false);

            // アプリ or パッチ単体 or AoC の展開
            ExtractCommand = collectionNotEmptyObservable.ToReactiveCommand().AddTo(CompositeDisposable);
            ExtractCommand.Subscribe(async _ =>
            {
                await ExtractSelectedItems(_nspFile, ExtractionType.Application);
            }).AddTo(CompositeDisposable);

            // パッチ間差分における、比較元からの展開
            ExtractSourceCommand = collectionNotEmptyObservable
                .Select(x => x && SelectedItems.Any(y => y.ModifiedType != NspFileModifiedType.Added))
                .ToReactiveCommand().AddTo(CompositeDisposable);
            ExtractSourceCommand.Subscribe(async _ =>
            {
                await ExtractSelectedItems(_nspFile, ExtractionType.SourcePatch);
            }).AddTo(CompositeDisposable);

            // パッチ間差分における、比較先からの展開
            ExtractTargetCommand = collectionNotEmptyObservable
                .Select(x => x && SelectedItems.Any(y => y.ModifiedType != NspFileModifiedType.Removed))
                .ToReactiveCommand().AddTo(CompositeDisposable);
            ExtractTargetCommand.Subscribe(async _ =>
                {
                    await ExtractSelectedItems(
                        new NspFile(DiffPatchFilePath.Value) { OriginalNspFile = _nspFile.OriginalNspFile },
                        ExtractionType.TargetPatch);
                }).AddTo(CompositeDisposable);

            StatusText = new ReactiveProperty<string>().AddTo(CompositeDisposable);
        }

        private void DoFileExtraction(Action extractAction)
        {
            IsExtracting.Value = true;
            StatusText.Value = Resources.Extracting;
            try
            {
                extractAction();
                StatusText.Value = string.Format(Resources.NspEntries_ExtractionComplete, 1);
            }
            finally
            {
                if (!_disposed)
                {
                    if (_cancel.IsCancellationRequested)
                        StatusText.Value = null;
                    IsExtracting.Value = false;
                }
            }
        }

        private async Task ExtractSelectedItems(INspFile nspFile, ExtractionType extractionType)
        {
            if (IsExtracting.Value)
                return;

            IsExtracting.Value = true;
            try
            {
                // 展開対象のファイルリストを構築
                var extractFileEntries = new Dictionary<string, FileEntry>();
                foreach (var item in SelectedItems.Where(x => !x.IsDirectory))
                {
                    extractFileEntries.Add(item.RawPath, item);
                }

                // 展開対象ファイルをフィルタして、対象のファイルがなければその時点で処理をキャンセル
                if (FilterExtractFileEntries(nspFile, extractionType, extractFileEntries))
                    return;

                // 展開先フォルダーの取得
                if (GetExtractDestinationDirectory(out var destDirectory))
                    return;

                // 展開を実行
                await ExtractFileFromPatches(nspFile, extractionType, extractFileEntries, destDirectory);
            }
            finally
            {
                IsExtracting.Value = false;
            }
        }

        private bool GetExtractDestinationDirectory(out string destDirectory)
        {
            destDirectory = default(string);
            {
                var r = Messenger.GetResponse(new FolderSelectionMessage(GuiConstants.MessageKey_DirectoryOpen)
                {
                    Title = Resources.DestinationDirectory
                });
                destDirectory = r?.Response;
                if (destDirectory == null)
                {
                    // ダイアログでキャンセルした
                    StatusText.Value = null;
                    return true;
                }
            }
            return false;
        }

        private async Task ExtractFileFromPatches(INspFile nspFile, ExtractionType extractionType,
            Dictionary<string, FileEntry> extractFileEntries, string destDirectory)
        {
            // 比較元 nsp パッチからの展開の場合はパスの置き換えを行う。詳しくは後述
            var sourceNca = default(string);
            if (extractionType == ExtractionType.SourcePatch)
            {
                var contentMeta = _project.ApplicationContentMeta ?? _project.PatchContentMeta as ContentMeta;
                sourceNca = $"{contentMeta.GetContentId(ContentType.Program)}.nca";
            }

            // 展開元のNSP内パス => 展開先パスのテーブルを構築
            var extractFileTable = new Dictionary<string, string>();
            foreach (var entry in extractFileEntries.Values)
            {
                // 展開先のローカルファイル構造
                var extractLocalPath = Path.Combine(destDirectory, entry.FullPath);

                // 比較元 nsp に存在するファイルで、比較先で削除・追加されているファイルは、パスの置き換えが必要
                // たとえば AuthoringTool diffpatch v0.nsp v1.nsp v2.nsp のとき
                //
                // <v2-nca-digest>.nca/fs1/path/to/asset.txt
                //
                // と出力されファイルが削除の記録が残るが、実際に extract が可能なファイルパスは
                //
                // <v1-nca-digest>.nca/fs1/path/to/asset.txt
                //
                // になる
                var removeOrChanged = entry.ModifiedType == NspFileModifiedType.Removed ||
                                      entry.ModifiedType == NspFileModifiedType.Changed;
                if (extractionType == ExtractionType.SourcePatch && removeOrChanged)
                {
                    var rawPath = $"{sourceNca}{entry.RawPath.Substring(sourceNca.Length)}";
                    extractFileTable[rawPath] = extractLocalPath;
                }
                else
                {
                    // 追加・更新されたファイルは比較先 nsp のパスをそのまま使えば OK
                    extractFileTable[entry.RawPath] = extractLocalPath;
                }
            }

            StatusText.Value = Resources.Extracting;

            var extractTime = Stopwatch.StartNew();

            await nspFile.ExtractFilesAsync(extractFileTable);

            extractTime.Stop();
            StatusText.Value = string.Format(Resources.NspEntries_ExtractionComplete, extractFileEntries.Count) +
                               $"({extractTime.Elapsed})";
        }

        private bool FilterExtractFileEntries(INspFile nspFile, ExtractionType extractionType, Dictionary<string, FileEntry> extractFileEntries)
        {
            // 比較元から extract -> 比較先で追加されたファイルは extract できない旨を通知
            if (EnsureExtractSourceMissingFiles(nspFile, extractionType, extractFileEntries))
            {
                // ダイアログでキャンセルした
                StatusText.Value = null;
                return true;
            }

            // 比較先から extract -> 比較先で削除されたファイルは extract できない旨を通知
            if (EnsureExtractTargetMissingFiles(nspFile, extractionType, extractFileEntries))
            {
                // ダイアログでキャンセルした
                StatusText.Value = null;
                return true;
            }

            // 展開対象のファイルがない
            if (extractFileEntries.IsEmpty())
            {
                StatusText.Value = string.Format(Resources.NspEntries_ExtractionComplete, 0);
                return true;
            }

            // 処理を継続する
            return false;
        }

        private bool EnsureExtractTargetMissingFiles(INspFile nspFile, ExtractionType extractionType, Dictionary<string, FileEntry> extractFileEntries)
        {
            if (extractionType != ExtractionType.TargetPatch)
            {
                return false;
            }

            var targetPatchMissingFiles = SelectedItems.Where(x => x.ModifiedType == NspFileModifiedType.Removed).ToArray();
            if (targetPatchMissingFiles.IsEmpty())
            {
                return false;
            }

            var r = Messenger.GetResponse(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Warning,
                Caption = Resources.Confirmation,
                Text = string.Format(Resources.Warning_ExtractTarget_Mixtured,
                    Path.GetFileName(nspFile.NspPath),
                    string.Join(Environment.NewLine, targetPatchMissingFiles.Select(x => x.FullPath))),
                StandardButtons = DialogMessage.StandardButtonsType.OkCancel
            });
            if (r?.Response?.DialogResult != DialogMessage.DialogResultType.Ok)
            {
                return true;
            }

            // 展開できないファイルを対象から除外
            foreach (var file in targetPatchMissingFiles)
            {
                extractFileEntries.Remove(file.RawPath);
            }

            return false;
        }

        private bool EnsureExtractSourceMissingFiles(INspFile nspFile, ExtractionType extractionType, Dictionary<string, FileEntry> extractFileEntries)
        {
            if (extractionType != ExtractionType.SourcePatch)
            {
                return false;
            }

            var sourcePatchMissingFiles = SelectedItems.Where(x => x.ModifiedType == NspFileModifiedType.Added).ToArray();
            if (sourcePatchMissingFiles.IsEmpty())
            {
                return false;
            }

            var r = Messenger.GetResponse(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Warning,
                Caption = Resources.Confirmation,
                Text = string.Format(Resources.Warning_ExtractSource_Mixtured,
                    Path.GetFileName(nspFile.NspPath),
                    string.Join(Environment.NewLine, sourcePatchMissingFiles.Select(x => x.FullPath))),
                StandardButtons = DialogMessage.StandardButtonsType.OkCancel
            });
            if (r?.Response?.DialogResult != DialogMessage.DialogResultType.Ok)
            {
                return true;
            }

            // 展開できないファイルを対象から除外
            foreach (var file in sourcePatchMissingFiles)
            {
                extractFileEntries.Remove(file.RawPath);
            }

            return false;
        }

        private NspFileEnumeration GetNspFileEnumerationType()
        {
            var isOnlyPatchDiff = IsDisplayOnlyAppAndPatchDiff.Value;
            var enumType = NspFileEnumeration.All;
            if (isOnlyPatchDiff)
            {
                enumType = NspFileEnumeration.PatchedOnly;
            }
            else if (!string.IsNullOrEmpty(DiffPatchFilePath.Value))
            {
                enumType = NspFileEnumeration.DiffPatch;
            }
            return enumType;
        }

        private INspFileEnumerable GetNspFileEnumerable()
        {
            var nspFileEnumerable = _nspFile.Files(new NspFileEnumerateParameter
            {
                EnumerationType = GetNspFileEnumerationType(),
                DiffPatchNspFilePath = DiffPatchFilePath.Value
            });
            return nspFileEnumerable;
        }

        private async Task OpenDiffPatchFile()
        {
            RETRY_FILEOPEN:
            var message = Messenger.GetResponse(new OpeningFileSelectionMessage(GuiConstants.MessageKey_FileOpen)
            {
                Title = Resources.OpenDiffPatchFilePath_Caption,
                Filter = Resources.DialogFilter_Nsp
            });
            var nspFilePath = message?.Response?.FirstOrDefault();
            if (nspFilePath == null)
                return;

            var diffPatchNspFile = new NspFile(nspFilePath);
            {
                var contentMetaType = await NspImporter.DetectContentMetaTypeAsync(diffPatchNspFile);
                if (contentMetaType != ContentMetaType.Patch)
                {
                    // 選択したファイルが NSP パッチではない
                    Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                    {
                        Icon = TaskDialogStandardIcon.Information,
                        Caption = Resources.Information,
                        Text = Resources.Error_DiffPatch_FileOpenError,
                        StandardButtons = DialogMessage.StandardButtonsType.Ok
                    });
                    goto RETRY_FILEOPEN;
                }
            }
            var patchContent = await Task.Run(() => NspImporter.ReadContentMeta<PatchContentMeta>(diffPatchNspFile));

            if (patchContent.ApplicationId != _project.PatchContentMeta.ApplicationId)
            {
                // 選択したパッチとアプリケーション ID が異なる
                Messenger.Raise(new DialogMessage(GuiConstants.MessageKey_Dialog)
                {
                    Icon = TaskDialogStandardIcon.Information,
                    Caption = Resources.Information,
                    Text = string.Format(Resources.Error_DiffPatch_NotMatchApplicationId,
                        _project.PatchContentMeta.ApplicationId.ToHex(),
                        patchContent.ApplicationId.ToHex()),
                    StandardButtons = DialogMessage.StandardButtonsType.Ok
                });
                goto RETRY_FILEOPEN;
            }

            DiffPatchFilePath.Value = nspFilePath;
        }

        private async Task BeginMakeFileTree(INspFileEnumerable nspFileEnumerable, INspFile targetDiffPatchNspFile)
        {
            _cancel?.Cancel();
            _cancel = new CancellationTokenSource();

            (_nspFileEnumerable as NspFileEnumerable)?.CancelEnumeration();
            _nspFileEnumerable = nspFileEnumerable;

            _cancelSync.Reset();

            Root.Clear();

            var token = _cancel.Token;

            try
            {
                await Task.Run(() =>
                {
                    var builder = new FileTreeBuilder(Root, _nspFile, targetDiffPatchNspFile, nspFileEnumerable);
                    builder.Build(Type, token);
                }, token).ConfigureAwait(false);
            }
            catch (Exception e) when (e is OperationCanceledException)
            {
                // タスクのキャンセルエラーは無視する
            }
            finally
            {
                _cancelSync.Set();
            }
        }

        private class FileTreeBuilder
        {
            private readonly Dictionary<string, FileEntry> _entries = new Dictionary<string, FileEntry>();
            private readonly INspFile _nspFile;
            private readonly INspFile _targetDiffPatchNspFile;
            private readonly INspFileEnumerable _nspFileEnumerable;
            private readonly FileEntry _root;

            public FileTreeBuilder(FileEntry root, INspFile nspFile, INspFile targetDiffPatchNspFile, INspFileEnumerable nspFileEnumerable)
            {
                _root = root;
                _nspFile = nspFile;
                _targetDiffPatchNspFile = targetDiffPatchNspFile;
                _nspFileEnumerable = nspFileEnumerable;
            }

            public bool Build(IReactiveProperty<ContentMetaType> type, CancellationToken token)
            {
                var contentMetaType = NspImporter.DetectContentMetaTypeAsync(_nspFile).Result;

                type.Value = contentMetaType;

                switch (contentMetaType)
                {
                    case ContentMetaType.AddOnContent:
                        return BuildAddonContentFileTree(token);
                    case ContentMetaType.Application:
                        return BuildApplicationFileTree(token);
                    case ContentMetaType.Patch:
                        return BuildApplicationFileTree(token);
                    default:
                        throw new ArgumentException(nameof(contentMetaType));
                }
            }

            private bool BuildAddonContentFileTree(CancellationToken token)
            {
                var contentMetas = NspImporter.ReadAllContentMeta<AocContentMeta>(_nspFile).WhenAll().Result;

                foreach (var contentMeta in contentMetas)
                {
                    var contentData = contentMeta.Contents.FirstOrDefault(x => x.Type == ContentType.Data);
                    if (contentData == null)
                        continue;

                    var tag = contentMeta.Tag;
                    if (string.IsNullOrEmpty(tag) == false)
                        tag = $" ({tag})";

                    var entry = new FileEntry
                    {
                        Name = $"{contentMeta.Index}{tag}",
                        FileSize = ulong.MaxValue,
                        Directory = "/"
                    };
                    _entries[$"{contentData.Id}.nca"] = entry;
                    _root.AddEntry(entry);
                }

                foreach (var srcEntry in _nspFileEnumerable)
                {
                    if (token.IsCancellationRequested)
                        break;

                    // XXXXXXX.nca/fs0/file/path/to.txt
                    // ^^^^^^^^^^^     ^^^^^^^^^^^^^^^^
                    // paths[0]        paths[2]
                    var paths = srcEntry.FilePath.Split(new[] { '/' }, 3);
                    if (paths.Length != 3)
                        continue;

                    var nca = paths[0];
                    var filePath = paths[2];
                    AddEntry($"{nca}/{filePath}", srcEntry.FilePath, srcEntry);
                }

                return token.IsCancellationRequested;
            }

            private bool BuildApplicationFileTree(CancellationToken token)
            {
                var contentMeta = NspImporter.ReadAllContentMeta<ApplicationContentMeta>(_targetDiffPatchNspFile ?? _nspFile)?.FirstOrDefault()?.Result;
                if (contentMeta == null)
                    return true;

                var programNca = $"{contentMeta.GetContentId(ContentType.Program)}.nca";

                // 予約されたルートエントリを追加
                {
                    var code = new FileEntry { Name = "code", Directory = "/", FileSize = ulong.MaxValue };
                    _entries["code"] = code; // fs0
                    _root.AddEntry(code);

                    var data = new FileEntry { Name = "data", Directory = "/", FileSize = ulong.MaxValue };
                    _entries["data"] = data; // fs1
                    _root.AddEntry(data);
                }

                foreach (var srcEntry in _nspFileEnumerable)
                {
                    if (token.IsCancellationRequested)
                        break;

                    // プログラム以外のエントリーはスキップ
                    if (srcEntry.FilePath.StartsWith(programNca) == false ||
                        programNca.Length + 1 > srcEntry.FilePath.Length)
                    {
                        continue;
                    }

                    // プログラム内のエントリーを格納
                    var filePath = GetProgramTreePath(srcEntry, programNca);
                    AddEntry(filePath, srcEntry.FilePath, srcEntry);
                }

                return token.IsCancellationRequested;
            }

            private static string GetProgramTreePath(NspFileEntry srcEntry, string programNca)
            {
                var filePath = srcEntry.FilePath.Substring(programNca.Length + 1);
                // ルートエントリの先頭部分 (fs0, fs1) をわかりやすい名前に置き換え
                var rootNameIndex = filePath.IndexOf('/');
                if (rootNameIndex == -1)
                    return filePath;
                var rootName = filePath.Substring(0, rootNameIndex);
                var remainPath = rootNameIndex + 1 >= filePath.Length
                    ? string.Empty
                    : $"/{filePath.Substring(rootNameIndex + 1)}";
                switch (rootName)
                {
                    case "fs0":
                        filePath = $"code{remainPath}";
                        break;
                    case "fs1":
                        filePath = $"data{remainPath}";
                        break;
                }
                return filePath;
            }

            private void AddEntry(string filePath, string basePath, NspFileEntry srcEntry)
            {
                var p = filePath.Split('/');
                if (_entries.ContainsKey(p[0]) == false)
                    return;

                var dir = new string[p.Length - 1];
                Array.Copy(p, dir, dir.Length);
                var dirString = string.Join("/", dir);

                var newEntry = new FileEntry
                {
                    Name = p[p.Length - 1],
                    Directory = dirString,
                    FileSize = srcEntry.FileSize,
                    RawPath = basePath,
                    ModifiedType = srcEntry.ModifiedType,
                    ModifiedFileSize = srcEntry.ModifiedFileSize
                };

                FileEntry target;
                if (_entries.TryGetValue(dirString, out target))
                {
                    target.AddEntry(newEntry);
                }
                else
                {
                    Debug.Assert(_entries.ContainsKey(p[0]));
                    _entries[p[0]].AddEntry(newEntry, dir, 1);
                }
            }
        }

    }
}
