﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reactive.Linq;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Mvvm;
using BezelEditor.Mvvm.Messages;
using Microsoft.WindowsAPICodePack.Dialogs;
using Nintendo.Authoring.AuthoringEditor.Convertes;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Nintendo.Authoring.AuthoringEditor.Properties;
using Reactive.Bindings;
using Reactive.Bindings.Extensions;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params
{
    public class TitleParamVm : ParamVm
    {
        public ReadOnlyReactiveCollection<TitleVm> Titles { get; set; }
        public ReactiveProperty<TitleVm> SelectedTitle { get; set; }

        public ReadOnlyReactiveProperty<string> ValidationMessage { get; }

        public ReactiveCommand AddCommand { get; }
        public ReactiveCommand<TitleVm> RemoveCommand { get; }

        public ReadOnlyReactiveProperty<bool> CanAddOrRemoveTitle { get; }

        public Container DiContainer { get; }

        public DisposableDirectory DisposableDirectory { get; }

        private Dictionary<LanguageType, Title> _PresetTitles = new Dictionary<LanguageType, Title>();

        private readonly object[] _PresetLanguages;

        public TitleParamVm(
            Container diContainer,
            string captionTag, string commentTag,
            ApplicationMeta appMeta, ObservableCollection<Title> model,
            ReactiveProperty<bool> isReadOnly, ReadOnlyReactiveProperty<bool> canAddOrRemoveTitle)
            : base(captionTag, commentTag, null)
        {
            Debug.Assert(appMeta != null);
            Debug.Assert(model != null);

            // ViewModel 初期化時点でのタイトル情報を保存しておく
            // nsp 編集において削除したタイトルと同一言語のタイトルを追加した際、元々 nsp に設定されていた画像を復元するために使用
            foreach (var t in model)
            {
                _PresetTitles[t.Language] = t;
            }
            _PresetLanguages = GetLanguageSelection(diContainer, appMeta);

            DiContainer = diContainer;
            DisposableDirectory = new DisposableDirectory().AddTo(CompositeDisposable);

            IsReadOnly = isReadOnly;
            CanAddOrRemoveTitle = canAddOrRemoveTitle;

            RemoveCommand = Observable.Merge(CanAddOrRemoveTitle.ToUnit())
                    .Select(_ => CanAddOrRemoveTitle.Value)
                    .ToReactiveCommand<TitleVm>()
                    .AddTo(CompositeDisposable);
            RemoveCommand.Subscribe(title => RemoveModel(model, title.Model)).AddTo(CompositeDisposable);

            AddCommand = Observable.Merge(CanAddOrRemoveTitle.ToUnit())
                    .Select(_ => CanAddOrRemoveTitle.Value)
                    .ToReactiveCommand()
                    .AddTo(CompositeDisposable);
            AddCommand.Subscribe(_ => model.Add(CreateTitle(model)))
                .AddTo(CompositeDisposable);

            Titles =
                model.ToReadOnlyReactiveCollection(title => new TitleVm(this, title, _PresetLanguages, () =>
                    {
                        ApplyPresetTitleIconOnNsp(title);
                        UpdateValidation();
                    }))
                    .AddTo(CompositeDisposable);

            SelectedTitle = new ReactiveProperty<TitleVm>(Titles.FirstOrDefault())
                .AddTo(CompositeDisposable);

            Titles.CollectionChangedAsObservable().Subscribe(_ =>
            {
                if (SelectedTitle.Value == null)
                    SelectedTitle.Value = Titles.FirstOrDefault();

                UpdateValidation();
            }).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            ValidationMessage = Observable
                .Merge(appMeta.Application.ObserveProperty(x => x.IsErrorTitles).ToUnit())
                .Merge(appMeta.Application.Titles.CollectionChangedAsObservable().ToUnit())
                .Merge(appMeta.Application.SupportedLanguages.ObserveElementProperty(x => x.IsSupported).ToUnit())
                .Merge(CultureService.Instance.ObserveProperty(x => x.Resources).ToUnit())
                .Select(x => diContainer.GetInstance<PageValidations>().TitleInMeta(appMeta))
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);

            UpdateValidation();
        }

        private void RemoveModel(ObservableCollection<Title> model, Title title)
        {
            var lang = new EnumToCaptionConverter().Convert(title.Language, null, null, null);

            var r = Messenger.GetResponse(new DialogMessage(GuiConstants.MessageKey_Dialog)
            {
                Icon = TaskDialogStandardIcon.Information,
                Caption = Resources.Confirmation,
                Text = string.Format(Resources.DialogMessage_Confirmation_DeleteTitle, lang),
                StandardButtons = DialogMessage.StandardButtonsType.OkCancel
            });

            if (r?.Response?.DialogResult == DialogMessage.DialogResultType.Ok)
                model.Remove(title);
        }

        private void UpdateValidation()
        {
            if (Titles == null)
                return;

            foreach (var t in Titles)
                t.LanguageValidationError.Value = Titles.Count(tt => tt.Language.Value == t.Language.Value) == 1
                    ? null
                    : Resources.Title_Language_Validate_Same;
        }

        private static Title CreateTitle(ObservableCollection<Title> model)
        {
            var current = model.Select(x => x.Language).ToHashSet();

            var newLanguage = Enum.GetValues(typeof(LanguageType))
                .Cast<LanguageType>()
                .FirstOrDefault(o => current.Contains(o) == false);

            return new Title {Language = newLanguage};
        }

        protected override bool IsVisibled(string keyword)
        {
            if (base.IsVisibled(keyword))
                return true;

            if (Titles.Any() == false)
                return false;

            var texts = new[]
            {
                Resources.Title_Language_Caption,
                Resources.Title_Name_Caption,
                Resources.Title_Publisher_Caption,
                Resources.Title_IconFilePath_Caption,
                Resources.Title_IconFilePath_Comment,
                Resources.IconPreview,
            };

            foreach (var text in texts)
                if (text.IndexOf(keyword, StringComparison.OrdinalIgnoreCase) != -1)
                    return true;

            return false;
        }

        private void ApplyPresetTitleIconOnNsp(Title model)
        {
            Title presetTitle;
            if (_PresetTitles.TryGetValue(model.Language, out presetTitle) == false)
                return;

            model.OriginalIconFilePath = presetTitle.OriginalIconFilePath;
            model.OriginalNxIconFilePath = presetTitle.OriginalNxIconFilePath;
            model.NspRawIconFilePath = presetTitle.NspRawIconFilePath;
            model.NspNxIconFilePath = presetTitle.NspNxIconFilePath;

            if (string.IsNullOrEmpty(model.IconFilePath)
                && File.Exists(model.OriginalIconFilePath)
                && File.Exists(model.OriginalNxIconFilePath))
            {
                model.IsReplaceIcon = false;
            }
        }

        private static object[] GetLanguageSelection(Container diContainer, ApplicationMeta appMeta)
        {
            var isSupportChineseAndKorean = diContainer.GetInstance<ApplicationCapability>()
                .IsSupportEditChineseAndKoreanLanguages;

            // ReSharper disable once HeapView.ObjectAllocation.Evident
            // ReSharper disable once UseObjectOrCollectionInitializer
            var l = new List<object>();
            {
                foreach (AreaType area in Enum.GetValues(typeof(AreaType)))
                {
                    l.Add(new AreaCaption(area.ToString()));

                    var languages = LanguageHelper.AreaToLanguages(area);
                    foreach (var lang in languages)
                    {
                        var supportedLanguage = appMeta.Application.SupportedLanguages.FirstOrDefault(x => x.Language == lang);
                        if (isSupportChineseAndKorean == false &&
                            supportedLanguage?.IsNotSupportedChineseOrKoreanLanguage == true)
                        {
                            // 中韓台が未サポートの環境で、かつ中韓台の言語が (nmeta 等で) 選択されていない
                            continue;
                        }

                        l.Add(lang);
                    }
                }
            }

            return l.ToArray();
        }
    }

    public class AreaCaption
    {
        public string Caption =>
            _captionTag == null ? null : Resources.ResourceManager.GetString(_captionTag, Resources.Culture);

        private readonly string _captionTag;

        public AreaCaption(string captionTag)
        {
            _captionTag = captionTag;
        }
    }

    public class TitleVm : ViewModelBase
    {
        public ReactiveProperty<LanguageType> Language { get; }
        public ReactiveProperty<string> LanguageValidationError { get; }
        public StringParamVm Name { get; }
        public StringParamVm Publisher { get; }
        public FilePathStringParamVm IconFilePath { get; }
        public IconPreviewParamVm IconPreview { get; }

        public ReadOnlyReactiveProperty<bool> HasErrors { get; }

        public ReactiveCommand<TitleVm> RemoveCommand { get; }

        public Title Model { get; }

        public object[] Languages { get; } // Language か AreaCaption

        // todo:viewがこの名前の必要ないプロパティを要求する。消したい
        public ReadOnlyReactiveProperty<bool> IsHidden { get; set; }

        public ReactiveProperty<bool> IsReadOnly { get; }

        public TitleVm(TitleParamVm parent, Title model, object[] languages, Action onLanguageChanged)
        {
            Debug.Assert(model != null);

            var diContainer = parent.DiContainer;
            var isAppNspMode = diContainer.GetInstance<AppProfile>().AppMode == AppModeType.ApplicationNsp;

            IsReadOnly = parent.IsReadOnly;

            Model = model;
            Languages = languages;
            RemoveCommand = parent.RemoveCommand;

            /////////////////////////////////////////////////////////
            Language = model.ToReactivePropertyAsSynchronized(x => x.Language).AddTo(CompositeDisposable);
            Language.Subscribe(_ => onLanguageChanged()).AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            LanguageValidationError = new ReactiveProperty<string>().AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            var nameRp = model
                .ToReactivePropertyAsSynchronized(x => x.Name).AddTo(CompositeDisposable)
                .SetValidateNotifyError(s => diContainer.GetInstance<PageValidations>().TitleName(s))
                .AddTo(CompositeDisposable);

            Name = new StringParamVm(
                nameof(Resources.Title_Name_Caption),
                null,
                nameRp
            ).AddTo(CompositeDisposable);
            Name.IsExpandParentWidth.Value = true;
            Name.MaxLength = Title.MaxNameLength;

            /////////////////////////////////////////////////////////
            var publisherRp = model
                .ToReactivePropertyAsSynchronized(x => x.Publisher).AddTo(CompositeDisposable)
                .SetValidateNotifyError(s => diContainer.GetInstance<PageValidations>().PublisherName(s))
                .AddTo(CompositeDisposable);

            Publisher = new StringParamVm(
                nameof(Resources.Title_Publisher_Caption),
                null,
                publisherRp
            ).AddTo(CompositeDisposable);
            Publisher.IsExpandParentWidth.Value = true;
            Publisher.MaxLength = Title.MaxPublisherLength;

            /////////////////////////////////////////////////////////
            var project = diContainer.GetInstance<Project>();

            /////////////////////////////////////////////////////////
            IconFilePath = Utilities.CreateExpandablePathVm(
                project,
                model.IconFilePath,
                filePathRp => new FilePathStringParamVm(
                    nameof(Resources.Title_IconFilePath_Caption),
                    nameof(Resources.Title_IconFilePath_Comment),
                    filePathRp,
                    nameof(Resources.DialogFilter_Title_IconFilePath),
                    null,
                    false,
                    filePathRp.Select(x => project.ToAbsolutePath(x))
                ),
                _ => diContainer.GetInstance<PageValidations>().IconFilePath(model),
                CompositeDisposable);

            var iconFilePathRp = (ReactiveProperty<string>) IconFilePath.Property;
            var iconFileExpandEnvObservable = model.IconFilePath.ObserveProperty(x => x.IsExpandEnvironmentVariable).ToUnit();

            Observable
                .Merge(iconFilePathRp.ToUnit())
                .Merge(iconFileExpandEnvObservable)
                .Subscribe(_ => model.ForceValidation())
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            IconPreview = new IconPreviewParamVm(
                diContainer,
                nameof(Resources.IconPreview),
                model,
                parent.DisposableDirectory
            ).AddTo(CompositeDisposable);

            if (parent.CanAddOrRemoveTitle.Value == false &&
                isAppNspMode &&
                File.Exists(model.OriginalIconFilePath) == false)
            {
                // NX Addon 0.12 系のみの対応:
                // * アプリ nsp でアイコンを編集可能なのは設定済みのものに限る
                // * アイコンが未設定のタイトルは全て読み取り専用項目とする
                IconFilePath.IsReadOnly.Value = true;
            }
            else
            {
                // それ以外の場合は IsReadOnly の状態に統一
                IsReadOnly.Subscribe(x =>
                {
                    Name.IsReadOnly.Value = x;
                    IconFilePath.IsReadOnly.Value = x;
                    Publisher.IsReadOnly.Value = x;
                }).AddTo(CompositeDisposable);

                IconFilePath.CaptionIsUseTag = isAppNspMode ? nameof(Resources.Replace) : nameof(Resources.IsUse);
                IconFilePath.IsUse = model.ToReactivePropertyAsSynchronized(x => x.IsReplaceIcon)
                    .AddTo(CompositeDisposable);

                model.ObserveProperty(x => x.IsReplaceIcon)
                    .Subscribe(_ => iconFilePathRp.ForceValidate())
                    .AddTo(CompositeDisposable);
            }

            /////////////////////////////////////////////////////////
            CultureService.Instance.ObserveProperty(x => x.Resources)
                .Subscribe(_ =>
                {
                    // ReSharper disable once ExplicitCallerInfoArgument
                    RaisePropertyChanged(nameof(Languages));

                    Language.ForceNotify();
                    nameRp.ForceNotify();
                    publisherRp.ForceNotify();
                    iconFilePathRp.ForceNotify();
                })
                .AddTo(CompositeDisposable);

            /////////////////////////////////////////////////////////
            HasErrors = Observable
                .Merge(model.ObserveProperty(x => x.Name).ToUnit())
                .Merge(model.ObserveProperty(x => x.Publisher).ToUnit())
                .Merge(iconFilePathRp.ToUnit())
                .Merge(iconFileExpandEnvObservable)
                .Merge(model.ObserveProperty(x => x.IsReplaceIcon).ToUnit())
                .Select(_ => model.HasErrors)
                .ToReadOnlyReactiveProperty()
                .AddTo(CompositeDisposable);
        }
    }
}
