﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.ObjectModel;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages
{
    public class PageValidations
    {
        private readonly Container _diContainer;

        private Project CurrentProject => _diContainer.GetInstance<Project>();

        private bool IsSkipValidation => CurrentProject.Meta.IsReadOnly;

        public PageValidations(Container diContainer)
        {
            _diContainer = diContainer;
        }

        public string ApplicationId(ulong i)
        {
            if (IsSkipValidation)
                return null;

            var r = Core.Core.ValidateApplicationId(i);
            switch (r)
            {
                case Core.Core.ApplicationIdValidationType.Ok:
                    return null;

                case Core.Core.ApplicationIdValidationType.RangeOver:
                    return string.Format(Resources.ApplicationId_Validate_RangeOver,
                        Constants.ProgramIdMinimum.ToUpperHex(), Constants.ProgramIdMaximum.ToUpperHex());

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string DisplayVersion(string s)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateDisplayVersion(s);
            switch (r)
            {
                case Application.DisplayVersionValidationType.Ok:
                    return null;

                case Application.DisplayVersionValidationType.Empty:
                    return Resources.DisplayVersion_Validate_Empty;

                case Application.DisplayVersionValidationType.MaxLengthError:
                    return string.Format(Resources.DisplayVersion_Validate_Length,
                        Application.MaxDisplayVersionLength);

                case Application.DisplayVersionValidationType.CharaError:
                    return Resources.DisplayVersion_Validate_ASCII;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string CardSize(ApplicationMeta appMeta)
        {
            if (IsSkipValidation)
                return null;

            var r = CardSpec.ValidateSize(appMeta.CardSpec.IsAutomaticSettingSize, appMeta.CardSpec.Size);

            switch (r)
            {
                case CardSpec.SizeValidationType.Ok:
                    return null;

                case CardSpec.SizeValidationType.SizeError:
                    return Resources.CardSpec_Size_Validate_SizeError;

                case CardSpec.SizeValidationType.UnsupportedSizeError:
                    return Resources.CardSpec_Size_Validate_UnsupportedSizeError;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string CardClockRate(ApplicationMeta appMeta)
        {
            if (IsSkipValidation)
                return null;

            var r = CardSpec.ValidateClockRate(appMeta.CardSpec.IsAutomaticSettingClockRate,
                appMeta.CardSpec.ClockRate, appMeta.CardSpec.Size);

            switch (r)
            {
                case CardSpec.ClockRateValidationType.Ok:
                    return null;

                case CardSpec.ClockRateValidationType.ClockRateError:
                    return Resources.CardSpec_ClockRate_Validate_ClockRateError;

                case CardSpec.ClockRateValidationType.Under8GB:
                    return Resources.CardSpec_ClockRate_Validate_Under8GB;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string MainThreadStackSize(ulong s)
        {
            if (IsSkipValidation)
                return null;

            var r = Core.Core.ValidateMainThreadStackSize(s);
            switch (r)
            {
                case Core.Core.MainThreadStackSizeValidationType.Ok:
                    return null;

                case Core.Core.MainThreadStackSizeValidationType.AlignError:
                    return Resources.MainThreadStackSize_Validate_Align;

                case Core.Core.MainThreadStackSizeValidationType.Empty:
                    return Resources.MainThreadStackSize_Validate_Empty;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string SystemResourceSize(uint s, bool isProcessSpace64Bit)
        {
            if (IsSkipValidation)
                return null;

            var r = Core.Core.ValidateSystemResourceSize(s, isProcessSpace64Bit);
            switch (r)
            {
                case Core.Core.SystemResourceSizeValidationType.Ok:
                    return null;

                case Core.Core.SystemResourceSizeValidationType.AlignError:
                    return Resources.SystemResourceSize_Validate_AlignError;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string ApplicationErrorCodeCategory(ApplicationMeta appMeta, string category)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateApplicationErrorCodeCategory(
                appMeta.Application.IsUseApplicationErrorCode, category);
            switch (r)
            {
                case Application.ApplicationErrorCodeCategoryValidationType.Ok:
                    return null;

                case Application.ApplicationErrorCodeCategoryValidationType.EmptyError:
                    return Resources.PleaseSet;

                case Application.ApplicationErrorCodeCategoryValidationType.LengthError:
                    return string.Format(Resources.ApplicationErrorCodeCategory_Validate_Length,
                        Application.MaxApplicationErrorCodeCategoryLength);

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string HtmlDocumentPath(Application app)
        {
            if (IsSkipValidation)
                return null;

            var r = app.ValidateHtmlDocumentPath(
                app.IsReplaceHtmlDocumentPath,
                CurrentProject.ToAbsolutePath(app.HtmlDocumentPath),
                app.HtmlDocumentPath.IsExpandEnvironmentVariable);
            switch (r)
            {
                case Application.HtmlDocumentPathValidationType.Ok:
                    return null;

                case Application.HtmlDocumentPathValidationType.DirectoryNotFound:
                    return Resources.DirectoryNotFound;

                case Application.HtmlDocumentPathValidationType.PathIsEmpty:
                    return Resources.PleaseSet;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string AccessibleUrlsFilePath(Application app)
        {
            if (IsSkipValidation)
                return null;

            var r = app.ValidateAccessibleUrlsFilePath(
                app.IsReplaceAccessibleUrlsFilePath,
                CurrentProject.ToAbsolutePath(app.AccessibleUrlsFilePath),
                app.AccessibleUrlsFilePath.IsExpandEnvironmentVariable);
            switch (r)
            {
                case Application.AccessibleUrlsFilePathValidationType.Ok:
                    return null;

                case Application.AccessibleUrlsFilePathValidationType.FileNotFound:
                    return Resources.FileNotFound;

                case Application.AccessibleUrlsFilePathValidationType.PathIsEmpty:
                    return Resources.PleaseSet;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string LegalInformationFilePath(Application app)
        {
            if (IsSkipValidation)
                return null;

            var r = app.ValidateLegalInformationFilePath(
                app.IsReplaceLegalInformationFilePath,
                CurrentProject.ToAbsolutePath(app.LegalInformationFilePath),
                app.LegalInformationFilePath.IsExpandEnvironmentVariable);
            switch (r)
            {
                case Application.LegalInformationFilePathValidationType.Ok:
                    return null;

                case Application.LegalInformationFilePathValidationType.EmptyError:
                    return Resources.PleaseSet;

                case Application.LegalInformationFilePathValidationType.FileNotFound:
                    return Resources.FileNotFound;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string SupportedLanguagesInMeta(ApplicationMeta appMeta)
        {
            if (IsSkipValidation)
                return null;

            switch (Application.ValidateSupportedLanguages(appMeta.Application.SupportedLanguages))
            {
                case Application.SupportedLanguagesValidationType.Ok:
                    return null;

                case Application.SupportedLanguagesValidationType.EmptyError:
                    return Resources.SupportedLanguags_Validate_Empty;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string TitleInMeta(ApplicationMeta appMeta)
        {
            if (IsSkipValidation)
                return null;

            switch (Application.ValidateTitle(appMeta.Application.Titles))
            {
                case Application.TitleValidationType.Ok:
                    return null;

                case Application.TitleValidationType.EmptyError:
                    return Resources.Title_Validate_Empty;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string TitleName(string s)
        {
            if (IsSkipValidation)
                return null;

            switch (Title.ValidateName(s))
            {
                case Title.NameValidationType.Ok:
                    return null;

                case Title.NameValidationType.EmptyError:
                    return Resources.PleaseSet;

                case Title.NameValidationType.LengthError:
                    return string.Format(Resources.Title_Name_Validate_Length, Title.MaxNameLength);

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string PublisherName(string s)
        {
            if (IsSkipValidation)
                return null;

            switch (Title.ValidatePublisher(s))
            {
                case Title.PublisherValidationType.Ok:
                    return null;

                case Title.PublisherValidationType.EmptyError:
                    return Resources.PleaseSet;

                case Title.PublisherValidationType.LengthError:
                    return string.Format(Resources.Title_Publisher_Validate_Length,
                        Title.MaxPublisherLength);

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string IconFilePath(Title model)
        {
            if (IsSkipValidation)
                return null;

            var r = model.ValidateIconFilePath(
                model.IsReplaceIcon,
                true,
                CurrentProject.ToAbsolutePath(model.IconFilePath),
                model.IconFilePath.IsExpandEnvironmentVariable,
                Title.OffDeviceIconWidth, Title.OffDeviceIconHeight);
            switch (r)
            {
                case Title.IconFilePathValidationType.Ok:
                    return null;

                case Title.IconFilePathValidationType.EmptyError:
                    return Resources.PleaseSet;

                case Title.IconFilePathValidationType.FileNotFound:
                    return Resources.FileNotFound;

                case Title.IconFilePathValidationType.SizeError:
                    return Resources.IconFilePath_Validate_SizeError;

                case Title.IconFilePathValidationType.FormatError:
                    return Resources.IconFilePath_Validate_FormatError;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string SaveDataSize(bool isUse, long s)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateSaveDataSize(isUse, s);

            switch (r)
            {
                case Application.SaveDataSizeValidationType.Ok:
                    return null;

                case Application.SaveDataSizeValidationType.SizeError:
                    return Resources.SaveData_NeedMoreThan16Kbyte;

                case Application.SaveDataSizeValidationType.AlignError:
                    return Resources.SaveData_16KbyteAlign;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string SaveDataJournalSize(bool isUse, long s)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateSaveDataJournalSize(isUse, s);

            switch (r)
            {
                case Application.SaveDataJournalSizeValidationType.Ok:
                    return null;

                case Application.SaveDataJournalSizeValidationType.SizeError:
                    return Resources.SaveDataJournal_NeedMoreThan16Kbyte;

                case Application.SaveDataJournalSizeValidationType.AlignError:
                    return Resources.SaveData_16KbyteAlign;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string SaveDataSizeMax(bool isUse, long s)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateSaveDataSizeMax(isUse, s);

            switch (r)
            {
                case Application.SaveDataSizeMaxValidationType.Ok:
                    return null;
                case Application.SaveDataSizeMaxValidationType.AlignError:
                    return Resources.SaveDataSizeMax_1MbyteAlign;
                case Application.SaveDataSizeMaxValidationType.SizeError:
                    return Resources.SaveDataSizeMax_NeedMoreThan1Mbyte;
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string TemporaryStorageSize(bool isUse, long s)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateTemporaryStorageSize(isUse, s);

            switch (r)
            {
                case Application.TemporaryStorageSizeValidationType.Ok:
                    return null;
                case Application.TemporaryStorageSizeValidationType.AlignError:
                    return Resources.SaveData_16KbyteAlign;
                case Application.TemporaryStorageSizeValidationType.MinSizeError:
                    return string.Format(Resources.SizeAtLeastAlignKBytes, 16);
                case Application.TemporaryStorageSizeValidationType.MaxSizeError:
                    return string.Format(Resources.SizeLimitExceeded, "4000MiB");
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string CacheStorageSize(bool isUse, long s)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateCacheStorageSize(isUse, s);

            switch (r)
            {
                case Application.CacheStorageSizeValidationType.Ok:
                    return null;
                case Application.CacheStorageSizeValidationType.AlignError:
                    return Resources.SaveData_16KbyteAlign;
                case Application.CacheStorageSizeValidationType.SizeError:
                    return string.Format(Resources.SizeAtLeastAlignKBytes, 16);
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string AocIndex(ulong s, AocContent ignore, ObservableCollection<AocContent> allContents)
        {
            if (IsSkipValidation)
                return null;

            var r = AocContent.ValidateIndex(s, ignore, allContents);
            switch (r)
            {
                case AocContent.IndexValidationType.Ok:
                    return null;

                case AocContent.IndexValidationType.RangeOver:
                    return string.Format(Resources.AocIndex_Validate_RangeOver,
                        Constants.AocIndexMinimum, Constants.AocIndexMaximum);

                case AocContent.IndexValidationType.AlreadyExist:
                    return Resources.AocIndex_Validate_AlreadyExists;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string AocTag(string s, AocContent ignore, ObservableCollection<AocContent> allContents)
        {
            if (IsSkipValidation)
                return null;

            var r = AocContent.ValidateTag(s, ignore, allContents);
            switch (r)
            {
                case AocContent.TagValidationType.Ok:
                    return null;

                case AocContent.TagValidationType.AlreadyExist:
                    return Resources.AocTag_Validate_AlreadyExists;

                case AocContent.TagValidationType.EmptyError:
                    return Resources.AocTag_Validate_Empty;

                case AocContent.TagValidationType.LengthError:
                    return string.Format(Resources.Aoc_Tag_Validate_Length, AocContent.MaxTagLength);

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string AocDataPath(AocContent content)
        {
            if (IsSkipValidation)
                return null;

            // 環境変数を展開が有効な場合、パスのチェックはスキップする
            if (content.DataPath.IsExpandEnvironmentVariable)
                return null;

            var s = _diContainer.GetInstance<Project>().ToAbsolutePath(content.DataPath.Path);
            var r = AocContent.ValidateDataPath(s);
            switch (r)
            {
                case AocContent.DataPathValidationType.Ok:
                    return null;

                case AocContent.DataPathValidationType.DirectoryNotFound:
                    return Resources.DirectoryNotFound;

                case AocContent.DataPathValidationType.PathIsEmpty:
                    return Resources.PleaseSet;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string AocMetaContents(ObservableCollection<AocContent> contents)
        {
            if (IsSkipValidation)
                return null;

            var r = AocMeta.ValidateContents(contents);
            switch (r)
            {
                case AocMeta.ContentsValidationType.Ok:
                    return null;

                case AocMeta.ContentsValidationType.EmptyError:
                    return Resources.AocContents_Validate_Empty;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string BcatPassphrase(string s, bool isUseBcat)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateBcatPassphrase(s, isUseBcat);
            switch (r)
            {
                case Application.BcatPassphraseValidationType.Ok:
                    return null;

                case Application.BcatPassphraseValidationType.FormatError:
                    return Resources.BcatPassphrase_Validate_FormatError;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string BcatDeliveryCacheStorageSize(long s, bool isUseBcat)
        {
            if (IsSkipValidation)
                return null;

            var r = Application.ValidateBcatDeliveryCacheStorageSize(s, isUseBcat);
            switch (r)
            {
                case Application.BcatDeliveryCacheStorageSizeValidationType.Ok:
                    return null;

                case Application.BcatDeliveryCacheStorageSizeValidationType.AlignError:
                    return Resources.BcatDeliveryCacheStorageSize_Validate_AlignError;

                case Application.BcatDeliveryCacheStorageSizeValidationType.SizeError:
                    return Resources.BcatDeliveryCacheStorageSize_Validate_SizeError;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string UserAccountSaveDataOperationCopy(CopyType copy)
        {
            if (IsSkipValidation)
                return null;

            var r = SaveDataOperation.ValidateCopy(copy);
            switch (r)
            {
                case SaveDataOperation.CopyValidationType.Ok:
                    return null;
                case SaveDataOperation.CopyValidationType.UnspecifiedNotAllowError:
                    return Resources.CopyValidationType_UnspecifiedNotAllowError;
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string UserAcountSaveDataOperationRollback(RollbackType rollback, SaveDataOperation op)
        {
            if (IsSkipValidation)
                return null;

            var r = SaveDataOperation.ValidateRollback(op.Copy, rollback, op.RollbackTerm);
            switch (r)
            {
                case SaveDataOperation.RollbackValidationType.Ok:
                    return null;
                case SaveDataOperation.RollbackValidationType.CopyAllowedTermError:
                    return Resources.RollbackValidationType_CopyAllowedTermError;
                case SaveDataOperation.RollbackValidationType.InvalidRollbackTermError:
                    return Resources.RollbackValidationType_InvalidRollbackTermError;
                case SaveDataOperation.RollbackValidationType.UnspecifiedNotAllowError:
                    return Resources.RollbackValidationType_UnspecifiedNotAllowError;
                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public string FilterDescriptionFilePath(Application app)
        {
            if (IsSkipValidation)
                return null;

            var r = app.ValidateFilterDescriptionFilePath(
                app.IsUseFilterDescriptionFilePath,
                CurrentProject.ToAbsolutePath(app.FilterDescriptionFilePath));
            switch (r)
            {
                case Application.FilterDescriptionFileValidationType.Ok:
                    return null;

                case Application.FilterDescriptionFileValidationType.PathIsEmpty:
                    return Resources.PleaseSet;

                case Application.FilterDescriptionFileValidationType.FileNotFound:
                    return Resources.FileNotFound;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }
    }
}
