﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.ObjectModel;
using System.IO;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.Properties;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.MainWindow.ComparisonPanel.ParamTreeBuilder
{
    public class AocNspParamTreeBuilder : ParamTreeBuilderBase
    {
        public AocNspParamTreeBuilder(Container diContainer, ObservableCollection<ComparisonPanelVm.Target> targets)
            : base(diContainer, targets)
        {
        }

        public override void Dispose()
        {
        }

        public override ObservableCollection<Param> UpdataParams(Config config)
        {
            var overview = new Param(config, nameof(Resources.AocOverview));
            var aoc = new Param(config, nameof(Resources.AddOnContent));

            UpdataParams_Overview(config, overview);
            UpdataParams_Aoc(config, aoc);

            return new ObservableCollection<Param>
            {
                overview,
                aoc
            };
        }

        private void UpdataParams_Overview(Config config, Param dst)
        {
            var appId = new Param(config, nameof(Resources.ApplicationId_Caption),
                () => AocMeta0.ApplicationId.ToHex(),
                () => AocMeta1.ApplicationId.ToHex());

            dst.AddChild(appId);
        }

        private void UpdataParams_Aoc(Config config, Param dst)
        {
            var contents0 = new AocContent[Constants.AocIndexMaximum + 1];
            var contents1 = new AocContent[Constants.AocIndexMaximum + 1];

            foreach (var c in AocMeta0.Contents)
                contents0[c.Index] = c;

            foreach (var c in AocMeta1.Contents)
                contents1[c.Index] = c;

            var count0 = 0;
            var count1 = 0;

            for (var i = Constants.AocIndexMinimum; i <= Constants.AocIndexMaximum; ++i)
            {
                var c0 = contents0[i];
                var c1 = contents1[i];

                if (c0 == null && c1 == null)
                    continue;

                var item = new Param(config, $"#{i}");
                {
                    {
                        var tag = new Param(config, nameof(Resources.AocContnetTag_Caption),
                            () => c0?.Tag, c0 != null,
                            () => c1?.Tag, c1 != null);

                        var relVer = new Param(config, nameof(Resources.ReleaseVersion_Caption),
                            () => c0?.ReleaseVersion.ToString(), c0 != null,
                            () => c1?.ReleaseVersion.ToString(), c1 != null);

                        var reqAppRelVer = new Param(config, nameof(Resources.RequiredApplicationReleaseVersion_Caption),
                            () => c0?.RequiredApplicationReleaseVersion.ToString(), c0 != null,
                            () => c1?.RequiredApplicationReleaseVersion.ToString(), c1 != null);

                        item.AddChild(tag);
                        item.AddChild(relVer);
                        item.AddChild(reqAppRelVer);
                    }

                    if (HasDigest(c0, c1))
                    {
                        var digest = new Param(config, nameof(Resources.RomDigest_Caption),
                            () => c0?.Digest, c0 != null,
                            () => c1?.Digest, c1 != null);
                        item.AddChild(digest);
                    }

                    if (HasSize(c0, c1))
                    {
                        var size = new Param(config, nameof(Resources.ContentDataSize),
                            () => c0?.ContentDataSize.ToReadableSizeString(), c0 != null,
                            () => c1?.ContentDataSize.ToReadableSizeString(), c1 != null);
                        item.AddChild(size);
                    }
                    else
                    {
                        var dataPath = new Param(config, nameof(Resources.DataPath_Caption),
                            new Param.ResultValue
                            {
                                IsIdentical = FileSystemHelper.IsEqualDirectory(
                                    Project0.ToAbsolutePath(c0?.DataPath.Path),
                                    Project1.ToAbsolutePath(c1?.DataPath.Path)
                                ),
                                HasValue0 = Directory.Exists(c0?.DataPath.Path),
                                HasValue1 = Directory.Exists(c1?.DataPath.Path)
                            },
                            c0 != null,
                            c1 != null,
                            TrimSecond);
                        item.AddChild(dataPath);
                    }

                    dst.AddChild(item);
                }

                if (c0 != null)
                    ++count0;

                if (c1 != null)
                    ++count1;

                if (count0 == AocMeta0.Contents.Count &&
                    count1 == AocMeta1.Contents.Count)
                    break;
            }
        }

        private bool HasDigest(AocContent c0, AocContent c1) =>
            string.IsNullOrEmpty(c0?.Digest) == false && string.IsNullOrEmpty(c1?.Digest) == false;

        private bool HasSize(AocContent c0, AocContent c1) =>
            c0?.ContentDataSize > 0 && c1?.ContentDataSize > 0;
    }
}
