﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Text.RegularExpressions;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Foundation
{
    public static class SerializeExtensions
    {
        public static string ToHex(this ulong self, int length = 16)
        {
            return "0x" + self.ToString("x" + length);
        }

        public static string ToHex(this long self, int length = 16)
        {
            return "0x" + self.ToString("x" + length);
        }

        public static string ToUpperHex(this ulong self, int length = 16)
        {
            return "0x" + self.ToString("X" + length);
        }

        public static string ToUpperHexGrouping(this ulong self, int length = 16, int groupLength = 4)
        {
            return "0x" + Regex.Replace(self.ToString("X" + length), "(.{4})(?=.)", "$1 ");
        }

        public static ulong ToUlong(this string self)
        {
            try
            {
                return Convert.ToUInt64(self, 16);
            }
            catch
            {
                throw new InvalidDataException();
            }
        }

        public static bool TryToUlong(this string self, out ulong value)
        {
            try
            {
                value = Convert.ToUInt64(self, 16);
                return true;
            }
            catch
            {
                value = 0UL;
                return false;
            }
        }

        public static bool TryToLong(this string self, out long value)
        {
            try
            {
                value = Convert.ToInt64(self, 16);
                return true;
            }
            catch
            {
                value = 0L;
                return false;
            }
        }

        public static long ToLong(this string self)
        {
            try
            {
                return Convert.ToInt64(self, 16);
            }
            catch
            {
                throw new InvalidDataException();
            }
        }

        public static bool IsUlong(this string self)
        {
            try
            {
                self.ToUlong();
                return true;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// 対象オブジェクトのディープコピークローンを取得します。
        /// Force.DeepCloner 版が望む動作をしない場合利用します。
        /// Force.DeepCloner 版に比べ数十倍以上低速です。
        /// </summary>
        /// <typeparam name="T">対象オブジェクトの型</typeparam>
        /// <param name="target">コピー対象オブジェクトを指定します。</param>
        /// <returns>ディープコピーのクローンを返します。</returns>
        /// <remarks>このメソッドでディープコピーするには、対象クラスがシリアライズ可能である必要があります。</remarks>
        public static T DeepCloneByYamlSerializer<T>(this T target) where T : class
        {
            if (target == null)
                return default(T);

            using (var tw = new StringWriter())
            {
                new SerializerBuilder().EmitDefaults().Build().Serialize(tw, target);

                var yaml = tw.ToString();

                return new DeserializerBuilder().IgnoreUnmatchedProperties().Build().Deserialize<T>(yaml);
            }
        }
    }
}
