﻿using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using BezelEditor.Foundation;
using System.Threading.Tasks;
using System.Xml.Linq;
using Nintendo.Authoring.AuthoringEditor.Foundation;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public static class TypeHelper
    {
        public static AppModeType ToAppMode(Project project)
        {
            var path = project.TargetFilePath;

            var extension = path.Extension.ToLowerInvariant();
            if (extension == ".narp")
                return AppModeType.Project;
            if (extension == ".nmeta" || extension == ".meta")
                return MakeMetaAppModeType(path);

            var isAppNsp = project.ApplicationContentMeta != null && project.PatchContentMeta == null;
            if (isAppNsp)
                return AppModeType.ApplicationNsp;

            var isPatchNsp = project.PatchContentMeta != null;
            if (isPatchNsp)
                return AppModeType.PatchNsp;

            var isAocNsp = project.AocContentMetas != null;
            if (isAocNsp)
                return AppModeType.AocNsp;

            throw new ArgumentOutOfRangeException();
        }

        public static AppModeType ToAppMode(PathString path)
        {
            var type = FilePathToImportableFileType(path);
            if (type.HasValue == false)
                throw new ArgumentOutOfRangeException(nameof(path), path, null);

            switch (type)
            {
                case ImportableFileType.Project:
                    return AppModeType.Project;

                case ImportableFileType.Meta:
                    return MakeMetaAppModeType(path);

                case ImportableFileType.Nsp:
                    return MakeNspAppModeType(path);

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public static AppModeType MakeMetaAppModeType(PathString path)
        {
            switch (DetectMetaKind(path))
            {
                case MetaKind.Application:
                    return AppModeType.ApplicationMeta;

                case MetaKind.AddOnContent:
                    return AppModeType.AocMeta;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        private static AppModeType MakeNspAppModeType(PathString path)
        {
            var nspFile = Project.GetNspFile(path);
            var metaType = Task.Run(async () => await NspImporter.DetectContentMetaTypeAsync(nspFile).ConfigureAwait(false)).Result;

            switch (metaType)
            {
                case ContentMetaType.Application:
                    return AppModeType.ApplicationNsp;

                case ContentMetaType.AddOnContent:
                    return AppModeType.AocNsp;

                case ContentMetaType.Patch:
                    return AppModeType.PatchNsp;

                default:
                    throw new ArgumentOutOfRangeException();
            }
        }

        public static ImportableFileType? FilePathToImportableFileType(string path)
        {
            var ext = Path.GetExtension(path);
            if (string.IsNullOrEmpty(ext))
                return null;

            return FileTypeHelper.ToImportableFileType(ext);
        }

        public static ImportableFileType ToImportable(OutputTypes outputType)
        {
            switch (outputType)
            {
                case OutputTypes.Nsp:
                    return ImportableFileType.Nsp;

                default:
                    throw new ArgumentOutOfRangeException(nameof(outputType), outputType, null);
            }
        }

        public static MetaKind DetectMetaKind(string metaFilePath)
        {
            if (string.IsNullOrEmpty(metaFilePath))
                return MetaKind.Unknown;

            try
            {
                var doc = XDocument.Parse(File.ReadAllText(metaFilePath));

                return doc.Descendants(nameof(MetaKind.Application)).Any()
                    ? MetaKind.Application
                    : MetaKind.AddOnContent;
            }
            catch
            {
                return MetaKind.Unknown;
            }
        }

        public static RootMetaKind DetectRootMetaKind(string metaFilePath)
            => DetectRootMetaKindFromXmlText(File.ReadAllText(metaFilePath));

        public static RootMetaKind DetectRootMetaKindFromXmlText(string xmlText)
        {
            if (string.IsNullOrEmpty(xmlText))
                return RootMetaKind.Unknown;

            try
            {
                var doc = XDocument.Parse(xmlText);

                return doc.Root?.Name == nameof(RootMetaKind.NintendoSdkMeta)
                    ? RootMetaKind.NintendoSdkMeta
                    : RootMetaKind.Meta;
            }
            catch
            {
                return RootMetaKind.Unknown;
            }
        }
    }
}
