﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Extentions;
using BezelEditor.Foundation.Utilities;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using SimpleInjector;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class NspReplacer
    {
        public class ReplaceOption
        {
            public string InputNspPath { get; set; }
            public string OutputNspPath { get; set; }
            public Project Project { get; set; }

            public Action<int> ProgressChanged { get; set; }
            public CancellationToken CancellationToken { get; set; }
        }

        private class ReplaceNca
        {
            public string SourceContentMetaId { get; set; }
            public ContentType Type { get; set; }
            public string LocalPath { get; set; }
        }

        private enum ReplaceOperation
        {
            Replace,
            Add,
            Remove
        }

        private class ReplaceFile
        {
            public ReplaceOperation Operation { get; set; } = ReplaceOperation.Replace;
            public string Path { get; set; }
            public string LocalPath { get; set; }

            public override string ToString()
            {
                switch (Operation)
                {
                    case ReplaceOperation.Add:
                        return $"add:{Path}\t{LocalPath}";
                    case ReplaceOperation.Remove:
                        return $"del:{Path}\tnull";
                    case ReplaceOperation.Replace:
                        return $"{Path}\t{LocalPath}";
                    default:
                        throw new ArgumentException(nameof(Operation));
                }
            }
        }

        private enum ReplaceIconType
        {
            Raw,
            Nx
        }

        private class ReplaceRootIconFile : ReplaceFile
        {
            public LanguageType Language { get; set; }
            public ReplaceIconType Type { get; set; }

            public override string ToString()
            {
                switch (Operation)
                {
                    case ReplaceOperation.Add:
                    {
                        var iconType = Type.ToString().ToLowerInvariant();
                        return $"null.icon@{Language}@{iconType}\t{LocalPath}";
                    }
                    case ReplaceOperation.Remove:
                        return $"del:{Path}\tnull";
                    case ReplaceOperation.Replace:
                        return $"{Path}\t{LocalPath}";
                    default:
                        throw new ArgumentException(nameof(Operation));
                }
            }
        }

        private class NspReplaceFilePath
        {
            public ContentType ContentType { get; set; }
            public string SourcePath { get; set; }
            public string SourceContentId { get; set; }
            public string TargetPath { get; set; }
            public string TargetContentId { get; set; }
        }

        private class NspReplacePathBuilder
        {
            public ContentMeta SourceContentMeta { get; set; }
            public ContentMeta TargetContentMeta { get; set; }

            public NspReplacePathBuilder()
            {
            }

            public NspReplacePathBuilder(ContentMeta contentMeta)
            {
                SourceContentMeta = contentMeta;
                TargetContentMeta = contentMeta;
            }

            public NspReplaceFilePath Build(ContentType contentType, string filePathInNsp)
            {
                var sourceContentId = SourceContentMeta.GetContentId(contentType);
                if (sourceContentId == null)
                    return null;
                // ファイル追加の場合、targetContentId は存在しないので null
                var targetContentId = TargetContentMeta.GetContentId(contentType);
                return new NspReplaceFilePath
                {
                    ContentType = contentType,
                    SourcePath = string.Format(filePathInNsp, sourceContentId),
                    SourceContentId = sourceContentId,
                    TargetPath = string.IsNullOrEmpty(targetContentId) ?
                        null :
                        string.Format(filePathInNsp, targetContentId),
                    TargetContentId = targetContentId,
                };
            }
        }

        private class NspReplaceEntry
        {
            public ConcurrentBag<ReplaceFile> Files { get; } = new ConcurrentBag<ReplaceFile>();
            public ConcurrentBag<ReplaceRootIconFile> IconFiles { get; } = new ConcurrentBag<ReplaceRootIconFile>();
            public ConcurrentBag<ReplaceNca> NcaContents { get; } = new ConcurrentBag<ReplaceNca>();
            public DisposableDirectory TempDirectory { get; set; }

            public void RemoveIcon(Title target, NspReplacePathBuilder pathBuilder)
            {
                IconFiles.Add(new ReplaceRootIconFile
                {
                    Language = target.Language,
                    Operation = ReplaceOperation.Remove,
                    Type = ReplaceIconType.Raw,
                    Path = target.NspRawIconFilePath,
                });
                IconFiles.Add(new ReplaceRootIconFile
                {
                    Language = target.Language,
                    Operation = ReplaceOperation.Remove,
                    Type = ReplaceIconType.Nx,
                    Path = target.NspNxIconFilePath,
                });
                Files.Add(new ReplaceFile
                {
                    Operation = ReplaceOperation.Remove,
                    Path = pathBuilder.Build(ContentType.Control, $"{{0}}.nca/fs0/icon_{target.Language}.dat").TargetPath
                });
            }

            public Task AddIcon(INspFile sourceNspFile, Title source, NspReplacePathBuilder pathBuilder)
            {
                return Task.WhenAll(
                    AddReplaceTarget(sourceNspFile, source.NspNxIconFilePath, x => IconFiles.Add(new ReplaceRootIconFile
                    {
                        Language = source.Language,
                        Operation = ReplaceOperation.Add,
                        Type = ReplaceIconType.Nx,
                        LocalPath = x
                    })),
                    AddReplaceTarget(sourceNspFile, source.NspRawIconFilePath, x => IconFiles.Add(new ReplaceRootIconFile
                    {
                        Language = source.Language,
                        Operation = ReplaceOperation.Add,
                        Type = ReplaceIconType.Raw,
                        LocalPath = x
                    })),
                    Add(ReplaceOperation.Add, sourceNspFile,
                        pathBuilder.Build(ContentType.Control, $"{{0}}.nca/fs0/icon_{source.Language}.dat"))
                );
            }

            public Task ReplaceIcon(INspFile sourceNspFile, Title source, Title target, NspReplacePathBuilder pathBuilder)
            {
                Debug.Assert(source.Language == target.Language);
                return Task.WhenAll(
                    AddReplaceTarget(sourceNspFile, source.NspNxIconFilePath, x => IconFiles.Add(new ReplaceRootIconFile
                    {
                        Language = target.Language,
                        Operation = ReplaceOperation.Replace,
                        Type = ReplaceIconType.Nx,
                        Path = target.NspNxIconFilePath,
                        LocalPath = x
                    })),
                    AddReplaceTarget(sourceNspFile, source.NspRawIconFilePath, x => IconFiles.Add(new ReplaceRootIconFile
                    {
                        Language = target.Language,
                        Operation = ReplaceOperation.Replace,
                        Type = ReplaceIconType.Raw,
                        Path = target.NspRawIconFilePath,
                        LocalPath = x
                    })),
                    Add(ReplaceOperation.Replace, sourceNspFile,
                        pathBuilder.Build(ContentType.Control, $"{{0}}.nca/fs0/icon_{target.Language}.dat"))
                );
            }

            public Task Add(ReplaceOperation operation, INspFile sourceNspFile, NspReplaceFilePath filePath)
            {
                return Add(operation, sourceNspFile, filePath, x => x);
            }

            public Task Add(ReplaceOperation operation, INspFile sourceNspFile, NspReplaceFilePath filePath, Func<string, string> extractedAction)
            {
                if (filePath == null)
                    return Task.CompletedTask;
                return AddReplaceTarget(
                    sourceNspFile,
                    filePath.SourcePath,
                    x =>
                    {
                        Files.Add(new ReplaceFile
                        {
                            Operation = operation,
                            Path = filePath.TargetPath,
                            LocalPath = extractedAction(x)
                        });
                    });
            }

            public Task AddOrReplaceNca(INspFile sourceNspFile, NspReplaceFilePath filePath)
            {
                if (filePath == null)
                    return Task.CompletedTask;
                return AddReplaceTarget(
                    sourceNspFile,
                    filePath.SourcePath,
                    x => NcaContents.Add(new ReplaceNca
                    {
                        SourceContentMetaId = filePath.TargetContentId,
                        Type = filePath.ContentType,
                        LocalPath = x
                    }));
            }

#if DEBUG
            private readonly HashSet<string> _extractingFileSet = new HashSet<string>();
#endif

            private async Task AddReplaceTarget(
                INspFile nspFile,
                string pathInNsp,
                Action<string> extractAction)
            {
#if DEBUG
                Debug.Assert(_extractingFileSet.Contains(pathInNsp) == false);
                _extractingFileSet.Add(pathInNsp);
#endif
                var extractedLocalPath = Path.Combine(TempDirectory.RootPath, Path.GetFileName(pathInNsp));
                await nspFile.ExtractAsync(pathInNsp, extractedLocalPath).ConfigureAwait(false);
                extractAction(extractedLocalPath);
            }
        }

        private readonly Container _diContainer;

        public NspReplacer(Container diContainer)
        {
            _diContainer = diContainer;
        }

        public async Task<NspHandleResultType> ReplaceNspAsync(ReplaceOption option)
        {
            var result = TestNspReplaceError(option);
            if (result != null)
                return result;

            return await ReplaceNspMetaAsync(option).ConfigureAwait(false);
        }

        private async Task<NspHandleResultType> ReplaceNspMetaAsync(ReplaceOption option)
        {
            using (var tempDir = new DisposableDirectory())
            using (var inputNspProject = Project.Import(_diContainer, ImportableFileType.Nsp, option.InputNspPath))
            using (var outputNspProject = option.Project.DeepClone())
            {
                // 置き換える nsp に含まれるオフライン HTML ドキュメントとソフトリーガル情報を抽出
                await inputNspProject.ExtraResourceImporter.ReadHtmlDocumentAsync().ConfigureAwait(false);
                await inputNspProject.ExtraResourceImporter.ReadLegalInformationAsync().ConfigureAwait(false);

                // プロジェクトフォルダは clone 対象外なので再設定
                outputNspProject.ProjectDirectory = option.Project.ProjectDirectory;
                FixupMetaResourcesPath(outputNspProject);

                // 入力内容から中間 nsp を生成
                var intermediateNspPath = Path.Combine(tempDir.RootPath, "intermediate.nsp");
                {
                    var r = await CreateIntermediateNspAsync(tempDir, intermediateNspPath, outputNspProject,
                        inputNspProject, option.CancellationToken).ConfigureAwait(false);
                    if (r.Result != NspHandleResult.Ok)
                        return r;
                }

                var replaceEntry = new NspReplaceEntry()
                {
                    TempDirectory = tempDir,
                };

                // 置き換え対象のファイルを準備
                {
                    var r = await PrepareReplaceFiles(
                        option.InputNspPath,
                        intermediateNspPath,
                        inputNspProject,
                        outputNspProject,
                        replaceEntry
                    ).ConfigureAwait(false);
                    if (r != null)
                        return r;
                }

                // nsp の内容差し替え
                {
                    var r = await ReplaceAsync(
                        inputNspProject,
                        tempDir,
                        option.ProgressChanged,
                        option.OutputNspPath,
                        option.InputNspPath,
                        replaceEntry,
                        option.CancellationToken
                    ).ConfigureAwait(false);
                    if (r.Result != NspHandleResult.Ok)
                        return r;
                }

                return new NspHandleResultType { Result = NspHandleResult.Ok };
            }
        }

        private static async Task<NspHandleResultType> PrepareReplaceFiles(
            string inputNspPath,
            string intermediateNspPath,
            Project inputNspProject,
            Project outputNspProject,
            NspReplaceEntry replaceEntry)
        {
            var inputNspFile = new NspFile(inputNspPath)
            {
                EnumerationMode = NspFileEnumeration.RootOnly
            };

            var intermediateNspFile = new NspFile(intermediateNspPath)
            {
                EnumerationMode = NspFileEnumeration.RootOnly
            };

            ApplicationContentMeta intermediateContentMeta;
            {
                var intermediateContentMetas =
                    await NspImporter.ReadAllContentMeta<ApplicationContentMeta>(intermediateNspFile)
                        .WhenAll()
                        .ConfigureAwait(false);
                intermediateContentMeta = intermediateContentMetas.FirstOrDefault();
                if (intermediateContentMeta == null)
                {
                    return new NspHandleResultType
                    {
                        Result = NspHandleResult.ExtractError,
                        ErrorMessages = new[] { "File not found: *.cnmt.xml" }
                    };
                }
            }

            var replaceFilePath = new NspReplacePathBuilder
            {
                SourceContentMeta = intermediateContentMeta,
                TargetContentMeta = inputNspProject.ApplicationContentMeta,
            };
            var tasks = new List<Task>();

            // アイコン
            AddIcons(intermediateNspFile, inputNspProject, outputNspProject, replaceFilePath, replaceEntry, tasks);
            // バイナリ形式の ContentMeta (*.cnmt)
            AddBinaryContentMeta(inputNspFile, inputNspProject, replaceEntry, tasks, outputNspProject.Meta);
            // オフライン HTML & Web アプレットからアクセス可能な URL のリスト
            tasks.Add(replaceEntry.AddOrReplaceNca(
                intermediateNspFile,
                replaceFilePath.Build(ContentType.HtmlDocument, "{0}.nca")));
            // リーガル情報
            tasks.Add(replaceEntry.AddOrReplaceNca(
                intermediateNspFile,
                replaceFilePath.Build(ContentType.LegalInformation, "{0}.nca")));
            // アプリケーション管理プロパティ
            tasks.Add(replaceEntry.Add(
                ReplaceOperation.Replace,
                intermediateNspFile,
                replaceFilePath.Build(ContentType.Control, "{0}.nca/fs0/control.nacp")));

            await Task.WhenAll(tasks).ConfigureAwait(false);

            return null;
        }

        private class TitleComparer : IEqualityComparer<Title>
        {
            public bool Equals(Title x, Title y) => x.Language == y.Language;

            public int GetHashCode(Title obj) => 0;
        }

        private static void AddIcons(NspFile intermediateNspFile,
            Project inputNspProject,
            Project outputNspProject,
            NspReplacePathBuilder pathBuilder,
            NspReplaceEntry replaceEntry,
            List<Task> tasks)
        {
            var inputTitles = inputNspProject.Meta.Application.Titles;   // 置き換える nsp
            var outputTitles = outputNspProject.Meta.Application.Titles; // AE 上で変更した nsp
            var iconUpdatedTitlesLanguageSet = outputTitles.Where(x => x.IsReplaceIcon).Select(x => x.Language).ToHashSet();

            // AE 上で変更して削除されたアイコンの設定状態を対象の nsp に反映
            {
                var removedTitles = inputTitles
                    .Except(outputTitles, new TitleComparer())
                    .Where(HasNspExtractedIcon);
                foreach (var removeTitle in removedTitles)
                    replaceEntry.RemoveIcon(removeTitle, pathBuilder);
            }

            // 中間 nsp に含まれるアイコンのリストを取得する:
            // AE 上でのアイコンの変更・追加に関わらず nsp 作成のため全言語のタイトルにアイコンが設定されているので、
            // AE 上で変更した nsp の情報を元に、追加・変更されたアイコンの情報のみを取得
            var sourceIcons = NspImporter.FindNspIcons(intermediateNspFile)
                .Where(x => iconUpdatedTitlesLanguageSet.Contains(x.Language)).ToArray();

            // AE 上で変更・追加したアイコンの設定状態を対象の nsp に反映
            var updateTitles = outputTitles
                .Select(x => new
                {
                    Input = inputTitles.FirstOrDefault(y => y.Language == x.Language),
                    Output = x,
                    IntermediateNspIcon = sourceIcons.FirstOrDefault(y => y.Language == x.Language)
                })
                .Where(x => x.IntermediateNspIcon != null);

            foreach (var title in updateTitles)
            {
                title.Output.NspNxIconFilePath = title.IntermediateNspIcon.NxFilePath;
                title.Output.NspRawIconFilePath = title.IntermediateNspIcon.RawFilePath;

                // 置き換える nsp にタイトルが存在しない or
                // 置き換える nsp にタイトルは存在するがアイコンが未設定
                if (title.Input == null || HasNspExtractedIcon(title.Input) == false)
                {
                    // AE 上で設定したアイコンを置き換える nsp に新規追加
                    tasks.Add(replaceEntry.AddIcon(intermediateNspFile, title.Output, pathBuilder));
                }
                else
                {
                    // AE 上で設定したアイコンを置き換える nsp の設定済みアイコンと置換
                    tasks.Add(replaceEntry.ReplaceIcon(intermediateNspFile, title.Output, title.Input, pathBuilder));
                }
            }
        }

        private static void AddBinaryContentMeta(INspFile inputNspFile,
            Project inputNspProject,
            NspReplaceEntry replaceEntry,
            List<Task> tasks,
            ApplicationMeta outputMeta)
        {
            var originalReleaseVersion = inputNspProject.Meta.Application.ReleaseVersion;
            if (originalReleaseVersion == outputMeta.Application.ReleaseVersion)
                return;

            // リリースバージョンの変更
            var contentMeta = inputNspProject.ApplicationContentMeta;
            var replaceFilePath = new NspReplacePathBuilder(contentMeta);
            var appId = inputNspProject.Meta.Core.ApplicationId.ToString("x16");
            // *.cnmt の置き換えは入出力に使用するnspが同一
            tasks.Add(replaceEntry.Add(
                ReplaceOperation.Replace,
                inputNspFile,
                replaceFilePath.Build(ContentType.Meta, $"{{0}}.cnmt.nca/fs0/Application_{appId}.cnmt"),
                x =>
                {
                    using (var fs = new FileStream(x, FileMode.Open, FileAccess.ReadWrite, FileShare.ReadWrite))
                    using (var writer = new BinaryWriter(fs))
                    {
                        var releaseVersion =
                            (contentMeta.Version & 0xffff) |
                            ((uint)outputMeta.Application.ReleaseVersion << 16);

                        // struct ContentMetaInfo
                        // {
                        //     Bit64 id;
                        //     uint32_t version; // ← ここを直接書き換える
                        // ...
                        writer.Seek(sizeof(ulong), SeekOrigin.Begin);
                        writer.Write(BitConverter.GetBytes(releaseVersion));
                    }
                    return x;
                }));
        }

        private async Task<NspHandleResultType> CreateIntermediateNspAsync(
            DisposableDirectory tempDir,
            PathString intermediateNspdPath,
            Project newNspProject,
            Project originalNspProject,
            CancellationToken cancellationToken)
        {
            var originalAppMeta = originalNspProject.Meta;

            if (newNspProject.Meta.Application.IsReplaceHtmlDocumentPath == false &&
                Directory.Exists(originalAppMeta.Application.OriginalHtmlDocumentPath))
            {
                // 置換元 nsp に存在する オフライン HTML を使用
                newNspProject.Meta.Application.IsReplaceHtmlDocumentPath = true;
                newNspProject.Meta.Application.HtmlDocumentPath =
                    new ExpandablePath(originalAppMeta.Application.OriginalHtmlDocumentPath);
            }

            if (newNspProject.Meta.Application.IsReplaceAccessibleUrlsFilePath == false &&
                File.Exists(originalAppMeta.Application.OriginalAccessibleUrlsFilePath))
            {
                // 置換元 nsp に存在する Web アプレットからアクセス可能な URL のリスト を使用
                newNspProject.Meta.Application.IsReplaceAccessibleUrlsFilePath = true;
                newNspProject.Meta.Application.AccessibleUrlsFilePath =
                    new ExpandablePath(originalAppMeta.Application.OriginalAccessibleUrlsFilePath);
            }

            // 仮の nmeta を出力
            var metaPath = Path.Combine(tempDir.RootPath, "intermediate.nmeta").ToPathString();
            newNspProject.OutputAppMetaXmlFileForAuthoringTool(metaPath);

            var descPath = NintendoSdkHelper.ApplicationDescFilePath;

            // 仮のプログラムディレクトリを用意
            var programDir = Path.Combine(tempDir.RootPath, "program")
                .ToDirectoryPathString()
                .DirectoryWithoutLastSeparator.ToPathString();

            // 空のプログラムとしてビルドするための *.npdm を用意
            {
                var errorResult = CreateIntermediateProgramDir(programDir, metaPath, descPath);
                if (errorResult != null)
                    return errorResult;
            }

            // AuthoringTool で control.nacp を得るためにのアプリ nsp を生成する
            var args = "creatensp --type Application" +
                $" -o {intermediateNspdPath.DirectoryWithoutLastSeparator.ToPathString().SurroundDoubleQuotes}" +
                $" --program {programDir.SurroundDoubleQuotes}" +
                $" --meta {metaPath.SurroundDoubleQuotes}" +
                $" --desc {descPath.SurroundDoubleQuotes}";

            // nsp の unpublishable error を回避するため、 nmeta にアイコンの設定がない場合は仮アイコンを設定
            SuppressNmetaUnpublishableError(metaPath, tempDir);

            using (var job = AuthoringToolWrapper.Run(args))
            {
                try
                {
                    return await job.WaitForExitAsync(x => job.MakeResultType(), cancellationToken).ConfigureAwait(false);
                }
                catch (TaskCanceledException)
                {
                    return job.MakeResultType(NspHandleResult.Canceled);
                }
            }
        }

        private static NspHandleResultType CreateIntermediateProgramDir(PathString programDir, PathString metaPath, PathString descPath)
        {
            if (!Directory.Exists(programDir))
                Directory.CreateDirectory(programDir);

            var makeMetaExe = NintendoSdkHelper.MakeMetaExe;
            if (!File.Exists(makeMetaExe))
            {
                return new NspHandleResultType
                {
                    Result = NspHandleResult.Error,
                    ErrorMessages = new []{$"MakeMeta ('{makeMetaExe}') is not found."}
                };
            }

            var makeMetaArgs = $" --meta {metaPath.SurroundDoubleQuotes}" +
                               $" --desc {descPath.SurroundDoubleQuotes}" +
                               $" -o {Path.Combine(programDir, "main.npdm").ToPathString().SurroundDoubleQuotes}";

            using (var p = ProcessUtility.CreateProcess(makeMetaExe, makeMetaArgs))
            {
                var error = new List<string>();
                p.StartInfo.RedirectStandardError = true;
                p.ErrorDataReceived += (o, e) =>
                {
                    if (e.Data != null)
                        error.Add(e.Data);
                };
                p.Start();
                p.BeginErrorReadLine();
                p.WaitForExit();
                if (p.ExitCode != 0)
                {
                    return new NspHandleResultType
                    {
                        Result = NspHandleResult.Error,
                        ErrorMessages = error.ToArray()
                    };
                }
            }
            return null;
        }

        private void SuppressNmetaUnpublishableError(PathString metaPath, DisposableDirectory tempDir)
        {
            using (var intermediateMeta = Project.Import(_diContainer, ImportableFileType.Meta, metaPath))
            {
                // SIGLO-62841 の修正によって常に nmeta のチェックが行われるようになったため、
                // アイコンが未設定なタイトル (アイコンの設定変更なし) があれば仮アイコンを設定してエラーを抑制
                var iconNotUpdatedTitles = intermediateMeta.Meta.Application.Titles
                    .Where(x => x.IsReplaceIcon == false)
                    .ToArray();
                if (iconNotUpdatedTitles.IsEmpty())
                {
                    return;
                }
                // 仮のアイコン画像を動的に生成して nmeta に設定
                var tempIconPath = Path.Combine(tempDir.RootPath, "icon.bmp");
                using (var bitmap = new Bitmap(Title.OffDeviceIconWidth, Title.OffDeviceIconHeight, PixelFormat.Format24bppRgb))
                {
                    bitmap.Save(tempIconPath, ImageFormat.Bmp);
                }
                foreach (var title in iconNotUpdatedTitles)
                {
                    title.IsReplaceIcon = true;
                    title.IconFilePath.Path = tempIconPath;
                }
                intermediateMeta.OutputAppMetaXmlFileForAuthoringTool(metaPath);
            }
        }

        private static void FixupMetaResourcesPath(Project project)
        {
            var appMeta = project.Meta;

            ResolveExpandablePath(project, appMeta.Application.AccessibleUrlsFilePath);
            ResolveExpandablePath(project, appMeta.Application.HtmlDocumentPath);
            ResolveExpandablePath(project, appMeta.Application.LegalInformationFilePath);

            foreach (var title in appMeta.Application.Titles)
            {
                ResolveExpandablePath(project, title.IconFilePath);
                ResolveExpandablePath(project, title.NxIconFilePath);
            }
        }

        private static void ResolveExpandablePath(Project project, ExpandablePath expandablePath)
        {
            if (expandablePath.IsEmpty)
                return;

            expandablePath.Path = project.ToAbsolutePath(expandablePath.Path);
        }

        private static async Task<NspHandleResultType> ReplaceAsync(
            Project inputNspProject,
            DisposableDirectory tempDir,
            Action<int> progressChanged,
            PathString outputNspPath,
            PathString inputNspFile,
            NspReplaceEntry replaceEntry,
            CancellationToken cancellationToken)
        {
            var outputDir = Path.GetDirectoryName(outputNspPath).ToPathString();

            // AuthoringToolが名付けるファイル名
            var defaultOutputNspPath =
                Path.Combine(outputDir, Path.GetFileNameWithoutExtension(inputNspFile) + "_replaced.nsp")
                    .ToPathString();

            var argsBuilder = new StringBuilder();

            argsBuilder.Append($"replace {inputNspFile.SurroundDoubleQuotes}");
            argsBuilder.Append($" -v -o {outputDir.DirectoryWithoutLastSeparator.ToPathString().SurroundDoubleQuotes}");

            // nsp 内に存在するファイルまたは nca の追加/削除/置き換え
            {
                var replaceRuleListPath = Path.Combine(tempDir.RootPath, "replace_files.txt");
                using (var writer = File.CreateText(replaceRuleListPath))
                {
                    Func<ReplaceFile, bool> selector;
                    // NX Addon 0.x 系はファイルの Replace にのみ対応
                    if (inputNspProject.AppCapability.IsSupportNspTitleAddOrRemove == false)
                        selector = x => x.Operation == ReplaceOperation.Replace;
                    else
                        selector = x => true;

                    BuildReplaceFileList(replaceEntry, writer, selector);
                    BuildReplaceNcaList(replaceEntry, writer);
                }
                argsBuilder.Append($" {replaceRuleListPath.ToPathString().SurroundDoubleQuotes}");
            }

            var args = argsBuilder.ToString();
            using (var job = AuthoringToolWrapper.Run(args, progressChanged))
            {
                try
                {
                    var result = await job.WaitForExitAsync(x => x.MakeResultType(), cancellationToken).ConfigureAwait(false);
                    if (job.ExitCode != 0)
                        return result;
                    if (defaultOutputNspPath == outputNspPath)
                        return result;

                    if (File.Exists(outputNspPath))
                    {
                        File.Replace(defaultOutputNspPath, outputNspPath, null, true);
                    }
                    else
                    {
                        File.Move(defaultOutputNspPath, outputNspPath);
                    }

                    return result;
                }
                catch (TaskCanceledException)
                {
                    return job.MakeResultType(NspHandleResult.Canceled);
                }
            }
        }

        private static void BuildReplaceFileList(NspReplaceEntry replaceEntry, StreamWriter writer, Func<ReplaceFile, bool> selector)
        {
            // nca 内のファイルの置き換え
            foreach (var file in replaceEntry.Files.Where(selector))
            {
                writer.WriteLine(file.ToString());
            }
            // ルートに存在するアイコンの置き換え
            foreach (var icon in replaceEntry.IconFiles.Where(selector))
            {
                writer.WriteLine(icon.ToString());
            }
        }

        private static void BuildReplaceNcaList(NspReplaceEntry replaceEntry, StreamWriter writer)
        {
            // nsp に存在する nca 自体の置き換え
            foreach (var nca in replaceEntry.NcaContents
                .Where(x => string.IsNullOrEmpty(x.SourceContentMetaId) == false))
            {
                writer.WriteLine($"{nca.SourceContentMetaId}.nca\t{nca.LocalPath}");
            }
            // nsp に存在しない nca の追加
            foreach (var ncaEntry in replaceEntry.NcaContents
                .Where(x => string.IsNullOrEmpty(x.SourceContentMetaId)))
            {
                writer.WriteLine($"null.nca@{ncaEntry.Type}\t{ncaEntry.LocalPath}");
            }
        }

        private static NspHandleResultType TestNspReplaceError(ReplaceOption option)
        {
            var project = option.Project;
            if (project == null)
                throw new ArgumentNullException(nameof(project));

            if (File.Exists(AuthoringToolHelper.AuthoringToolExe) == false)
                return new NspHandleResultType { Result = NspHandleResult.NotFoundAuthoringToolExe };

            if (File.Exists(Environment.ExpandEnvironmentVariables(option.InputNspPath ?? string.Empty)) == false)
                return new NspHandleResultType { Result = NspHandleResult.NotFoundNspFile };

            var appMeta = project.Meta.Application;

            var accessibleUrlsFilePath = project.ToAbsolutePath(appMeta.AccessibleUrlsFilePath);
            if (appMeta.IsReplaceAccessibleUrlsFilePath &&
                File.Exists(accessibleUrlsFilePath) == false)
            {
                return new NspHandleResultType
                {
                    Result = NspHandleResult.Error,
                    ErrorMessages = new[] {$"Not found: {accessibleUrlsFilePath}"}
                };
            }

            var htmlDocumentPath = project.ToAbsolutePath(appMeta.HtmlDocumentPath);
            if (appMeta.IsReplaceHtmlDocumentPath &&
                Directory.Exists(htmlDocumentPath) == false)
            {
                return new NspHandleResultType
                {
                    Result = NspHandleResult.Error,
                    ErrorMessages = new[] {$"Not found: {appMeta.HtmlDocumentPath}"}
                };
            }

            var legalInformationFilePath = project.ToAbsolutePath(appMeta.LegalInformationFilePath);
            if (appMeta.IsReplaceLegalInformationFilePath &&
                File.Exists(legalInformationFilePath) == false)
            {
                return new NspHandleResultType
                {
                    Result = NspHandleResult.Error,
                    ErrorMessages = new[] { $"Not found: {legalInformationFilePath}" }
                };
            }

            return null;
        }

        private static bool HasNspExtractedIcon(Title x)
        {
            return string.IsNullOrEmpty(x.NspRawIconFilePath) == false &&
                   string.IsNullOrEmpty(x.NspNxIconFilePath) == false;
        }
    }
}
