﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Serialization;
using BezelEditor.Foundation;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Reactive.Bindings.Extensions;
using SimpleInjector;
using YamlDotNet.Serialization;

namespace Nintendo.Authoring.AuthoringEditor.Core
{
    public class App : DisposableModelBase
    {
        #region Project

        private Project _Project;

        public Project Project
        {
            get { return _Project; }
            set
            {
                var old = _Project;
                if (SetProperty(ref _Project, value))
                    old?.Dispose();
            }
        }

        #endregion

        #region OriginalProject

        private string _OriginalProject = string.Empty;

        public string OriginalProject
        {
            get { return _OriginalProject; }
            set { SetProperty(ref _OriginalProject, value); }
        }

        #endregion

        #region HasErrors

        private bool _HasErrors;

        [YamlIgnore]
        [XmlIgnore]
        public bool HasErrors
        {
            get { return _HasErrors; }
            set { SetProperty(ref _HasErrors, value); }
        }

        #endregion

        #region IsReadOnly

        private bool _IsReadOnly;

        [YamlIgnore]
        [XmlIgnore]
        public bool IsReadOnly
        {
            get { return _IsReadOnly; }
            set { SetProperty(ref _IsReadOnly, value); }
        }

        #endregion

        #region WorkingKind

        private WorkingKind _WorkingKind;

        public WorkingKind WorkingKind
        {
            get { return _WorkingKind; }
            set { SetProperty(ref _WorkingKind, value); }
        }

        #endregion

        #region WorkingState

        private WorkingState _WorkingState;

        public WorkingState WorkingState
        {
            get { return _WorkingState; }
            set { SetProperty(ref _WorkingState, value); }
        }

        #endregion

        #region WorkingProgress

        private int _workingProgress;

        public int WorkingProgress
        {
            get { return _workingProgress; }
            set { SetProperty(ref _workingProgress, value); }
        }

        #endregion

        #region WorkingFilePathForExplorer

        private string _WorkingFilePath;

        public string WorkingFilePath
        {
            get { return _WorkingFilePath; }
            set { SetProperty(ref _WorkingFilePath, value); }
        }

        #endregion

        #region OpenGeneratedObjectAction

        private Action<string> _OpenGeneratedObjectAction;

        public Action<string> OpenGeneratedObjectAction
        {
            get { return _OpenGeneratedObjectAction; }
            set { SetProperty(ref _OpenGeneratedObjectAction, value); }
        }

        #endregion

        #region OpenGeneratedMessageTag

        private string _OpenGeneratedMessageTag;

        public string OpenGeneratedMessageTag
        {
            get { return _OpenGeneratedMessageTag; }
            set { SetProperty(ref _OpenGeneratedMessageTag, value); }
        }

        #endregion

        public CancellationTokenSource WorkingCancellationTokenSource { get; set; }

        #region NspPatch

        private NspPatch _NspPatch;

        public NspPatch NspPatch
        {
            get { return _NspPatch; }
            set { SetProperty(ref _NspPatch, value); }
        }

        #endregion

        public bool IsChangedTarget => OriginalProject != YamlHelper.SaveToString(Project);

        private IDisposable _hasErrorObserver;
        private IDisposable _isReadOnlyObserver;

        private readonly Container _diContainer;

        public App(Container diContainer)
        {
            _diContainer = diContainer;
            _Project = new Project();
            _NspPatch = new NspPatch();

            CompositeDisposable.Add(() => Project?.Dispose());
            CompositeDisposable.Add(() => _hasErrorObserver?.Dispose());
            CompositeDisposable.Add(() => _isReadOnlyObserver?.Dispose());

            CompositeDisposable.Add(() =>
            {
                foreach (var p in _tempDirs)
                {
                    try
                    {
                        Directory.Delete(p, true);
                    }
                    catch
                    {
                        // ignored
                    }
                }
            });

            UpdateHasErrorObserver();
            UpdateIsReadOnlyObserver();

            Observable
                .Merge(this.ObserveProperty(x => x.Project).ToUnit())
                .Merge(_diContainer.GetInstance<AppProfile>().ObserveProperty(x => x.AppMode).ToUnit())
                .Subscribe(_ =>
                {
                    UpdateHasErrorObserver();
                    UpdateIsReadOnlyObserver();
                    UpdateOriginalProject();
                })
                .AddTo(CompositeDisposable);
        }

        public void CancelWorking()
        {
            WorkingCancellationTokenSource?.Cancel();
            WorkingCancellationTokenSource = new CancellationTokenSource();
        }

        private void UpdateHasErrorObserver()
        {
            _hasErrorObserver?.Dispose();

            var isAoc = _diContainer.GetInstance<AppProfile>().AppMode.IsAoc();

            if (isAoc)
                _hasErrorObserver = Project.AocMeta.ObserveProperty(x => x.HasErrors)
                    .Subscribe(i => HasErrors = i);
            else
                _hasErrorObserver = Project.Meta.ObserveProperty(x => x.HasErrors)
                    .Subscribe(i => HasErrors = i);
        }

        private void UpdateIsReadOnlyObserver()
        {
            _isReadOnlyObserver?.Dispose();
            _isReadOnlyObserver = Project.Meta.ObserveProperty(x => x.IsReadOnly)
                .Subscribe(i => IsReadOnly = i);
        }

        public void CreateNewProject()
        {
            Project = Project.CreateNew(_diContainer);
        }

        public bool Save(ExportableFileType type) => Save(type, Project.TargetFilePath);

        public bool Save(ExportableFileType type, PathString filePath)
        {
            var oldProjectDir = Project.ProjectDirectory;

            Project.PrepareOnBeforeSave(filePath.Parent);

            using (var clonedProject = Project.DeepClone())
            {
                clonedProject.ProjectDirectory = oldProjectDir;

                switch (type)
                {
                    case ExportableFileType.Project:
                        if (YamlHelper.Save(filePath, clonedProject) == false)
                            return false;
                        break;

                    case ExportableFileType.AppMeta:
                        if (clonedProject.OutputAppMetaXmlFileForAuthoringTool(filePath) == false)
                            return false;
                        break;

                    case ExportableFileType.AocMeta:
                        if (clonedProject.OutputAocMetaXmlFileForAuthoringTool(filePath) == false)
                            return false;
                        break;

                    default:
                        throw new ArgumentOutOfRangeException(nameof(type), type, null);
                }

                Project.TargetFilePath = filePath;

                _diContainer.GetInstance<Config>().AddToMruList(type.ToImportableFileType(), filePath);

                UpdateOriginalProject();

                return true;
            }
        }

        public void Open(ImportableFileType type, PathString filePath, PathString originalNspFilePath = null)
        {
            Project = Project.Import(_diContainer, type, filePath, originalNspFilePath);
        }

        public void ImportPartial(PathString filePath)
        {
            Project.ImportPartial(filePath);
        }

        public IEnumerable<string> ImportPartialDryRun(PathString filePath)
        {
            return Project.ImportPartialDryRun(filePath);
        }

        public async Task<NspHandleResultType> BuildNspAsync(OutputTypes nspType, PathString outputPath,
            CancellationToken cancellationToken)
        {
            var dir = Project.TargetFilePath.Parent;
            var tempMetafilePath = Path.Combine(dir, Path.GetRandomFileName());

            using (Disposable.Create(() =>
            {
                File.Delete(tempMetafilePath);
            }))
            {
                Project.OutputAppMetaXmlFileForAuthoringTool(tempMetafilePath);

                var option = new NspBuilder.ExportOption
                {
                    OutputPath = outputPath,
                    OutputType = nspType,

                    InputMetaFilePath = tempMetafilePath,
                    InputProgramCodePath = Project.ProgramCodeDirectoryPath,
                    InputProgramDataPath = Project.ProgramDataDirectoryPath,

                    ProgressChanged = i => WorkingProgress = i,
                    CancellationToken = cancellationToken
                };

                var builder = _diContainer.GetInstance<NspBuilder>();
                return await builder.ExportNspAsync(option);
            }
        }

        public async Task<NspHandleResultType> SaveNspAsync(
            PathString outputNspFilePath,
            PathString inputNspFilePath,
            CancellationToken cancellationToken)
        {
            var option = new NspReplacer.ReplaceOption
            {
                InputNspPath = inputNspFilePath,
                OutputNspPath = outputNspFilePath,
                Project = Project,
                ProgressChanged = i => WorkingProgress = i,
                CancellationToken = cancellationToken
            };

            var replacer = _diContainer.GetInstance<NspReplacer>();
            var r = await replacer.ReplaceNspAsync(option);
            if (r.Result != NspHandleResult.Ok)
                return r;

            _diContainer.GetInstance<Config>().AddToMruList(ImportableFileType.Nsp, outputNspFilePath);

            Project.TargetFilePath = outputNspFilePath;
            Project.ProjectDirectory = outputNspFilePath.Parent;

            UpdateOriginalProject();

            // アイコン以外のファイルパスの参照を差し替える
            {
                var appMeta = Project.Meta.Application;

                if (appMeta.IsReplaceAccessibleUrlsFilePath)
                    appMeta.OriginalAccessibleUrlsFilePath = appMeta.AccessibleUrlsFilePath;

                if (appMeta.IsReplaceHtmlDocumentPath)
                    appMeta.OriginalHtmlDocumentPath = appMeta.HtmlDocumentPath;

                if (appMeta.IsReplaceLegalInformationFilePath)
                {
                    // zip を展開したパスになる点に注意
                    try
                    {
                        var tempPath = AddTempDir();
                        ZipFile.ExtractToDirectory(appMeta.LegalInformationFilePath, tempPath);
                        appMeta.OriginalLegalInformationPath = tempPath;
                    }
                    catch {} // ignore
                }
            }

            return r;
        }

        public async Task<NspBuilder.PatchResultType> MakePatchNspAsync(CancellationToken cancellationToken)
        {
            var builder = new NspBuilder();
            var option = new NspBuilder.MakePatchOption
            {
                OutputPath = NspPatch.OutputNspPath,
                OriginalNspFilePath = NspPatch.OriginalNspPath,
                CurrentNspFilePath = NspPatch.CurrentNspPath,
                PreviousNspFilePath = NspPatch.IsUsePreviousNspPath ? NspPatch.PreviousNspPath : null,
                InputDescFilePath =NspPatch.ApplicationDescPath,
                ProgressChanged = i => WorkingProgress = i,
                CancellationToken = cancellationToken
            };
            return await builder.MakePatchNspAsync(option);
        }

        public async Task<NspBuilder.AocResultType> MakeAocNspAsync(
            PathString outputNspFilePath,
            CancellationToken cancellationToken)
        {
            var builder = new NspBuilder();
            var option = new NspBuilder.MakeAocOption
            {
                OutputPath = outputNspFilePath,
                Project = Project,
                ProgressChanged = i => WorkingProgress = i,
                CancellationToken = cancellationToken
            };
            return await builder.MakeAocNspAsync(option);
        }

        private readonly List<string> _tempDirs = new List<string>();

        public string AddTempDir(string path = null)
        {
            if (string.IsNullOrEmpty(path))
            {
                path = Path.Combine(
                    Path.GetTempPath(),
                    @"Nintendo\AuthoringEditor",
                    Path.GetRandomFileName());
            }
            Debug.Assert(string.IsNullOrEmpty(path) == false);
            _tempDirs.Add(path);
            return path;
        }

        private void UpdateOriginalProject()
        {
            OriginalProject = YamlHelper.SaveToString(Project);
        }
    }
}
