﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Threading.Tasks;
using System.Collections.Generic;
using System.Collections;
using Nintendo.AudioTool;

namespace SystemAudioMonitor
{
    public class WaveReceiver : IDisposable
    {
        List<IWaveProcessor> _processorList = new List<IWaveProcessor>();

        SystemAudioCapture _capture;

        public WaveformFormat Format { get; }

        public EventHandler<Exception> CaptureStopped { get; set; }

        public int AdspLoad
        {
#if DEBUG
            get { return _capture == null ? 0 : _capture.AdspLoad; }
#else  // DEBUG
            get { return -1; }
#endif  // DEBUG
        }

        public WaveReceiver(string name = null)
        {
            var tempCapture = new SystemAudioCapture(name);
            var format = tempCapture.Format;

            Format = new WaveformFormat(format.SampleRate, format.BitDepth, format.ChannelCount);
        }

        private void OnDataAvailable(object sender, SystemAudioCaptureEventArgs e)
        {
            lock (((ICollection)_processorList).SyncRoot)
            {
                foreach (var processor in _processorList)
                {
                    processor.AddSamples(e.Buffer, 0, e.BytesCaptured);
                }
            }
        }

        private void OnCaptureStopped(object sender, SystemAudioCaptureStoppedEventArgs e)
        {
            Console.WriteLine("{0}", e?.Exception?.Message);
            lock (((ICollection)_processorList).SyncRoot)
            {
                foreach (var processor in _processorList)
                {
                    processor.OnCaptureStopped(e.Exception);
                }
            }
            CaptureStopped(this, e.Exception);
        }


        public bool AddWaveProcessor(IWaveProcessor recorder)
        {
            if(recorder == null)
            {
                return false;
            }
            lock (((ICollection)_processorList).SyncRoot)
            {
                _processorList.Add(recorder);
            }
            return true;
        }

        public void RemoveWaveProcessor(IWaveProcessor recorder)
        {
            lock (((ICollection)_processorList).SyncRoot)
            {
                _processorList.Remove(recorder);
            }
        }

        public async Task Start(string name)
        {
            _capture = new SystemAudioCapture(name);
            _capture.DataAvailable += OnDataAvailable;
            _capture.CaptureStopped += OnCaptureStopped;
            await _capture.StartAsync();
        }


        public void Stop()
        {
            _capture?.Stop();
            _capture?.Dispose();
            _capture = null;
        }

        public void Dispose()
        {
            Stop();
        }
    }


}
